package Chart::Plotly::Trace::Sankey::Domain;
use Moose;
use MooseX::ExtraArgs;
use Moose::Util::TypeConstraints qw(enum union);
if ( !defined Moose::Util::TypeConstraints::find_type_constraint('PDL') ) {
    Moose::Util::TypeConstraints::type('PDL');
}

our $VERSION = '0.039';    # VERSION

# ABSTRACT: This attribute is one of the possible options for the trace sankey.

sub TO_JSON {
    my $self       = shift;
    my $extra_args = $self->extra_args // {};
    my $meta       = $self->meta;
    my %hash       = %$self;
    for my $name ( sort keys %hash ) {
        my $attr = $meta->get_attribute($name);
        if ( defined $attr ) {
            my $value = $hash{$name};
            my $type  = $attr->type_constraint;
            if ( $type && $type->equals('Bool') ) {
                $hash{$name} = $value ? \1 : \0;
            }
        }
    }
    %hash = ( %hash, %$extra_args );
    delete $hash{'extra_args'};
    if ( $self->can('type') && ( !defined $hash{'type'} ) ) {
        $hash{type} = $self->type();
    }
    return \%hash;
}

has column => (
       is            => "rw",
       isa           => "Int",
       documentation => "If there is a layout grid, use the domain for this column in the grid for this sankey trace .",
);

has row => (
          is            => "rw",
          isa           => "Int",
          documentation => "If there is a layout grid, use the domain for this row in the grid for this sankey trace .",
);

has x => ( is            => "rw",
           isa           => "ArrayRef|PDL",
           documentation => "Sets the horizontal domain of this sankey trace (in plot fraction).",
);

has y => ( is            => "rw",
           isa           => "ArrayRef|PDL",
           documentation => "Sets the vertical domain of this sankey trace (in plot fraction).",
);

__PACKAGE__->meta->make_immutable();
1;

__END__

=pod

=encoding utf-8

=head1 NAME

Chart::Plotly::Trace::Sankey::Domain - This attribute is one of the possible options for the trace sankey.

=head1 VERSION

version 0.039

=head1 SYNOPSIS

 use Chart::Plotly qw(show_plot);
 use Chart::Plotly::Trace::Sankey;
 # Example data from: https://plot.ly/javascript/sankey-diagram/#basic-sankey-diagram
 my $sankey = Chart::Plotly::Trace::Sankey->new(
     orientation => "h",
     node        => {
         pad       => 15,
         thickness => 30,
         line      => {
             color => "black",
             width => 0.5
         },
         label     => [ "A1", "A2", "B1", "B2", "C1", "C2" ],
         color     => [ "blue", "blue", "blue", "blue", "blue", "blue" ]
     },
 
     link        => {
         source => [ 0, 1, 0, 2, 3, 3 ],
         target => [ 2, 3, 3, 4, 4, 5 ],
         value  => [ 8, 4, 2, 8, 4, 2 ]
     }
 );
 
 show_plot([ $sankey ]);

=head1 DESCRIPTION

This attribute is part of the possible options for the trace sankey.

This file has been autogenerated from the official plotly.js source.

If you like Plotly, please support them: L<https://plot.ly/> 
Open source announcement: L<https://plot.ly/javascript/open-source-announcement/>

Full reference: L<https://plot.ly/javascript/reference/#sankey>

=head1 DISCLAIMER

This is an unofficial Plotly Perl module. Currently I'm not affiliated in any way with Plotly. 
But I think plotly.js is a great library and I want to use it with perl.

=head1 METHODS

=head2 TO_JSON

Serialize the trace to JSON. This method should be called only by L<JSON> serializer.

=head1 ATTRIBUTES

=over

=item * column

If there is a layout grid, use the domain for this column in the grid for this sankey trace .

=item * row

If there is a layout grid, use the domain for this row in the grid for this sankey trace .

=item * x

Sets the horizontal domain of this sankey trace (in plot fraction).

=item * y

Sets the vertical domain of this sankey trace (in plot fraction).

=back

=head1 AUTHOR

Pablo Rodríguez González <pablo.rodriguez.gonzalez@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2020 by Pablo Rodríguez González.

This is free software, licensed under:

  The MIT (X11) License

=cut
