#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2017-2018 -- leonerd@leonerd.org.uk

package String::Tagged::Terminal::Win32Console;

use strict;
use warnings;
use Win32::Console;

our $VERSION = '0.03_002';

=head1 NAME

C<String::Tagged::Terminal::Win32Console> - Windows-specific code for L<String::Tagged::Terminal>

=head1 SYNOPSIS

   # No user serviceable parts inside
   use String::Tagged::Terminal;

=head1 DESCRIPTION

This module provides support for L<String::Tagged::Terminal> to print to the
console on C<MSWin32>. It is not intended to be used directly.

=cut

# We can only ever allocate a single console on Windows
our $WIN32_CONSOLE;

sub print_to_console
{
   my $self = shift;
   my ( $fh, %opts ) = @_;

   # Convert filenos to native Win32 file handles, this should also try
   # Win32API::File::FdGetOsFHandle( $fh );
   my $fileno = {
       1 => Win32::Console::STD_OUTPUT_HANDLE(),
       2 => Win32::Console::STD_ERROR_HANDLE(),
   }->{ $fh->fileno } || $fh->fileno;

   my %output_options = (
      ( $opts{no_color} ? ( except => [qw( fgindex bgindex )] ) : () ),
      only => [qw( fgindex bgindex under reverse )], # only process what we can handle
   );

   if( $fileno < 0 ) {
      # This looks like a Perl-internal FH, let's not output any formatting
      $fh->print( $self->build_terminal( %opts ) );
   }
   else {
      my $console = $opts{console} || do { $WIN32_CONSOLE ||= Win32::Console->new( $fileno ); };
      my $saved = $console->Attr();
      my $attr = $saved;

      $self->iter_substr_nooverlap( sub {
         my ( $s, %tags ) = @_;

         # Simple boolean attributes first
         foreach (
         #   #[ bold      =>  1, 21 ],
            [ under     =>  0x8000, 0 ],  # COMMON_LVB_UNDERSCORE, Windows 10 onwards
            # Rendering is flakey under Windows 10
         #   #[ italic    =>  3, 23 ],
         #   #[ strike    =>  9, 29 ],
         #   #[ blink     =>  5, 25 ],
            [ reverse   =>  0x4000, 0 ], # COMMON_LVB_REVERSE_VIDEO, Windows 10 onwards
         ) {
            my ( $tag, $on, $off ) = @$_;
            if( exists $tags{ $tag } ) {
               my $keep = 0xFFFF ^ $on;
               $attr = $attr & $keep;
               if( $tags{ $tag }) {
                  $attr = $attr |= $on;
               }
            }
         }

         # Colour index attributes
         foreach (
            [ fgindex => 0, ],
            [ bgindex => 4, ],
         ) {
            my ( $tag, $shift ) = @$_;
            if( exists $tags{ $tag }) {
               my $val = $tags{ $tag };

               # The bits for red and blue are swapped between ANSI and Win32 console
               my $r = $val & 1;
               my $b = $val & 4;
               $val &= 0b1111111111111010;
               $val |= 0b0000000000000100 if( $r );
               $val |= 0b0000000000000001 if( $b );
               my $nibble = $val << $shift;
               my $mask   = 0x000F << $shift;
               my $keep   = 0xFFFF ^ $mask;

               $attr = $attr & $keep | $nibble;
            }
         }

         $console->Attr($attr);
         $console->Write($s);
      }, %output_options );

      $console->Attr( $saved );
   }
}

=head1 COMPATIBILITY NOTES

On Windows before Windows 10, only C<fgindex> and C<bgindex> are supported.

Starting with Windows 10, also C<under> and C<reverse> are supported.

On Windows, only a single output console is supported.

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>,
Max Maischein <corion@corion.net>

=cut

0x55AA;
