use Mojolicious::Lite;
use Mojo::Loader 'data_section';
use Time::Piece;
use Test::More;
use Mojo::WebService::Twitter;
use Mojo::Util 'b64_encode';

use constant TEST_API_KEY => 'testapikey';
use constant TEST_API_SECRET => 'testapisecret';
use constant TEST_ACCESS_TOKEN => 'testaccesstoken';

my %tweet_data = (
	'657606994744860672' => {
		data_section => 'tweet',
		coordinates => undef,
		created_at => scalar gmtime(1445620699),
		favorites => 382,
		in_reply_to_screen_name => undef,
		in_reply_to_status_id => undef,
		in_reply_to_user_id => undef,
		retweets => 289,
		text => q{The @Space_Station crew works  ~9.5 hours a day, with ~4 hours free time during the week... https://t.co/2cdarZPqND https://t.co/HlAnj3eTKk},
		user_id => '1451773004',
	},
	'657324783294676992' => {
		data_section => 'tweet_reply',
		coordinates => undef,
		created_at => scalar gmtime(1445553415),
		favorites => 21,
		in_reply_to_screen_name => 'starlingLX',
		in_reply_to_status_id => '657257155855294465',
		in_reply_to_user_id => '348968125',
		retweets => 13,
		text => q{@starlingLX @bnjacobs @StationCDRKelly @Flickr The higher-res is added but doesn't have @StationCDRKelly's edits...https://t.co/wnKeCAdLfg},
		user_id => '1451773004',
	},
	'657627567948587008' => {
		data_section => 'tweet_retweet',
		coordinates => undef,
		created_at => scalar gmtime(1445625604),
		favorites => 0,
		in_reply_to_screen_name => undef,
		in_reply_to_status_id => undef,
		in_reply_to_user_id => undef,
		retweets => 35897,
		text => q{RT @StationCDRKelly: Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https://t.co/6LP2xCYcGD},
		user_id => '1451773004',
	},
);

my %user_data = (
	'1451773004' => {
		data_section => 'user',
		created_at => scalar gmtime(1369322728),
		description => q{NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.},
		followers => 314219,
		friends => 230,
		last_tweet_ts => scalar gmtime(1445625604),
		name => 'Intl. Space Station',
		protected => 0,
		screen_name => 'Space_Station',
		statuses => 3228,
		time_zone => 'Central Time (US & Canada)',
		url => 'http://t.co/9Gk2GZYDsP',
		utc_offset => -18000,
		verified => 1,
	},
);

post '/oauth2/token' => sub {
	my $c = shift;
	my $auth_header = $c->req->headers->authorization;
	my $token = b64_encode(TEST_API_KEY . ':' . TEST_API_SECRET, '');
	is $auth_header, "Basic $token", 'received API key and secret';
	$c->render(format => 'json', text => '{"token_type":"bearer","access_token":"'.TEST_ACCESS_TOKEN.'"}');
};
group {
	under '/api' => sub {
		my $c = shift;
		my $auth_header = $c->req->headers->authorization;
		is $auth_header, 'Bearer '.TEST_ACCESS_TOKEN, 'received access token';
		return 1;
	};
	get '/statuses/show.json' => sub {
		my $c = shift;
		my $id = $c->param('id');
		die "Unknown tweet ID $id" unless defined $id and exists $tweet_data{$id};
		my $data_section = $tweet_data{$id}{data_section} // 'tweet';
		$c->render(format => 'json', text => data_section('main', $data_section));
	};
	get '/users/show.json' => sub {
		my $c = shift;
		my $id;
		if ($id = $c->param('user_id')) {
			die "Unknown user ID $id" unless exists $user_data{$id};
		} elsif (my $name = $c->param('screen_name')) {
			($id) = grep { lc $user_data{$_}{screen_name} eq lc $name } keys %user_data;
			die "Unknown user screen name $name" unless defined $id;
		}
		my $data_section = $user_data{$id}{data_section} // 'user';
		$c->render(format => 'json', text => data_section('main', $data_section));
	};
	get '/search/tweets.json' => { format => 'json', text => data_section('main', 'search_tweets') };
};

my $api_key = $ENV{TWITTER_API_KEY};
my $api_secret = $ENV{TWITTER_API_SECRET};

if ($ENV{AUTHOR_TESTING} and defined $api_key and defined $api_secret) {
	diag 'Running online test for Twitter';
} else {
	diag 'Running offline test for Twitter; set AUTHOR_TESTING and TWITTER_API_KEY/TWITTER_API_SECRET for online test';
	$Mojo::WebService::Twitter::Util::API_BASE_URL = '/api/';
	$Mojo::WebService::Twitter::Util::OAUTH_BASE_URL = '/oauth/';
	$Mojo::WebService::Twitter::Util::OAUTH2_BASE_URL = '/oauth2/';
	$api_key = TEST_API_KEY;
	$api_secret = TEST_API_SECRET;
}

my $twitter = Mojo::WebService::Twitter->new;
$twitter->ua->server->app->log->level('warn');

ok !eval { $twitter->get_tweet("657618739492474880"); 1 }, 'no API key set';
is $twitter->api_key($api_key)->api_key, $api_key, 'set API key';
is $twitter->api_secret($api_secret)->api_secret, $api_secret, 'set API secret';

my $access;
ok(eval { $access = $twitter->request_oauth2; 1 }, 'requested OAuth 2 access token') or diag $@;
ok defined($access->{access_token}), 'received access token';
$twitter->authentication(oauth2 => $access->{access_token});

foreach my $id (keys %tweet_data) {
	my $data = $tweet_data{$id};
	my $tweet;
	ok(eval { $tweet = $twitter->get_tweet($id); 1 }, "retrieved tweet $id") or diag $@;
	is $tweet->id, $id, 'right tweet ID';
	is_deeply $tweet->coordinates, $data->{coordinates}, 'right coordinates';
	is $tweet->created_at, $data->{created_at}, 'right creation timestamp';
	ok $tweet->favorites >= $data->{favorites}, "at least $data->{favorites} favorites";
	is $tweet->in_reply_to_screen_name, $data->{in_reply_to_screen_name}, 'right in-reply-to screen name';
	is $tweet->in_reply_to_status_id, $data->{in_reply_to_status_id}, 'right in-reply-to status ID';
	is $tweet->in_reply_to_user_id, $data->{in_reply_to_user_id}, 'right in-reply-to user ID';
	ok $tweet->retweets >= $data->{retweets}, "at least $data->{retweets} retweets";
	is $tweet->text, $data->{text}, 'right text';
	is $tweet->user->id, $data->{user_id}, 'right user';
}

foreach my $id (keys %user_data) {
	my $data = $user_data{$id};
	my $name = $data->{screen_name};
	my $user;
	ok(eval { $user = $twitter->get_user(user_id => $id); 1 }, "retrieved user $id") or diag $@;
	is $user->id, $id, 'right user ID';
	is lc $user->screen_name, lc $name, 'right user screen name';
	my $user2;
	ok(eval { $user2 = $twitter->get_user(screen_name => $name); 1 }, "retrieved user $name") or diag $@;
	is $user2->id, $id, 'right user ID';
	is lc $user2->screen_name, lc $name, 'right user screen name';
	
	is $user->created_at, $data->{created_at}, 'right creation timestamp';
	is $user->description, $data->{description}, 'right description';
	ok $user->followers >= $data->{followers}, "at least $data->{followers} followers";
	ok $user->friends >= $data->{friends}, "at least $data->{friends} friends";
	is $user->name, $data->{name}, 'right name';
	ok !($user->protected xor $data->{protected}), 'right protected status';
	ok $user->statuses >= $data->{statuses}, "at least $data->{statuses} statuses";
	is $user->time_zone, $data->{time_zone}, 'right time zone';
	is $user->url, $data->{url}, 'right url';
	ok defined($user->utc_offset), 'UTC offset is set';
	ok !($user->verified xor $data->{verified}), 'right verified status';
	ok $user->last_tweet->created_at >= $data->{last_tweet_ts}, "last tweet after $data->{last_tweet_ts}";
}

my $search_term = 'defiantly';
my $tweets;
ok(eval { $tweets = $twitter->search_tweets($search_term); 1 }, "searched tweets for \"$search_term\"") or diag $@;
ok scalar(@$tweets), 'retrieved ' . @$tweets . ' search results';
like $tweets->[0]->text, qr/defiantly/i, 'search result contains search term';

done_testing;

__DATA__

@@ user
{"id":1451773004,"id_str":"1451773004","name":"Intl. Space Station","screen_name":"Space_Station","location":"Low Earth Orbit","profile_location":null,"description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","url":"http:\/\/t.co\/9Gk2GZYDsP","entities":{"url":{"urls":[{"url":"http:\/\/t.co\/9Gk2GZYDsP","expanded_url":"http:\/\/www.nasa.gov\/station","display_url":"nasa.gov\/station","indices":[0,22]}]},"description":{"urls":[]}},"protected":false,"followers_count":314219,"friends_count":230,"listed_count":3711,"created_at":"Thu May 23 15:25:28 +0000 2013","favourites_count":1233,"utc_offset":-18000,"time_zone":"Central Time (US & Canada)","geo_enabled":false,"verified":true,"statuses_count":3228,"lang":"en","status":{"created_at":"Fri Oct 23 18:40:04 +0000 2015","id":657627567948587008,"id_str":"657627567948587008","text":"RT @StationCDRKelly: Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https:\/\/t.co\/6LP2xCYcGD","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Fri Oct 23 18:05:00 +0000 2015","id":657618739492474880,"id_str":"657618739492474880","text":"Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https:\/\/t.co\/6LP2xCYcGD","source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"geo":null,"coordinates":null,"place":null,"contributors":null,"retweet_count":35624,"favorite_count":22474,"entities":{"hashtags":[{"text":"Patricia","indices":[10,19]},{"text":"Mexico","indices":[73,80]},{"text":"YearInSpace","indices":[82,94]}],"symbols":[],"user_mentions":[{"screen_name":"Space_Station","name":"Intl. Space Station","id":1451773004,"id_str":"1451773004","indices":[40,54]}],"urls":[],"media":[{"id":657618738447958017,"id_str":"657618738447958017","indices":[95,118],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","url":"https:\/\/t.co\/6LP2xCYcGD","display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"http:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","type":"photo","sizes":{"medium":{"w":600,"h":399,"resize":"fit"},"small":{"w":340,"h":226,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":1024,"h":681,"resize":"fit"}}}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"retweet_count":35624,"favorite_count":0,"entities":{"hashtags":[{"text":"Patricia","indices":[31,40]},{"text":"Mexico","indices":[94,101]},{"text":"YearInSpace","indices":[103,115]}],"symbols":[],"user_mentions":[{"screen_name":"StationCDRKelly","name":"Scott Kelly","id":65647594,"id_str":"65647594","indices":[3,19]},{"screen_name":"Space_Station","name":"Intl. Space Station","id":1451773004,"id_str":"1451773004","indices":[61,75]}],"urls":[],"media":[{"id":657618738447958017,"id_str":"657618738447958017","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","url":"https:\/\/t.co\/6LP2xCYcGD","display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"http:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","type":"photo","sizes":{"medium":{"w":600,"h":399,"resize":"fit"},"small":{"w":340,"h":226,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":1024,"h":681,"resize":"fit"}},"source_status_id":657618739492474880,"source_status_id_str":"657618739492474880","source_user_id":65647594,"source_user_id_str":"65647594"}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1434028060","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null}

@@ tweet
{"created_at":"Fri Oct 23 17:18:19 +0000 2015","id":657606994744860672,"id_str":"657606994744860672","text":"The @Space_Station crew works  ~9.5 hours a day, with ~4 hours free time during the week... https:\/\/t.co\/2cdarZPqND https:\/\/t.co\/HlAnj3eTKk","source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1451773004,"id_str":"1451773004","name":"Intl. Space Station","screen_name":"Space_Station","location":"Low Earth Orbit","description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","url":"http:\/\/t.co\/9Gk2GZYDsP","entities":{"url":{"urls":[{"url":"http:\/\/t.co\/9Gk2GZYDsP","expanded_url":"http:\/\/www.nasa.gov\/station","display_url":"nasa.gov\/station","indices":[0,22]}]},"description":{"urls":[]}},"protected":false,"followers_count":314231,"friends_count":230,"listed_count":3711,"created_at":"Thu May 23 15:25:28 +0000 2013","favourites_count":1233,"utc_offset":-18000,"time_zone":"Central Time (US & Canada)","geo_enabled":false,"verified":true,"statuses_count":3228,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1434028060","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":289,"favorite_count":382,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"Space_Station","name":"Intl. Space Station","id":1451773004,"id_str":"1451773004","indices":[4,18]}],"urls":[{"url":"https:\/\/t.co\/2cdarZPqND","expanded_url":"http:\/\/www.nasa.gov\/feature\/5-fun-things-to-do-without-gravity","display_url":"nasa.gov\/feature\/5-fun-\u2026","indices":[92,115]}],"media":[{"id":657606857159143425,"id_str":"657606857159143425","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","media_url_https":"https:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","url":"https:\/\/t.co\/HlAnj3eTKk","display_url":"pic.twitter.com\/HlAnj3eTKk","expanded_url":"http:\/\/twitter.com\/Space_Station\/status\/657606994744860672\/photo\/1","type":"photo","sizes":{"medium":{"w":400,"h":258,"resize":"fit"},"small":{"w":340,"h":219,"resize":"fit"},"large":{"w":400,"h":258,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"}}}]},"extended_entities":{"media":[{"id":657606857159143425,"id_str":"657606857159143425","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","media_url_https":"https:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","url":"https:\/\/t.co\/HlAnj3eTKk","display_url":"pic.twitter.com\/HlAnj3eTKk","expanded_url":"http:\/\/twitter.com\/Space_Station\/status\/657606994744860672\/photo\/1","type":"animated_gif","sizes":{"medium":{"w":400,"h":258,"resize":"fit"},"small":{"w":340,"h":219,"resize":"fit"},"large":{"w":400,"h":258,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"}},"video_info":{"aspect_ratio":[200,129],"variants":[{"bitrate":0,"content_type":"video\/mp4","url":"https:\/\/pbs.twimg.com\/tweet_video\/CSBJ89LU8AEPKrl.mp4"}]}}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"lang":"en"}

@@ tweet_reply
{"created_at":"Thu Oct 22 22:36:55 +0000 2015","id":657324783294676992,"id_str":"657324783294676992","text":"@starlingLX @bnjacobs @StationCDRKelly @Flickr The higher-res is added but doesn't have @StationCDRKelly's edits...https:\/\/t.co\/wnKeCAdLfg","source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":657257155855294465,"in_reply_to_status_id_str":"657257155855294465","in_reply_to_user_id":348968125,"in_reply_to_user_id_str":"348968125","in_reply_to_screen_name":"starlingLX","user":{"id":1451773004,"id_str":"1451773004","name":"Intl. Space Station","screen_name":"Space_Station","location":"Low Earth Orbit","description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","url":"http:\/\/t.co\/9Gk2GZYDsP","entities":{"url":{"urls":[{"url":"http:\/\/t.co\/9Gk2GZYDsP","expanded_url":"http:\/\/www.nasa.gov\/station","display_url":"nasa.gov\/station","indices":[0,22]}]},"description":{"urls":[]}},"protected":false,"followers_count":314238,"friends_count":230,"listed_count":3711,"created_at":"Thu May 23 15:25:28 +0000 2013","favourites_count":1233,"utc_offset":-18000,"time_zone":"Central Time (US & Canada)","geo_enabled":false,"verified":true,"statuses_count":3228,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1434028060","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":13,"favorite_count":21,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"starlingLX","name":"Alex von Eckartsberg","id":348968125,"id_str":"348968125","indices":[0,11]},{"screen_name":"bnjacobs","name":"Bob Jacobs","id":17897744,"id_str":"17897744","indices":[12,21]},{"screen_name":"StationCDRKelly","name":"Scott Kelly","id":65647594,"id_str":"65647594","indices":[22,38]},{"screen_name":"Flickr","name":"Flickr","id":21237045,"id_str":"21237045","indices":[39,46]},{"screen_name":"StationCDRKelly","name":"Scott Kelly","id":65647594,"id_str":"65647594","indices":[88,104]}],"urls":[{"url":"https:\/\/t.co\/wnKeCAdLfg","expanded_url":"https:\/\/www.flickr.com\/photos\/nasa2explore\/21772465134\/in\/dateposted-public\/","display_url":"flickr.com\/photos\/nasa2ex\u2026","indices":[115,138]}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"lang":"en"}

@@ tweet_retweet
{"created_at":"Fri Oct 23 18:40:04 +0000 2015","id":657627567948587008,"id_str":"657627567948587008","text":"RT @StationCDRKelly: Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https:\/\/t.co\/6LP2xCYcGD","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1451773004,"id_str":"1451773004","name":"Intl. Space Station","screen_name":"Space_Station","location":"Low Earth Orbit","description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","url":"http:\/\/t.co\/9Gk2GZYDsP","entities":{"url":{"urls":[{"url":"http:\/\/t.co\/9Gk2GZYDsP","expanded_url":"http:\/\/www.nasa.gov\/station","display_url":"nasa.gov\/station","indices":[0,22]}]},"description":{"urls":[]}},"protected":false,"followers_count":314525,"friends_count":230,"listed_count":3712,"created_at":"Thu May 23 15:25:28 +0000 2013","favourites_count":1233,"utc_offset":-18000,"time_zone":"Central Time (US & Canada)","geo_enabled":false,"verified":true,"statuses_count":3228,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/517439388741931008\/iRbQw1ch.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/647082562125459456\/pmT48eHQ_nor mal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1434028060","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Fri Oct 23 18:05:00 +0000 2015","id":657618739492474880,"id_str":"657618739492474880","text":"Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https:\/\/t.co\/6LP2xCYcGD","source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":65647594,"id_str":"65647594","name":"Scott Kelly","screen_name":"StationCDRKelly","location":"International Space Station","description":"","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":566929,"friends_count":137,"listed_count":5419,"created_at":"Fri Aug 14 14:31:39 +0000 2009","favourites_count":9,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":true,"statuses_count":1712,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"010505","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/31467382\/Scott_Twitter.jpg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/31467382\/Scott_Twitter.jpg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/558447158597136385\/P9TpCaRn_normal.jpeg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/558447158597136385\/P9TpCaRn_normal.jpeg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/65647594\/1445202282","profile_link_color":"3B94D9","profile_sidebar_border_color":"838582","profile_sidebar_fill_color":"262626","profile_text_color":"727273","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":35897,"favorite_count":22703,"entities":{"hashtags":[{"text":"Patricia","indices":[10,19]},{"text":"Mexico","indices":[73,80]},{"text":"YearInSpace","indices":[82,94]}],"symbols":[],"user_mentions":[{"screen_name":"Space_Station","name":"Intl. Space Station","id":1451773004,"id_str":"1451773004","indices":[40,54]}],"urls":[],"media":[{"id":657618738447958017,"id_str":"657618738447958017","indices":[95,118],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","url":"https:\/\/t.co\/6LP2xCYcGD","display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"http:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","type":"photo","sizes":{"medium":{"w":600,"h":399,"resize":"fit"},"small":{"w":340,"h":226,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":1024,"h":681,"resize":"fit"}}}]},"extended_entities":{"media":[{"id":657618738447958017,"id_str":"657618738447958017","indices":[95,118],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","url":"https:\/\/t.co\/6LP2xCYcGD","display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"http:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","type":"photo","sizes":{"medium":{"w":600,"h":399,"resize":"fit"},"small":{"w":340,"h":226,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":1024,"h":681,"resize":"fit"}}}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"lang":"en"},"is_quote_status":false,"retweet_count":35897,"favorite_count":0,"entities":{"hashtags":[{"text":"Patricia","indices":[31,40]},{"text":"Mexico","indices":[94,101]},{"text":"YearInSpace","indices":[103,115]}],"symbols":[],"user_mentions":[{"screen_name":"StationCDRKelly","name":"Scott Kelly","id":65647594,"id_str":"65647594","indices":[3,19]},{"screen_name":"Space_Station","name":"Intl. Space Station","id":1451773004,"id_str":"1451773004","indices":[61,75]}],"urls":[],"media":[{"id":657618738447958017,"id_str":"657618738447958017","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","url":"https:\/\/t.co\/6LP2xCYcGD","display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"http:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","type":"photo","sizes":{"medium":{"w":600,"h":399,"resize":"fit"},"small":{"w":340,"h":226,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":1024,"h":681,"resize":"fit"}},"source_status_id":657618739492474880,"source_status_id_str":"657618739492474880","source_user_id":65647594,"source_user_id_str":"65647594"}]},"extended_entities":{"media":[{"id":657618738447958017,"id_str":"657618738447958017","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","url":"https:\/\/t.co\/6LP2xCYcGD","display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"http:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","type":"photo","sizes":{"medium":{"w":600,"h":399,"resize":"fit"},"small":{"w":340,"h":226,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":1024,"h":681,"resize":"fit"}},"source_status_id":657618739492474880,"source_status_id_str":"657618739492474880","source_user_id":65647594,"source_user_id_str":"65647594"}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"lang":"en"}

@@ search_tweets
{"statuses":[{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:21:39 +0000 2016","id":696896717866823681,"id_str":"696896717866823681","text":"@CurtieStillman defiantly !! And we'll have to get Bayleigh to come too \ud83d\ude0c\ud83d\ude0c","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":696895810852638720,"in_reply_to_status_id_str":"696895810852638720","in_reply_to_user_id":2407557835,"in_reply_to_user_id_str":"2407557835","in_reply_to_screen_name":"CurtieStillman","user":{"id":29800895,"id_str":"29800895","name":"Megan Woodcock","screen_name":"Meauhgan","location":"Ontario, Canada","description":"19 | probably drunk | Instagram: Meauhgan","url":"https:\/\/t.co\/oUsqbBLB2z","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/oUsqbBLB2z","expanded_url":"http:\/\/tragicalbeauty.tumblr.com\/","display_url":"tragicalbeauty.tumblr.com","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":39427,"friends_count":253,"listed_count":128,"created_at":"Wed Apr 08 19:44:23 +0000 2009","favourites_count":663,"utc_offset":-18000,"time_zone":"Quito","geo_enabled":true,"verified":false,"statuses_count":42361,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"7EE0A5","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/656259041\/ko6mnos5ryhlappm5tz3.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/656259041\/ko6mnos5ryhlappm5tz3.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/688181577763786752\/VmerNfvn_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/688181577763786752\/VmerNfvn_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/29800895\/1454440365","profile_link_color":"81D3DE","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"252429","profile_text_color":"A504DB","profile_use_background_image":false,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"CurtieStillman","name":"Ginga Ninga","id":2407557835,"id_str":"2407557835","indices":[0,15]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:21:02 +0000 2016","id":696896562031513602,"id_str":"696896562031513602","text":"Defiantly missed Eve first and second hour today  @evedilly","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1689301400,"id_str":"1689301400","name":"emily \u0950","screen_name":"bollock_s","location":"possum kingdom","description":"Just your average weirdo || Cody Dilly \u2764\ufe0f","url":"https:\/\/t.co\/jSHAdrg0Ip","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/jSHAdrg0Ip","expanded_url":"http:\/\/instagram.com\/bolloc.ks","display_url":"instagram.com\/bolloc.ks","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":274,"friends_count":99,"listed_count":0,"created_at":"Wed Aug 21 21:42:05 +0000 2013","favourites_count":2432,"utc_offset":-21600,"time_zone":"Central Time (US & Canada)","geo_enabled":false,"verified":false,"statuses_count":1471,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"1A1B1F","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/645962629517869057\/6i0YRCcP.jpg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/645962629517869057\/6i0YRCcP.jpg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/675354281537486848\/ZUZUj3UX_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/675354281537486848\/ZUZUj3UX_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1689301400\/1454806216","profile_link_color":"F5ABB5","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"000000","profile_text_color":"000000","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"evedilly","name":"Eve Dilly","id":2806665674,"id_str":"2806665674","indices":[50,59]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:20:53 +0000 2016","id":696896523544588288,"id_str":"696896523544588288","text":"RT @NVOjy: It's definitely not defiantly.. \ud83d\ude11\ud83e\udd14\ud83d\ude10\ud83d\ude44","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":2266987230,"id_str":"2266987230","name":"Kayla","screen_name":"Kayla_1610","location":"Somewhere between Mile 4 and 5","description":"They don't want us to be successful. They don't want us to win. You think they want us to be happy? You crazy!","url":"https:\/\/t.co\/fCE8MsNwqF","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/fCE8MsNwqF","expanded_url":"http:\/\/youtu.be\/50XgOtXaLXI","display_url":"youtu.be\/50XgOtXaLXI","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":395,"friends_count":377,"listed_count":2,"created_at":"Sun Dec 29 06:28:46 +0000 2013","favourites_count":11239,"utc_offset":-32400,"time_zone":"Alaska","geo_enabled":true,"verified":false,"statuses_count":9667,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/692650600396271616\/UJp6G-XV_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/692650600396271616\/UJp6G-XV_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/2266987230\/1453880311","profile_link_color":"0084B4","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:12:17 +0000 2016","id":696894358059089920,"id_str":"696894358059089920","text":"It's definitely not defiantly.. \ud83d\ude11\ud83e\udd14\ud83d\ude10\ud83d\ude44","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":298874099,"id_str":"298874099","name":"Kobe Jyant","screen_name":"NVOjy","location":"Columbus, OH","description":"I hit that Jynobil\u00ed with my left hand up like wooo! PIKACHU! young pablito de seis dios Hartley Alumni| J'Air Wheelwright!","url":"https:\/\/t.co\/fus4SjPrhm","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/fus4SjPrhm","expanded_url":"http:\/\/www.hudl.com\/athlete\/2619550\/highlights\/248028376\/v2","display_url":"hudl.com\/athlete\/261955\u2026","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":1545,"friends_count":693,"listed_count":6,"created_at":"Sun May 15 03:10:46 +0000 2011","favourites_count":8172,"utc_offset":-18000,"time_zone":"Quito","geo_enabled":false,"verified":false,"statuses_count":50553,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"022330","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/651613847963443200\/tLlILEXo.jpg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/651613847963443200\/tLlILEXo.jpg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/695885381703753728\/ztJRKak9_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/695885381703753728\/ztJRKak9_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/298874099\/1454798421","profile_link_color":"000000","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"000000","profile_text_color":"000000","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":2,"favorite_count":4,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},"is_quote_status":false,"retweet_count":2,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"NVOjy","name":"Kobe Jyant","id":298874099,"id_str":"298874099","indices":[3,9]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:20:33 +0000 2016","id":696896440589750272,"id_str":"696896440589750272","text":"RT @SydneyDoyle24: One of my biggest pet peeves is when people try to spell \"definitely\", but spell it as \"defiantly\".","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":2687466094,"id_str":"2687466094","name":"lil leah","screen_name":"leahxcarr","location":"","description":"do what you love. don't let anyone steal your happiness.","url":"https:\/\/t.co\/g3eWCQtRFc","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/g3eWCQtRFc","expanded_url":"http:\/\/vsco.co\/leahxcarr","display_url":"vsco.co\/leahxcarr","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":405,"friends_count":222,"listed_count":0,"created_at":"Tue Jul 08 15:30:28 +0000 2014","favourites_count":3307,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":845,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"F5ABB5","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/558500103896313856\/Wk0YB1eQ.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/558500103896313856\/Wk0YB1eQ.jpeg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/694778439543439360\/3B5ib_i0_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/694778439543439360\/3B5ib_i0_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/2687466094\/1454399958","profile_link_color":"F5ABB5","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:10:28 +0000 2016","id":696893903841189888,"id_str":"696893903841189888","text":"One of my biggest pet peeves is when people try to spell \"definitely\", but spell it as \"defiantly\".","source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":511060987,"id_str":"511060987","name":"sydney","screen_name":"SydneyDoyle24","location":"Canada","description":"18","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":437,"friends_count":493,"listed_count":2,"created_at":"Thu Mar 01 23:54:08 +0000 2012","favourites_count":9239,"utc_offset":-18000,"time_zone":"Eastern Time (US & Canada)","geo_enabled":true,"verified":false,"statuses_count":9244,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"131516","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/378800000114951118\/6a19a610eb6505dc126510c926235087.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/378800000114951118\/6a19a610eb6505dc126510c926235087.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/658096662330671104\/3cJYok_U_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/658096662330671104\/3cJYok_U_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/511060987\/1445261153","profile_link_color":"2C0263","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"EFEFEF","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":2,"favorite_count":4,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},"is_quote_status":false,"retweet_count":2,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"SydneyDoyle24","name":"sydney","id":511060987,"id_str":"511060987","indices":[3,17]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:19:34 +0000 2016","id":696896191158595584,"id_str":"696896191158595584","text":"Defiantly time for a new phone now, my screen cracked this morning \ud83d\ude43\ud83d\ude12\ud83d\ude21","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":311007243,"id_str":"311007243","name":"kassie","screen_name":"kassaaaaandra","location":"under your bed.","description":"I'm searching for abstract ways of expressing reality, abstract forms that will enlighten my own mystery \u2728","url":"http:\/\/t.co\/F3A1sRu6LN","entities":{"url":{"urls":[{"url":"http:\/\/t.co\/F3A1sRu6LN","expanded_url":"http:\/\/exsancy.tumblr.com","display_url":"exsancy.tumblr.com","indices":[0,22]}]},"description":{"urls":[]}},"protected":false,"followers_count":324,"friends_count":156,"listed_count":1,"created_at":"Sat Jun 04 18:49:00 +0000 2011","favourites_count":1228,"utc_offset":-18000,"time_zone":"Eastern Time (US & Canada)","geo_enabled":true,"verified":false,"statuses_count":19810,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"030303","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/378800000088000887\/d0360ef0b1fe7e4df713b3c9126173fe.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/378800000088000887\/d0360ef0b1fe7e4df713b3c9126173fe.jpeg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/685479244760219648\/7Lz8vOs6_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/685479244760219648\/7Lz8vOs6_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/311007243\/1451172518","profile_link_color":"C838EF","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"F5E8FA","profile_text_color":"141314","profile_use_background_image":false,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":{"id":"66cd2cdb819f4414","url":"https:\/\/api.twitter.com\/1.1\/geo\/id\/66cd2cdb819f4414.json","place_type":"city","name":"Dale City","full_name":"Dale City, VA","country_code":"US","country":"United States","contained_within":[],"bounding_box":{"type":"Polygon","coordinates":[[[-77.443065,38.6109903],[-77.2748133,38.6109903],[-77.2748133,38.68125],[-77.443065,38.68125]]]},"attributes":{}},"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:19:18 +0000 2016","id":696896122674094081,"id_str":"696896122674094081","text":"@evilsuperalice tbh I of the storm remind me a lil more of Sitri but I can defiantly see Dan in it too, I can see Dan in everything","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":696895258676879360,"in_reply_to_status_id_str":"696895258676879360","in_reply_to_user_id":1478660221,"in_reply_to_user_id_str":"1478660221","in_reply_to_screen_name":"evilsuperalice","user":{"id":501254154,"id_str":"501254154","name":"Mock \u2601\ufe0f","screen_name":"BlueRose_weirdo","location":"The trash can :^)","description":"Pleasure to meet ya, call me Mock. I post art sometimes. || 15 || she\/her || HetRom || Gray-Ace|| ||Matching w\/ @citrii || [Multifandom:mostly Makai Ouji trash]","url":"https:\/\/t.co\/atdXRbHvM1","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/atdXRbHvM1","expanded_url":"http:\/\/shslcheshirecat.tumblr.com","display_url":"shslcheshirecat.tumblr.com","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":48,"friends_count":86,"listed_count":1,"created_at":"Thu Feb 23 23:37:57 +0000 2012","favourites_count":2371,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":2028,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"000000","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/691095821916475396\/xuZBOuIy_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/691095821916475396\/xuZBOuIy_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/501254154\/1453605082","profile_link_color":"3B94D9","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"000000","profile_text_color":"000000","profile_use_background_image":false,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"evilsuperalice","name":"dantalion trash","id":1478660221,"id_str":"1478660221","indices":[0,15]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:18:57 +0000 2016","id":696896038116859904,"id_str":"696896038116859904","text":"@Sir_Bmadd @NFL not a Broncos fan but defiantly not a Panthers or Cam fan.","source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":696894982062415872,"in_reply_to_status_id_str":"696894982062415872","in_reply_to_user_id":127744029,"in_reply_to_user_id_str":"127744029","in_reply_to_screen_name":"Sir_Bmadd","user":{"id":730068781,"id_str":"730068781","name":"Cheech","screen_name":"faggotass_chyy","location":"IG: @chy.barragan16","description":"SoCal","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":546,"friends_count":497,"listed_count":3,"created_at":"Wed Aug 01 05:43:44 +0000 2012","favourites_count":19723,"utc_offset":-28800,"time_zone":"Pacific Time (US & Canada)","geo_enabled":true,"verified":false,"statuses_count":45996,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/784586509\/671dab103509e779498c2ffbd3b641e4.png","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/784586509\/671dab103509e779498c2ffbd3b641e4.png","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/687691406610022400\/sie9WEgj_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/687691406610022400\/sie9WEgj_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/730068781\/1454331634","profile_link_color":"0084B4","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":false,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":{"id":"0052e87c791459e6","url":"https:\/\/api.twitter.com\/1.1\/geo\/id\/0052e87c791459e6.json","place_type":"city","name":"Oak Hills","full_name":"Oak Hills, CA","country_code":"US","country":"United States","contained_within":[],"bounding_box":{"type":"Polygon","coordinates":[[[-117.499609,34.358385],[-117.354797,34.358385],[-117.354797,34.469174],[-117.499609,34.469174]]]},"attributes":{}},"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"Sir_Bmadd","name":"SpinCyCle_DownWard","id":127744029,"id_str":"127744029","indices":[0,10]},{"screen_name":"NFL","name":"NFL","id":19426551,"id_str":"19426551","indices":[11,15]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:18:45 +0000 2016","id":696895984945594368,"id_str":"696895984945594368","text":"RT @DeeWildMannered: You are defiantly my favorite notification \ud83d\ude0d","source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":2578720687,"id_str":"2578720687","name":"LOX\u2734","screen_name":"edzellox","location":"mhgnylox","description":"LOX IS REAL","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":2394,"friends_count":366,"listed_count":5,"created_at":"Fri Jun 20 13:46:31 +0000 2014","favourites_count":10085,"utc_offset":14400,"time_zone":"Yerevan","geo_enabled":true,"verified":false,"statuses_count":27797,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"022330","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/618991688493592580\/kDLiBKgS.jpg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/618991688493592580\/kDLiBKgS.jpg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/696534732687802368\/4T2jBfuK_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/696534732687802368\/4T2jBfuK_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/2578720687\/1454639927","profile_link_color":"89C9FA","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"EFEFEF","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Sun Feb 07 03:00:30 +0000 2016","id":696166616430587904,"id_str":"696166616430587904","text":"You are defiantly my favorite notification \ud83d\ude0d","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1373490896,"id_str":"1373490896","name":"DEE BARAJAS","screen_name":"DeeWildMannered","location":"Los Angeles, CA","description":"TURN MY NOTIFICATIONS ON THEN TWEET ME A SCREENSHOT I'LL GIVE YOU MY NUMBER \u263a\ufe0f INSTAGRAM\/SNAPCHAT: DeeWildMannered Business: WildManneredBusiness@Gmail.com","url":"https:\/\/t.co\/AhoxsgI3on","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/AhoxsgI3on","expanded_url":"https:\/\/youtu.be\/PbhSQ_m3K4Y","display_url":"youtu.be\/PbhSQ_m3K4Y","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":47223,"friends_count":10999,"listed_count":106,"created_at":"Tue Apr 23 01:14:34 +0000 2013","favourites_count":57471,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":2131,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/689994631015174144\/qeDwYPwN_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/689994631015174144\/qeDwYPwN_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1373490896\/1454416885","profile_link_color":"0084B4","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":112,"favorite_count":385,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},"is_quote_status":false,"retweet_count":112,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"DeeWildMannered","name":"DEE BARAJAS","id":1373490896,"id_str":"1373490896","indices":[3,19]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:18:11 +0000 2016","id":696895842049916928,"id_str":"696895842049916928","text":"If I wasn't married, I would have a few more tattoos and most defiantly a cute lil septic in my nose!!","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":20063414,"id_str":"20063414","name":"Trac\u00e9e Mahree","screen_name":"Tracee_Maree","location":"Atlanta, Ga \u2708\ufe0f San Diego, Ca","description":"Jesus Enthusiast. Lovatarian. Eccentric Bohemian. Minimalist. Culture Addict. Foodie. Jazz Lover. Day Dreamer. Time Traveler. Self Talker. Random Ranter. #INFJ","url":"https:\/\/t.co\/4EcUlQNrZK","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/4EcUlQNrZK","expanded_url":"http:\/\/personalityjunkie.com\/the-infj\/","display_url":"personalityjunkie.com\/the-infj\/","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":389,"friends_count":413,"listed_count":10,"created_at":"Wed Feb 04 15:57:38 +0000 2009","favourites_count":53,"utc_offset":-21600,"time_zone":"Central Time (US & Canada)","geo_enabled":true,"verified":false,"statuses_count":12427,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"4A913C","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme5\/bg.gif","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme5\/bg.gif","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/620342504383078400\/fzCz4kgo_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/620342504383078400\/fzCz4kgo_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/20063414\/1436715522","profile_link_color":"4A913C","profile_sidebar_border_color":"EEEEEE","profile_sidebar_fill_color":"EFEFEF","profile_text_color":"333333","profile_use_background_image":false,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":{"id":"a592bd6ceb1319f7","url":"https:\/\/api.twitter.com\/1.1\/geo\/id\/a592bd6ceb1319f7.json","place_type":"city","name":"San Diego","full_name":"San Diego, CA","country_code":"US","country":"United States","contained_within":[],"bounding_box":{"type":"Polygon","coordinates":[[[-117.282538,32.53962],[-116.9274403,32.53962],[-116.9274403,33.0804044],[-117.282538,33.0804044]]]},"attributes":{}},"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:18:06 +0000 2016","id":696895824161325056,"id_str":"696895824161325056","text":"RT @SydneyDoyle24: One of my biggest pet peeves is when people try to spell \"definitely\", but spell it as \"defiantly\".","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":499252786,"id_str":"499252786","name":"jord\u2601\ufe0f","screen_name":"jxrdyncrawford","location":"ogg\u2661","description":"rip papa \/\/ swc","url":"https:\/\/t.co\/mnnX6c2cWE","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mnnX6c2cWE","expanded_url":"http:\/\/jordyncrawford.vsco.co","display_url":"jordyncrawford.vsco.co","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":465,"friends_count":330,"listed_count":0,"created_at":"Tue Feb 21 23:01:30 +0000 2012","favourites_count":9805,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":16473,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"000000","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/693513915024740353\/R2obVPMp_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/693513915024740353\/R2obVPMp_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/499252786\/1454784312","profile_link_color":"000000","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":false,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:10:28 +0000 2016","id":696893903841189888,"id_str":"696893903841189888","text":"One of my biggest pet peeves is when people try to spell \"definitely\", but spell it as \"defiantly\".","source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":511060987,"id_str":"511060987","name":"sydney","screen_name":"SydneyDoyle24","location":"Canada","description":"18","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":437,"friends_count":493,"listed_count":2,"created_at":"Thu Mar 01 23:54:08 +0000 2012","favourites_count":9239,"utc_offset":-18000,"time_zone":"Eastern Time (US & Canada)","geo_enabled":true,"verified":false,"statuses_count":9244,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"131516","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/378800000114951118\/6a19a610eb6505dc126510c926235087.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/378800000114951118\/6a19a610eb6505dc126510c926235087.jpeg","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/658096662330671104\/3cJYok_U_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/658096662330671104\/3cJYok_U_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/511060987\/1445261153","profile_link_color":"2C0263","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"EFEFEF","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":2,"favorite_count":4,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},"is_quote_status":false,"retweet_count":2,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"SydneyDoyle24","name":"sydney","id":511060987,"id_str":"511060987","indices":[3,17]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:17:39 +0000 2016","id":696895708868182016,"id_str":"696895708868182016","text":"@kali_mendez @bridgetemersonn defiantly don't \ud83d\ude31","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":696895613309472768,"in_reply_to_status_id_str":"696895613309472768","in_reply_to_user_id":956346336,"in_reply_to_user_id_str":"956346336","in_reply_to_screen_name":"kali_mendez","user":{"id":500968811,"id_str":"500968811","name":"nicole","screen_name":"NicoleSeifert1","location":"","description":"get up on the hood of my daddy's tractor","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":535,"friends_count":785,"listed_count":0,"created_at":"Thu Feb 23 17:15:20 +0000 2012","favourites_count":2958,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":15853,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"020812","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/770594630\/d13af8540d1ac13abee0b3c1cd7d12d4.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/770594630\/d13af8540d1ac13abee0b3c1cd7d12d4.jpeg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/696502695176359937\/-jlPXD5m_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/696502695176359937\/-jlPXD5m_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/500968811\/1454894158","profile_link_color":"17406C","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"020812","profile_text_color":"2280A9","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":1,"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"kali_mendez","name":"kali","id":956346336,"id_str":"956346336","indices":[0,12]},{"screen_name":"bridgetemersonn","name":"Bridget Emerson","id":2555203646,"id_str":"2555203646","indices":[13,29]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:17:10 +0000 2016","id":696895587522838528,"id_str":"696895587522838528","text":"I defiantly have multiple different styles. #Nikes #vans #athletic #chill #prep #differentdaydifferentshoe #cute #g\u2026 https:\/\/t.co\/txSAarYTOZ","source":"\u003ca href=\"http:\/\/ifttt.com\" rel=\"nofollow\"\u003eIFTTT\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":3910326328,"id_str":"3910326328","name":"Mauricio Martinez","screen_name":"MarteMau","location":"Distrito Federal, M\u00e9xico","description":"","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":54,"friends_count":16,"listed_count":48,"created_at":"Fri Oct 09 16:02:06 +0000 2015","favourites_count":0,"utc_offset":-28800,"time_zone":"Pacific Time (US & Canada)","geo_enabled":false,"verified":false,"statuses_count":1270,"lang":"es","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"1A1B1F","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme9\/bg.gif","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme9\/bg.gif","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/664986206933905408\/dMSPKnIh_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/664986206933905408\/dMSPKnIh_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/3910326328\/1447380048","profile_link_color":"2FC2EF","profile_sidebar_border_color":"181A1E","profile_sidebar_fill_color":"252429","profile_text_color":"666666","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[{"text":"Nikes","indices":[44,50]},{"text":"vans","indices":[51,56]},{"text":"athletic","indices":[57,66]},{"text":"chill","indices":[67,73]},{"text":"prep","indices":[74,79]},{"text":"differentdaydifferentshoe","indices":[80,106]},{"text":"cute","indices":[107,112]},{"text":"g","indices":[113,115]}],"symbols":[],"user_mentions":[],"urls":[],"media":[{"id":696895587388690433,"id_str":"696895587388690433","indices":[117,140],"media_url":"http:\/\/pbs.twimg.com\/media\/Cave2CvXEAER_mH.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/Cave2CvXEAER_mH.jpg","url":"https:\/\/t.co\/txSAarYTOZ","display_url":"pic.twitter.com\/txSAarYTOZ","expanded_url":"http:\/\/twitter.com\/MarteMau\/status\/696895587522838528\/photo\/1","type":"photo","sizes":{"large":{"w":1024,"h":768,"resize":"fit"},"medium":{"w":600,"h":450,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"small":{"w":340,"h":255,"resize":"fit"}}}]},"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:17:09 +0000 2016","id":696895585425620992,"id_str":"696895585425620992","text":"i dunno if you mad at me but I defiantly don't know if im mad at you","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":3144200847,"id_str":"3144200847","name":"kc","screen_name":"krystellechloe","location":"","description":"","url":"https:\/\/t.co\/uTHEQIVJ6G","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/uTHEQIVJ6G","expanded_url":"https:\/\/instagram.com\/krystellechloe\/","display_url":"instagram.com\/krystellechloe\/","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":31,"friends_count":35,"listed_count":1,"created_at":"Tue Apr 07 15:27:49 +0000 2015","favourites_count":65,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":389,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"000000","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/684879347958743040\/Trton9zR_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/684879347958743040\/Trton9zR_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/3144200847\/1433744018","profile_link_color":"F5ABB5","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"000000","profile_text_color":"000000","profile_use_background_image":false,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:15:56 +0000 2016","id":696895276234072064,"id_str":"696895276234072064","text":"@iMAV3RIQ Have you heard of a Christian Rapper by the name of NF? If not you should defiantly listen to his stuff. So motivational. #G1GB","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":388895171,"in_reply_to_user_id_str":"388895171","in_reply_to_screen_name":"iMAV3RIQ","user":{"id":3073853007,"id_str":"3073853007","name":"Hacked By JOHN CENA","screen_name":"pain_layn","location":"","description":"A Golden Ticket Brandon Browner Puller! Pack Luck Master! Just Gotta Dabble! #FlyEaglesFly #GFvsBF #PVPSAAANNN Elite Pulls Are Automatic","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":885,"friends_count":2649,"listed_count":2,"created_at":"Fri Mar 06 01:14:54 +0000 2015","favourites_count":5014,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":1286,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/583231208458493952\/G7rLZva5_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/583231208458493952\/G7rLZva5_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/3073853007\/1427157257","profile_link_color":"0084B4","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[{"text":"G1GB","indices":[132,137]}],"symbols":[],"user_mentions":[{"screen_name":"iMAV3RIQ","name":"iMAV3RIQ","id":388895171,"id_str":"388895171","indices":[0,9]}],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"},{"metadata":{"iso_language_code":"en","result_type":"recent"},"created_at":"Tue Feb 09 03:15:02 +0000 2016","id":696895049326600192,"id_str":"696895049326600192","text":"I defiantly need to recruit that! Lol","source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","truncated":false,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1226212255,"id_str":"1226212255","name":"Eric Ayala","screen_name":"EA_Glo","location":"302 ","description":"GG","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":1621,"friends_count":616,"listed_count":4,"created_at":"Wed Feb 27 22:46:24 +0000 2013","favourites_count":3211,"utc_offset":-36000,"time_zone":"Hawaii","geo_enabled":false,"verified":false,"statuses_count":14859,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/pbs.twimg.com\/profile_background_images\/378800000011484828\/68fb71d0c87dd15c0a976a9213eba0ae.jpeg","profile_background_image_url_https":"https:\/\/pbs.twimg.com\/profile_background_images\/378800000011484828\/68fb71d0c87dd15c0a976a9213eba0ae.jpeg","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/696806673227370498\/eCIn422i_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/696806673227370498\/eCIn422i_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1226212255\/1448895390","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"favorited":false,"retweeted":false,"lang":"en"}],"search_metadata":{"completed_in":0.095,"max_id":696896717866823681,"max_id_str":"696896717866823681","next_results":"?max_id=696895049326600191&q=defiantly&include_entities=1","query":"defiantly","refresh_url":"?since_id=696896717866823681&q=defiantly&include_entities=1","count":15,"since_id":0,"since_id_str":"0"}}
