#!/usr/bin/perl

use v5.36;

use Text::Treesitter::Language;
use Text::Treesitter::Parser;

use Convert::Color;
use File::Slurper qw( read_text );
use Getopt::Long;
use String::Tagged;
use String::Tagged::Terminal;

GetOptions(
   'language|l=s'   => \( my $LANGUAGE = "c" ),
   'unrecognised|u' => \( my $PRINT_UNRECOGNISED ),
) or exit 1;

my $LANGUAGE_DIR = "languages/tree-sitter-$LANGUAGE";
my $LANGUAGE_OBJECT = $LANGUAGE_DIR . "/tree-sitter-$LANGUAGE.so";
unless( -f $LANGUAGE_OBJECT ) {
   require Text::Treesitter::Language;
   Text::Treesitter::Language::build( $LANGUAGE_OBJECT, $LANGUAGE_DIR );
}

my $p = Text::Treesitter::Parser->new;
my $lang = Text::Treesitter::Language::load( $LANGUAGE_OBJECT, $LANGUAGE );
$p->set_language( $lang );

my $str = String::Tagged->new( read_text $ARGV[0] // "/dev/stdin" );

my $tree = $p->parse_string( $str );

my %FORMATS = (
   comment    => { bg => "vga:blue", italic => 1 },
   identifier => { fg => "vga:cyan" },
   keyword    => { fg => "vga:yellow", bold => 1 },
   literal    => { fg => "vga:magenta" },
   preproc    => { fg => "vga:blue", bold => 1 },
   type       => { fg => "vga:green" },
);

foreach ( values %FORMATS ) {
   $_->{fg} and
      $_->{fg} = Convert::Color->new( $_->{fg} )->as_xterm;
   $_->{bg} and
      $_->{bg} = Convert::Color->new( $_->{bg} )->as_xterm;
}

my %FORMAT_FOR_NODE = (
   ( map { $_ => $FORMATS{identifier} } qw( field_identifier ) ),
   ( map { $_ => $FORMATS{literal} } qw( char_literal number_literal string_literal true false null ) ),
   ( map { $_ => $FORMATS{preproc} } '"#include"', '"#define"' ),
   ( map { $_ => $FORMATS{type}    } qw( primitive_type type_identifier ) ),

   ( map {; qq("$_"), $FORMATS{keyword} }
      qw( break case else enum for if return sizeof struct switch union until while ) ),

   ( map { $_ => $FORMATS{$_} } qw( comment identifier ) ),

   system_lib_string => { $FORMATS{literal}->%*, italic => 1 },

   storage_class_specifier => { $FORMATS{type}->%*, bold => 1 },
   type_qualifier          => { $FORMATS{type}->%*, bold => 1 },
);

my %UNRECOGNISED_NODES;

sub walk_applying_tags
{
   my ( $node ) = @_;

   my $start = $node->start_byte;
   my $len   = $node->end_byte - $start;

   my $type = $node->is_named ? $node->type : '"'.$node->type.'"';

   return if $type eq qq("\n");

   if( my $format = $FORMAT_FOR_NODE{$type} ) {
      $str->apply_tag( $start, $len, $_, $format->{$_} ) for keys %$format;
   }
   else {
      $UNRECOGNISED_NODES{$type}++;
   }

   foreach my $child ( $node->child_nodes ) {
      walk_applying_tags( $child );
   }
}

walk_applying_tags( $tree->root_node );

foreach my $line ( $str->split( qr/\n/ ) ) {
   String::Tagged::Terminal->new_from_formatting( $line )
      ->say_to_terminal;
}

if( $PRINT_UNRECOGNISED and keys %UNRECOGNISED_NODES ) {
   print STDERR "-------\nUnrecognised:\n";
   foreach ( sort keys %UNRECOGNISED_NODES ) {
      print STDERR "  $_\n";
   }
}
