package IPC::Shareable;

use warnings;
use strict;
use 5.10.0;

use Carp qw(carp croak confess);
use Data::Dumper;
use IPC::Semaphore;
use IPC::Shareable::SharedMem;
use IPC::SysV qw(
                 IPC_PRIVATE
                 IPC_CREAT
                 IPC_EXCL
                 IPC_NOWAIT
                 SEM_UNDO
                 );
use Storable 0.6 qw(freeze thaw);
use Scalar::Util;

our $VERSION = '0.99_01';

use constant {
    LOCK_SH       => 1,
    LOCK_EX       => 2,
    LOCK_NB       => 4,
    LOCK_UN       => 8,

    DEBUGGING     => ($ENV{SHAREABLE_DEBUG} || 0),
    SHM_BUFSIZ    =>  65536,
    SEM_MARKER    =>  0,
    SHM_EXISTS    =>  1
};

require Exporter;
our @ISA = 'Exporter';
our @EXPORT_OK = qw(LOCK_EX LOCK_SH LOCK_NB LOCK_UN);
our %EXPORT_TAGS = (
        all     => [qw( LOCK_EX LOCK_SH LOCK_NB LOCK_UN )],
        lock    => [qw( LOCK_EX LOCK_SH LOCK_NB LOCK_UN )],
        flock   => [qw( LOCK_EX LOCK_SH LOCK_NB LOCK_UN )],
);
Exporter::export_ok_tags('all', 'lock', 'flock');

my %semop_args = (
    (LOCK_EX),
        [       
                1, 0, 0,                        # wait for readers to finish
                2, 0, 0,                        # wait for writers to finish
                2, 1, SEM_UNDO,                 # assert write lock
        ],
    (LOCK_EX|LOCK_NB),
        [
                1, 0, IPC_NOWAIT,               # wait for readers to finish
                2, 0, IPC_NOWAIT,               # wait for writers to finish
                2, 1, (SEM_UNDO | IPC_NOWAIT),  # assert write lock
        ],
    (LOCK_EX|LOCK_UN),
        [
                2, -1, (SEM_UNDO | IPC_NOWAIT),
        ],

    (LOCK_SH),
        [
                2, 0, 0,                        # wait for writers to finish
                1, 1, SEM_UNDO,                 # assert shared read lock
        ],
    (LOCK_SH|LOCK_NB),
        [
                2, 0, IPC_NOWAIT,               # wait for writers to finish
                1, 1, (SEM_UNDO | IPC_NOWAIT),  # assert shared read lock
        ],
    (LOCK_SH|LOCK_UN),
        [
                1, -1, (SEM_UNDO | IPC_NOWAIT), # remove shared read lock
        ],
);
my %default_options = (
                key       => IPC_PRIVATE,
                create    => '',
                exclusive => '',
                destroy   => '',
                mode      => 0666,
                size      => SHM_BUFSIZ,
                );
my %global_register;
my %process_register;

my $persist = 0;
my @persist_ids;

# public methods

sub shlock {
    my ($self, $typelock) = @_;
    ($typelock = LOCK_EX) unless defined $typelock;

    return $self->shunlock if ($typelock & LOCK_UN);

    return 1 if ($self->{_lock} & $typelock);

    # If they have a different lock than they want, release it first
    $self->shunlock if ($self->{_lock});

    my $sem = $self->{_sem};

    my $return_val = $sem->op(@{ $semop_args{$typelock} });
    if ($return_val) {
        $self->{_lock} = $typelock;
        $self->{_data} = _thaw($self->{_shm}),
    }
    return $return_val;
}
sub shunlock {
    my $self = shift;

    return 1 unless $self->{_lock};
    if ($self->{_was_changed}) {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!\n";
        };
        $self->{_was_changed} = 0;
    }
    my $sem = $self->{_sem};
    my $typelock = $self->{_lock} | LOCK_UN;
    $typelock ^= LOCK_NB if ($typelock & LOCK_NB);
    $sem->op(@{ $semop_args{$typelock} });

    $self->{_lock} = 0;

    1;
}
sub clean_up {
    my $class = shift;

    for my $s (values %process_register) {
        if ($s->{_type} eq 'HASH' && $persist) {
            last if _persist_clean();
            next if grep { $_ == $s->{_shm}->id } $s->persist_ids;
        }
        next unless $s->{_opts}->{_owner} == $$;
        remove($s);
    }
}
sub clean_up_all {
    my $class = shift;

    for my $s (values %global_register) {
        if ($s->{_type} eq 'HASH' && $persist) {
            last if _persist_clean();
            next if grep { $_ == $s->{_shm}->id } $s->persist_ids;
        }
        remove($s);
    }
}
sub remove {
    my $self = shift;

    my $s = $self->{_shm};
    my $id = $s->id;

    $s->remove or do {
        croak "Couldn't remove shared memory segment $id: $!";
    };

    $s = $self->{_sem};
    $s->remove or do {
        require Carp;
        croak "Couldn't remove semaphore set $id: $!";
    };
    delete $process_register{$id};
    delete $global_register{$id};
}
sub erase {
    for my $s (values %global_register) {
        shunlock($s);
        remove($s);
    }
}
sub persist_ids {
    return @persist_ids;
}

# magic methods

sub TIESCALAR {
    return _tie(SCALAR => @_);
}
sub TIEARRAY {
    return _tie(ARRAY => @_);
}    
sub TIEHASH {
    return _tie(HASH => @_);
}
sub STORE {
    my $self = shift;

    my $sid = $self->{_shm}->{_id};

    $global_register{$self->{_shm}->id} ||= $self;

    $self->{_data} = _thaw($self->{_shm}) unless ($self->{_lock});
  TYPE: {
      if ($self->{_type} eq 'SCALAR') {
          my $val = shift;
          _mg_tie($self => $val) if _need_tie($val);
          $self->{_data} = \$val;
          last TYPE;
      }
      if ($self->{_type} eq 'ARRAY') {
          my $i   = shift;
          my $val = shift;
          _mg_tie($self => $val) if _need_tie($val);
          $self->{_data}->[$i] = $val;
          last TYPE;
      }   
      if ($self->{_type} eq 'HASH') {
          my $key = shift;
          my $val = shift;
          _mg_tie($self => $val) if _need_tie($val);
          $self->{_data}->{$key} = $val;
          last TYPE;
      }
      croak "Variables of type $self->{_type} not supported";
  }

    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!\n";
        };
    }
    return 1;
}
sub FETCH {
    my $self = shift;

    $global_register{$self->{_shm}->id} ||= $self;

    my $data;
    if ($self->{_lock} || $self->{_iterating}) {
        $self->{_iterating} = ''; # In case we break out
        $data = $self->{_data};
    } else {
        $data = _thaw($self->{_shm});
        $self->{_data} = $data;
    }

    my $val;
  TYPE: {
      if ($self->{_type} eq 'SCALAR') {
          if (defined $data) {
              $val = $$data;
              last TYPE;
          } else {
              return;
          }
      }
      if ($self->{_type} eq 'ARRAY') {
          if (defined $data) {
              my $i = shift;
              $val = $data->[$i];
              last TYPE;
          } else {
              return;
          }
      }
      if ($self->{_type} eq 'HASH') {
          if (defined $data) {
              my $key = shift;
              $val = $data->{$key};
              last TYPE;
          } else {
              return;
          }
      }
      croak "Variables of type $self->{_type} not supported";
  }

    if (my $inner = _is_kid($val)) {
        my $s = $inner->{_shm};
        $inner->{_data} = _thaw($s);
    }
    return $val;

}
sub CLEAR {
    my $self = shift;

    if ($self->{_type} eq 'ARRAY') {
        $self->{_data} = [ ];
    } elsif ($self->{_type} eq 'HASH') {
        $self->{_data} = { };
    } else {
        croak "Attempt to clear non-aggegrate";
    }

    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
}
sub DELETE {
    my $self = shift;
    my $key  = shift;

    $self->{_data} = _thaw($self->{_shm}) unless $self->{_lock};
    my $val = delete $self->{_data}->{$key};
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }

    return $val;
}
sub EXISTS {
    my $self = shift;
    my $key  = shift;

    $self->{_data} = _thaw($self->{_shm}) unless $self->{_lock};
    return exists $self->{_data}->{$key};
}
sub FIRSTKEY {
    my $self = shift;
    my $key  = shift;

    $self->{_iterating} = 1;
    $self->{_data} = _thaw($self->{_shm}) unless $self->{_lock};
    my $reset = keys %{$self->{_data}};
    my $first = each %{$self->{_data}};
    return $first;
}
sub NEXTKEY {
    my $self = shift;

    # caveat emptor if hash was changed by another process
    my $next = each %{$self->{_data}};
    if (not defined $next) {
        $self->{_iterating} = '';
        return;
    } else {
        $self->{_iterating} = 1;
        return $next;
    }
}
sub EXTEND {
    #XXX Noop
}
sub PUSH {
    my $self = shift;

    $global_register{$self->{_shm}->id} ||= $self;
    $self->{_data} = _thaw($self->{_shm}, $self->{_data}) unless $self->{_lock};

    push @{$self->{_data}} => @_;
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
}
sub POP {
    my $self = shift;

    $self->{_data} = _thaw($self->{_shm}, $self->{_data}) unless $self->{_lock};

    my $val = pop @{$self->{_data}};
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
    return $val;
}
sub SHIFT {
    my $self = shift;

    $self->{_data} = _thaw($self->{_shm}, $self->{_data}) unless $self->{_lock};
    my $val = shift @{$self->{_data}};
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
    return $val;
}
sub UNSHIFT {
    my $self = shift;

    $self->{_data} = _thaw($self->{_shm}, $self->{_data}) unless $self->{_lock};
    my $val = unshift @{$self->{_data}} => @_;
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
    return $val;
}
sub SPLICE {
    my($self, $off, $n, @av) = @_;

    $self->{_data} = _thaw($self->{_shm}, $self->{_data}) unless $self->{_lock};
    my @val = splice @{$self->{_data}}, $off, $n => @av;
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
    return @val;
}
sub FETCHSIZE {
    my $self = shift;
    $self->{_data} = _thaw($self->{_shm}) unless $self->{_lock};
    return scalar(@{$self->{_data}});
}
sub STORESIZE {
    my $self = shift;
    my $n    = shift;

    $self->{_data} = _thaw($self->{_shm}) unless $self->{_lock};
    $#{$self->{_data}} = $n - 1;
    if ($self->{_lock} & LOCK_EX) {
        $self->{_was_changed} = 1;
    } else {
        defined _freeze($self->{_shm} => $self->{_data}) or do {
            croak "Could not write to shared memory: $!";
        };
    }
    return $n;
}

# private methods

sub _persist_clean {
    if ($persist){
        return 1 if values %global_register == 1;
    }
    return 0;
}
sub _freeze {
    my $s  = shift;
    my $water = shift;

    my $ice = freeze $water;
    # Could be a large string.  No need to copy it.  substr more efficient
    substr $ice, 0, 0, 'IPC::Shareable';

    if (length($ice) > $s->size) {
        croak "Length of shared data exceeds shared segment size";
    };
    $s->shmwrite($ice);
}
sub _thaw {
    my $s = shift;

    my $ice = $s->shmread;
    return undef if ! defined $ice;

    my $tag = substr $ice, 0, 14, '';

    if ($tag eq 'IPC::Shareable') {
        my $water = thaw $ice;
        defined($water) or do {
            croak "Munged shared memory segment (size exceeded?)";
        };

        # persistence

        if ($persist){
            if ($s->{_type} eq 'HASH') {
                $water->{__ipc__}{persist_ids} ||= $s->id;
                push @persist_ids, $water->{__ipc__}{persist_ids};
            }
            else {
                croak "\nERROR: Data structure consists of a '$s->{_type}'. " .
                      "We can't preserve it persistently. Only hashes and " .
                      "scalars are allowed as values...\n\n";
            }
        }
        else {
            delete $water->{__ipc__}{persist_ids} if $s->{_type} eq 'HASH';
        }

        return $water;
    } else {
        return;
    }
}
sub _tie {
    my $type  = shift;
    my $class = shift;
    my $opts  = _parse_args(@_);

    my $key      = _shm_key($opts);
    my $flags    = _shm_flags($opts);
    my $shm_size = $opts->{size};

    my $s = IPC::Shareable::SharedMem->new($key, $shm_size, $flags, $type);

    defined $s or do {
        croak "Could not create shared memory segment: $!\n";
    };

    my $sem = IPC::Semaphore->new($key, 3, $flags);

    defined $sem or do {
        croak "Could not create semaphore set: $!\n";
    };

    unless ( $sem->op(@{ $semop_args{(LOCK_SH)} }) ) {
        croak "Could not obtain semaphore set lock: $!\n";
    }

    my $sh = {
        _iterating => '',
        _key       => $key,
        _lock      => 0,
        _opts      => $opts,
        _shm       => $s,
        _sem       => $sem,
        _type      => $type,
        _was_changed => 0,
    };

    $sh->{_data} = _thaw($s),

    my $sem_marker = $sem->getval(SEM_MARKER);

    if ($sem_marker != SHM_EXISTS) {
        $process_register{$sh->{_shm}->id} ||= $sh;

        $sem->setval(SEM_MARKER, SHM_EXISTS) or do {
            croak "Couldn't set semaphore during object creation: $!";
          };
    }

    $sem->op(@{ $semop_args{(LOCK_SH|LOCK_UN)} });

    return bless $sh => $class;
}
sub _parse_args {
    my($proto, $opts) = @_;

    $proto = defined $proto ? $proto :  0;
    $opts  = defined $opts  ? $opts  : { %default_options };

    if (ref $proto eq 'HASH') {
        $opts = $proto;
    } else {
        $opts->{key} = $proto;
    }

    for my $k (keys %default_options) {
        if (! defined $opts->{$k}) {
            $opts->{$k} = $default_options{$k};
        } elsif ($opts->{$k} eq 'no') {
            if ($^W) {
                carp("Use of `no' in IPC::Shareable args is obsolete");
            }

            $opts->{$k} = '';
        }
    }

    $persist = $opts->{persist} //= 0;
    $opts->{_owner} = ($opts->{_owner} or $$);
    $opts->{_magic} = ($opts->{_magic} or '');

    return $opts;
}
sub _shm_key {
    my $hv = shift;
    my $val = ($hv->{key} or '');

    if ($val eq '') {
        return IPC_PRIVATE;
    } elsif ($val =~ /^\d+$/) {
        return $val;
    } else {
        # XXX This only uses the first four characters
        $val = pack   A4 => $val;
        $val = unpack i  => $val;
        return $val;
    }
}
sub _shm_flags {
    # --- Parses the anonymous hash passed to constructors; returns a list
    # --- of args suitable for passing to shmget
    my $hv = shift;
    my $flags = 0;
    
    $flags |= IPC_CREAT if $hv->{create};
    $flags |= IPC_EXCL  if $hv->{exclusive};
    $flags |= ($hv->{mode} or 0666);

    return $flags;
}
sub _mg_tie {
    my $dad = shift;
    my $val = shift;

    # XXX How to generate a unique id ?
    my $key;
    if ($dad->{_key} == IPC_PRIVATE) {
        $key = IPC_PRIVATE;
    } else {
        $key = int(rand(1_000_000));
    }
    my %opts = (
                %{$dad->{_opts}},
                key       => $key,
                exclusive => 'yes',
                create    => 'yes',
                _magic    => 'yes'
               );

    # XXX I wish I didn't have to take a copy of data here and copy it back in
    # XXX Also, have to peek inside potential objects to see their implementation
    my $kid;
    my $type = Scalar::Util::reftype( $val ) || '';
    if ($type eq "SCALAR") {
        my $copy = $$val;
        $kid = tie $$val => 'IPC::Shareable', $key, { %opts } or do {
            croak "Could not create inner tie";
        };
        $$val = $copy;
    } elsif ($type eq "ARRAY") {
        my @copy = @$val;
        $kid = tie @$val => 'IPC::Shareable', $key, { %opts } or do {
            croak "Could not create inner tie";
        };
        @$val = @copy;
    } elsif ($type eq "HASH") {
        my %copy = %$val;
        $kid = tie %$val => 'IPC::Shareable', $key, { %opts } or do {
            croak "Could not create inner tie";
        };
        %$val = %copy;
    } else {
        croak "Variables of type $type not implemented";
    }

    return $kid;
}
sub _is_kid {
    my $data = shift or return;

    my $type = Scalar::Util::reftype( $data );
    return unless $type;

    my $obj;
    if ($type eq "HASH") {
        $obj = tied %$data;
    } elsif ($type eq "ARRAY") { 
        $obj = tied @$data;
    } elsif ($type eq "SCALAR") {
        $obj = tied $$data;
    }

    if (ref $obj eq 'IPC::Shareable') {
        return $obj;
    } else {
        return;
    }
}
sub _need_tie {
    my $val = shift;

    my $type = Scalar::Util::reftype( $val );
    return unless $type;
    if ($type eq "SCALAR") {
        return !(tied $$val);
    } elsif ($type eq "ARRAY") {
        return !(tied @$val);
    } elsif ($type eq "HASH") {
        return !(tied %$val);
    } else {
        return;
    }
}

END {
    for my $s (values %process_register) {
        if ($s->{_type} eq 'HASH' && $persist) {
            last if _persist_clean();
            next if grep { $_ == $s->{_shm}->id } $s->persist_ids;
        }
        next unless $s->{_opts}->{destroy};
        next unless $s->{_opts}->{_owner} == $$;

        shunlock($s);
        remove($s);
    }
}

1;

__END__

=head1 NAME

IPC::Shareable - Share variables across separate processes

=head1 SYNOPSIS

=head1 DESCRIPTION

IPC::Shareable allows you to tie a variable to shared memory making it
easy to share the contents of that variable with other Perl processes.

Scalars, arrays, and hashes can be tied.  The variable being tied may
contain arbitrarily complex data structures - including references to
arrays, hashes of hashes, etc.

The association between variables in distinct processes is provided by
GLUE.  This is an integer number or 4 character string[1] that serves
as a common identifier for data across process space.  Hence the
statement

 tie $scalar, 'IPC::Shareable', 'data';

in program one and the statement

 tie $variable, 'IPC::Shareable', 'data';

in program two will bind $scalar in program one and $variable in
program two.

There is no pre-set limit to the number of processes that can bind to
data; nor is there a pre-set limit to the complexity of the underlying
data of the tied variables[2].  The amount of data that can be shared
within a single bound variable is limited by the system's maximum size
for a shared memory segment (the exact value is system-dependent).

The bound data structures are all linearized (using Raphael Manfredi's
Storable module) before being slurped into shared memory.  Upon
retrieval, the original format of the data structure is recovered.
Semaphore flags can be used for locking data between competing processes.

=head1 OPTIONS

Options are specified by passing a reference to a hash as the fourth
argument to the tie() function that enchants a variable.
Alternatively you can pass a reference to a hash as the third
argument; IPC::Shareable will then look at the field named B<key> in
this hash for the value of GLUE.  So,

 tie $variable, 'IPC::Shareable', 'data', \%options;

is equivalent to

 tie $variable, 'IPC::Shareable', { key => 'data', ... };

The following fields are recognized in the options hash.

=head2 key

The B<key> field is used to determine the GLUE when using the
three-argument form of the call to tie().  This argument is then, in
turn, used as the KEY argument in subsequent calls to shmget() and
semget().

The default value is C<IPC_PRIVATE>, meaning that your variables cannot
be shared with other processes.

Default: B<IPC_PRIVATE>

=head2 create

B<create> is used to control whether calls to C<tie()> create new shared
memory segments or not.  If B<create> is set to a true value,
L<IPC::Shareable> will create a new binding associated with GLUE as
needed.  If B<create> is false, C<IPC::Shareable> will not attempt to
create a new shared memory segment associated with GLUE.  In this
case, a shared memory segment associated with GLUE must already exist
or the call to C<tie()> will fail and return undef.

Default: B<false>

=head2 exclusive

If B<exclusive> field is set to a true value, calls to C<tie()> will fail
(returning C<undef>) if a data binding associated with GLUE already
exists.  If set to a false value, calls to C<tie()> will succeed even if
a shared memory segment associated with GLUE already exists.

Default: B<false>

=head2 persist

This option if set to a true value will allow you to retain access to the
shared memory segments underlying the data, even after all processes finish.

Meaning that you can regain access to the shared data at any later time. This
feature also allows you to share data between completely separate processes
(ie: two separate scripts running in separate windows), not only between
forked processes.

B<NOTE>: Only hashes can be used in B<persist> mode, and all values of the
top-level hash must be either scalar or hash reference values. If you try to
save any other structure underneath of your hash, we C<croak()>.

Default: B<false>

=head2 mode

The B<mode> argument is an octal number specifying the access
permissions when a new data binding is being created.  These access
permission are the same as file access permissions in that C<0666> is
world readable, C<0600> is readable only by the effective UID of the
process creating the shared variable, etc.

Default: B<0666> (world read and writeable)

=head2 destroy

If set to a true value, the shared memory segment underlying the data
binding will be removed when the process calling C<tie()> exits (gracefully)[3].

Only those memory segments that were created by the current process will be
removed.

Use this option with care.  In particular you should not use this option in a
program that will fork after binding the data.  On the other hand, shared memory
is a finite resource and should be released if it is not needed.

B<NOTE>: If the B<persist> option is set to a true value, the memory segments
used to house the shared data structure will *NOT* be removed, even if
C<destroy> is set to a true value.

Default: B<false>

=head2 size

This field may be used to specify the size of the shared memory segment
allocated.

Default: C<IPC::Shareable::SHM_BUFSIZ()> (ie. B<65536>)

=head2 Default Option Values

Default values for options are:

     key       => IPC_PRIVATE,
     create    => 0,
     exclusive => 0,
     persist   => 0,
     destroy   => 0,
     mode      => 0,
     size      => IPC::Shareable::SHM_BUFSIZ(),

=head1 METHODS

=head2 erase

Class/tied object call.

Example:

    IPC::Shareable->erase;

    # or

    tied($var)->erase;

    # or

    $knot->erase;

This method will absolutely and completely remove all semaphores and shared
memory segments related to the shared data, regardless of which process created
them, or whether the B<destroy> parameter is set.

It's primary use is for when using the B<persist> option, and you know for fact
that all other work in all other processes is complete.

=head2 persist_ids

Class/tied object call. Example:

    my @ids = IPC::Shareable->persist_ids;

    # or

    my @ids = tied($var)->persist_ids;

    # or

    my @ids = $knot->persist_ids;

Returns an array of IDs that are currently registered for the persistent parts
of the data structure, when the B<persist> option has been used.

=head2 shlock($type)

Obtains a lock on the shared memory.  C<$type> specifies the type
of lock to acquire.  If C<$type> is not specified, an exclusive
read/write lock is obtained.  Acceptable values for C<$type> are
the same as for the C<flock()> system call.

Returns C<true> on success, and C<undef> on error.  For non-blocking calls
(see below), the method returns C<0> if it would have blocked.

Obtain an exclusive lock like this:

        tied($var)->lock(LOCK_EX); # same as default

Only one process can hold an exclusive lock on the shared memory at
a given time.

Obtain a shared lock this this:

        tied($var)->lock(LOCK_SH);

Multiple processes can hold a shared lock at a given time.  If a process
attempts to obtain an exclusive lock while one or more processes hold
shared locks, it will be blocked until they have all finished.

Either of the locks may be specified as non-blocking:

        tied($var)->lock( LOCK_EX|LOCK_NB );
        tied($var)->lock( LOCK_SH|LOCK_NB );

A non-blocking lock request will return C<0> if it would have had to
wait to obtain the lock.

Note that these locks are advisory (just like flock), meaning that
all cooperating processes must coordinate their accesses to shared memory
using these calls in order for locking to work.  See the C<flock()> call for
details.

Locks are inherited through forks, which means that two processes actually
can possess an exclusive lock at the same time.  Don't do that.

The constants C<LOCK_EX>, C<LOCK_SH>, C<LOCK_NB>, and C<LOCK_UN> are available
for import using any of the following export tags:

        use IPC::Shareable qw(:lock);
        use IPC::Shareable qw(:flock);
        use IPC::Shareable qw(:all);

Or, just use the flock constants available in the Fcntl module.

See L</LOCKING> for further details.

=head2 shunlock()

Removes a lock. Takes no parameters, returns C<true> on success.

This is equivalent of calling C<shlock(LOCK_UN)>.

See L</LOCKING> for further details.

=head1 LOCKING

IPC::Shareable provides methods to implement application-level
advisory locking of the shared data structures.  These methods are
called C<shlock()> and C<shunlock()>.  To use them you must first get the
object underlying the tied variable, either by saving the return
value of the original call to C<tie()> or by using the built-in C<tied()>
function.

To lock and subsequently unlock a variable, do this:

    my $knot = tie my $scalar, 'IPC::Shareable', $glue, { %options };

    $knot->shlock;
    $scalar = 'foo';
    $knot->shunlock;

or equivalently, if you've decided to throw away the return of C<tie()>:

    tie my $scalar, 'IPC::Shareable', $glue, { %options };

    tied($sv)->shlock;
    $scalar = 'foo';
    tied($sv)->shunlock;

This will place an exclusive lock on the data of C<$scalar>.  You can
also get shared locks or attempt to get a lock without blocking.

L<IPC::Shareable> makes the constants C<LOCK_EX>, C<LOCK_SH>, C<LOCK_UN>, and
C<LOCK_NB> exportable to your address space with the export tags
C<:lock>, C<:flock>, or C<:all>.  The values should be the same as
the standard C<flock> option arguments.

    if (tied($scalar)->shlock(LOCK_SH|LOCK_NB)){
        print "The value is $scalar\n";
        tied($scalar)->shunlock;
    } else {
        print "Another process has an exlusive lock.\n";
    }

If no argument is provided to C<shlock>, it defaults to C<LOCK_EX>.  To
unlock a variable do this:

There are some pitfalls regarding locking and signals about which you
should make yourself aware; these are discussed in L</NOTES>.

If you use the advisory locking, IPC::Shareable assumes that you know
what you are doing and attempts some optimizations.  When you obtain
a lock, either exclusive or shared, a fetch and thaw of the data is
performed.  No additional fetch/thaw operations are performed until
you release the lock and access the bound variable again.  During the
time that the lock is kept, all accesses are perfomed on the copy in
program memory.  If other processes do not honor the lock, and update
the shared memory region unfairly, the process with the lock will not be in
sync.  In other words, IPC::Shareable does not enforce the lock
for you.  

A similar optimization is done if you obtain an exclusive lock.
Updates to the shared memory region will be postponed until you
release the lock (or downgrade to a shared lock).

Use of locking can significantly improve performance for operations
such as iterating over an array, retrieving a list from a slice or 
doing a slice assignment.

=head1 REFERENCES

When a reference to a non-tied scalar, hash, or array is assigned to a
C<tie()>d variable, C<IPC::Shareable> will attempt to C<tie()> the thingy being
referenced[4].  This allows disparate processes to see changes to not
only the top-level variable, but also changes to nested data.  This
feature is intended to be transparent to the application, but there
are some caveats to be aware of.

First of all, C<IPC::Shareable> does not (yet) guarantee that the ids
shared memory segments allocated automagically are unique.  The more
automagical C<tie()>ing that happens, the greater the chance of a
collision.

Secondly, since a new shared memory segment is created for each thingy
being referenced, the liberal use of references could cause the system
to approach its limit for the total number of shared memory segments
allowed.

=head1 OBJECTS

L<IPC::Shareable> implements C<tie()>ing objects to shared memory too.
Since an object is just a reference, the same principles (and caveats)
apply to C<tie()>ing objects as other reference types (see L</REFERENCES>).

=head1 DESTRUCTION

perl(1) will destroy the object underlying a tied variable when then
tied variable goes out of scope.  Unfortunately for L<IPC::Shareable>,
this may not be desirable: other processes may still need a handle on
the relevant shared memory segment.

L<IPC::Shareable> therefore provides several options to control the timing of
removal of shared memory segments.

=head2 destroy Option

As described in L</OPTIONS>, specifying the B<destroy> option when
C<tie()>ing a variable coerces L<IPC::Shareable> to remove the underlying
shared memory segment when the process calling C<tie()> exits gracefully.

B<NOTE>: The destruction is handled in an C<END> block. Only those memory
segments that are tied to the current process will be removed.

B<NOTE>: If the B<persist> option is specified on a data structure, we will
remove only the memory segments that aren't required for persistent storage
of the variable.

=head2 remove()

    $knot->remove;

    # or

    tied($var)->remove;

Calling C<remove()> on the object underlying a C<tie()>d variable removes
the associated shared memory segments.  The segment is removed
irrespective of whether it has the B<destroy> option set or not and
irrespective of whether the calling process created the segment.

B<NOTE>: If the B<persist> option is set, we will retain just enough memory
segments to keep the shared data available for future runs.

=head2 clean_up()

    IPC::Shareable->clean_up;

    # or

    tied($var)->clean_up;

    # or

    $knot->clean_up;

This is a class method that provokes L<IPC::Shareable> to remove all
shared memory segments created by the process.  Segments not created
by the calling process are not removed. We'll retain the minimum segments for
persistent storage if B<persist> is set to a true value.

=head2 clean_up_all()

    IPC::Shareable->clean_up_all;

    # or

    tied($var)->clean_up_all;

    # or

    $knot->clean_up_all

This is a class method that provokes L<IPC::Shareable> to remove all
shared memory segments encountered by the process.  Segments are
removed even if they were not created by the calling process. We'll retain the
minimum segments for persistent storage if B<persist> is set to a true value.

=head1 RETURN VALUES

Calls to C<tie()> that try to implement L<IPC::Shareable> will return an
instance of C<IPC::Shareable> on success, and C<undef> otherwise.

=head1 AUTHOR

Benjamin Sugars <bsugars@canoe.ca>

=head1 NOTES

=head2 Footnotes from the above sections

=over 4

=item 1

If GLUE is longer than 4 characters, only the 4 most significant
characters are used.  These characters are turned into integers by
unpack()ing them.  If GLUE is less than 4 characters, it is space
padded.

=item 2

IPC::Shareable provides no pre-set limits, but the system does.
Namely, there are limits on the number of shared memory segments that
can be allocated and the total amount of memory usable by shared
memory.

=item 3

If the process has been smoked by an untrapped signal, the binding
will remain in shared memory.  If you're cautious, you might try

 $SIG{INT} = \&catch_int;
 sub catch_int {
     die;
 }
 ...
 tie $variable, IPC::Shareable, 'data', { 'destroy' => 'Yes!' };

which will at least clean up after your user hits CTRL-C because
IPC::Shareable's END method will be called.  Or, maybe you'd like to
leave the binding in shared memory, so subsequent process can recover
the data...

=item 4

This behaviour is markedly different from previous versions of
IPC::Shareable.  Older versions would sometimes tie() referenced
thingies, and sometimes not.  The new approach is more reliable (I
think) and predictable (certainly) but uses more shared memory
segments.

=back

=head2 General Notes

=over 4

=item o

When using C<shlock()> to lock a variable, be careful to guard against
signals.  Under normal circumstances, C<IPC::Shareable>'s C<END> method
unlocks any locked variables when the process exits.  However, if an
untrapped signal is received while a process holds an exclusive lock,
C<DESTROY> will not be called and the lock may be maintained even though
the process has exited.  If this scares you, you might be better off
implementing your own locking methods.  

One advantage of using C<flock> on some known file instead of the
locking implemented with semaphores in C<IPC::Shareable> is that when a
process dies, it automatically releases any locks.  This only happens
with C<IPC::Shareable> if the process dies gracefully.

The alternative
is to attempt to account for every possible calamitous ending for your
process (robust signal handling in Perl is a source of much debate,
though it usually works just fine) or to become familiar with your
system's tools for removing shared memory and semaphores.  This
concern should be balanced against the significant performance
improvements you can gain for larger data structures by using the
locking mechanism implemented in IPC::Shareable.

=item o

There is a program called C<ipcs>(1/8) (and C<ipcrm>(1/8)) that is
available on at least Solaris and Linux that might be useful for
cleaning moribund shared memory segments or semaphore sets produced
by bugs in either IPC::Shareable or applications using it.

Examples:

    # list all semaphores and memory segments in use on the system

    ipcs -a

    # list all memory segments along with each one's associated process ID

    ipcs -ap

    # remove *all* semaphores and memory segments

    ipcrm -a

=item o

This version of L<IPC::Shareable> does not understand the format of
shared memory segments created by versions prior to C<0.60>.  If you try
to tie to such segments, you will get an error.  The only work around
is to clear the shared memory segments and start with a fresh set.

=item o

Iterating over a hash causes a special optimization if you have not
obtained a lock (it is better to obtain a read (or write) lock before
iterating over a hash tied to LShareable>, but we attempt this
optimization if you do not).

The C<fetch>/C<thaw> operation is performed
when the first key is accessed.  Subsequent key and and value
accesses are done without accessing shared memory.  Doing an
assignment to the hash or fetching another value between key
accesses causes the hash to be replaced from shared memory.  The
state of the iterator in this case is not defined by the Perl
documentation.  Caveat Emptor.

=back

=head1 CREDITS

Thanks to all those with comments or bug fixes, especially

    Steve Bertrand      <steveb@cpan.org>
    Maurice Aubrey      <maurice@hevanet.com>
    Stephane Bortzmeyer <bortzmeyer@pasteur.fr>
    Doug MacEachern     <dougm@telebusiness.co.nz>
    Robert Emmery       <roberte@netscape.com>
    Mohammed J. Kabir   <kabir@intevo.com>
    Terry Ewing         <terry@intevo.com>
    Tim Fries           <timf@dicecorp.com>
    Joe Thomas          <jthomas@women.com>
    Paul Makepeace      <Paul.Makepeace@realprogrammers.com>
    Raphael Manfredi    <Raphael_Manfredi@pobox.com>
    Lee Lindley         <Lee.Lindley@bigfoot.com>
    Dave Rolsky         <autarch@urth.org>

=head1 SEE ALSO

L<perltie>, L<Storable>, C<shmget>, C<ipcs>, C<ipcrm> and other SysV IPC manual
pages.
