use Mojolicious::Lite;
use Mojo::Loader 'data_section';
use Time::Piece;
use Test::More;
use Mojo::WebService::Twitter;
use Mojo::Util 'b64_encode';

use constant TEST_API_KEY => 'testapikey';
use constant TEST_API_SECRET => 'testapisecret';
use constant TEST_ACCESS_TOKEN => 'testaccesstoken';

my %tweet_data = (
	'657606994744860672' => {
		data_section => 'tweet',
		coordinates => undef,
		created_at => scalar gmtime(1445620699),
		favorites => 382,
		in_reply_to_screen_name => undef,
		in_reply_to_status_id => undef,
		in_reply_to_user_id => undef,
		media => [{id => '657606857159143425', media_url => 'https://pbs.twimg.com/tweet_video_thumb/CSBJ89LU8AEPKrl.png', type => 'animated_gif'}],
		retweets => 289,
		text => q{The @Space_Station crew works  ~9.5 hours a day, with ~4 hours free time during the week... https://t.co/2cdarZPqND https://t.co/HlAnj3eTKk},
		user_id => '1451773004',
	},
	'657324783294676992' => {
		data_section => 'tweet_reply',
		coordinates => undef,
		created_at => scalar gmtime(1445553415),
		favorites => 21,
		in_reply_to_screen_name => 'starlingLX',
		in_reply_to_status_id => '657257155855294465',
		in_reply_to_user_id => '348968125',
		media => [],
		retweets => 13,
		text => q{@starlingLX @bnjacobs @StationCDRKelly @Flickr The higher-res is added but doesn't have @StationCDRKelly's edits...https://t.co/wnKeCAdLfg},
		user_id => '1451773004',
	},
	'657627567948587008' => {
		data_section => 'tweet_retweet',
		coordinates => undef,
		created_at => scalar gmtime(1445625604),
		favorites => 0,
		in_reply_to_screen_name => undef,
		in_reply_to_status_id => undef,
		in_reply_to_user_id => undef,
		media => [{id => '657618738447958017', media_url => 'https://pbs.twimg.com/media/CSBUwibUwAEg0fF.jpg', type => 'photo'}],
		retweets => 30000,
		text => q{RT @StationCDRKelly: Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https://t.co/6LP2xCYcGD},
		user_id => '1451773004',
	},
);

my %user_data = (
	'1451773004' => {
		data_section => 'user',
		created_at => scalar gmtime(1369322728),
		description => q{NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.},
		followers => 314219,
		friends => 181,
		last_tweet_ts => scalar gmtime(1445625604),
		name => 'Intl. Space Station',
		protected => 0,
		screen_name => 'Space_Station',
		statuses => 3228,
		url => 'https://t.co/9Gk2H0gekn',
		verified => 1,
	},
);

post '/oauth2/token' => sub {
	my $c = shift;
	my $auth_header = $c->req->headers->authorization;
	my $token = b64_encode(TEST_API_KEY . ':' . TEST_API_SECRET, '');
	is $auth_header, "Basic $token", 'received API key and secret';
	$c->render(format => 'json', text => '{"token_type":"bearer","access_token":"'.TEST_ACCESS_TOKEN.'"}');
};
group {
	under '/api' => sub {
		my $c = shift;
		my $auth_header = $c->req->headers->authorization;
		is $auth_header, 'Bearer '.TEST_ACCESS_TOKEN, 'received access token';
		return 1;
	};
	get '/statuses/show.json' => sub {
		my $c = shift;
		my $id = $c->param('id');
		die "Unknown tweet ID $id" unless defined $id and exists $tweet_data{$id};
		my $data_section = $tweet_data{$id}{data_section} // 'tweet';
		$c->render(format => 'json', text => data_section('main', $data_section));
	};
	get '/users/show.json' => sub {
		my $c = shift;
		my $id;
		if ($id = $c->param('user_id')) {
			die "Unknown user ID $id" unless exists $user_data{$id};
		} elsif (my $name = $c->param('screen_name')) {
			($id) = grep { lc $user_data{$_}{screen_name} eq lc $name } keys %user_data;
			die "Unknown user screen name $name" unless defined $id;
		}
		my $data_section = $user_data{$id}{data_section} // 'user';
		$c->render(format => 'json', text => data_section('main', $data_section));
	};
	get '/search/tweets.json' => { format => 'json', text => data_section('main', 'search_tweets') };
};

my $api_key = $ENV{TWITTER_API_KEY};
my $api_secret = $ENV{TWITTER_API_SECRET};

if ($ENV{AUTHOR_TESTING} and defined $api_key and defined $api_secret) {
	diag 'Running online test for Twitter';
} else {
	diag 'Running offline test for Twitter; set AUTHOR_TESTING and TWITTER_API_KEY/TWITTER_API_SECRET for online test';
	$Mojo::WebService::Twitter::Util::API_BASE_URL = '/api/';
	$Mojo::WebService::Twitter::Util::OAUTH_BASE_URL = '/oauth/';
	$Mojo::WebService::Twitter::Util::OAUTH2_BASE_URL = '/oauth2/';
	$api_key = TEST_API_KEY;
	$api_secret = TEST_API_SECRET;
}

my $twitter = Mojo::WebService::Twitter->new;
$twitter->ua->server->app->log->level('warn');

ok !eval { $twitter->get_tweet("657618739492474880"); 1 }, 'no API key set';
is $twitter->api_key($api_key)->api_key, $api_key, 'set API key';
is $twitter->api_secret($api_secret)->api_secret, $api_secret, 'set API secret';

my $access;
ok(eval { $access = $twitter->request_oauth2; 1 }, 'requested OAuth 2 access token') or diag $@;
ok defined($access->{access_token}), 'received access token';
$twitter->authentication(oauth2 => $access->{access_token});

foreach my $id (keys %tweet_data) {
	my $data = $tweet_data{$id};
	my $tweet;
	ok(eval { $tweet = $twitter->get_tweet($id); 1 }, "retrieved tweet $id") or diag $@;
	is $tweet->id, $id, 'right tweet ID';
	is_deeply $tweet->coordinates, $data->{coordinates}, 'right coordinates';
	is $tweet->created_at, $data->{created_at}, 'right creation timestamp';
	ok $tweet->favorites >= $data->{favorites}, "at least $data->{favorites} favorites";
	is $tweet->in_reply_to_screen_name, $data->{in_reply_to_screen_name}, 'right in-reply-to screen name';
	is $tweet->in_reply_to_status_id, $data->{in_reply_to_status_id}, 'right in-reply-to status ID';
	is $tweet->in_reply_to_user_id, $data->{in_reply_to_user_id}, 'right in-reply-to user ID';
	is scalar(@{$tweet->media}), scalar(@{$data->{media}}), 'right media';
	foreach my $i (0..$#{$data->{media}}) {
		my $tweet_media = $tweet->media->[$i];
		my $data_media = $data->{media}[$i];
		ok defined $tweet_media, "media $i exists";
		is $tweet_media->id, $data_media->{id}, 'right media ID';
		is $tweet_media->type, $data_media->{type}, 'right media type';
		is $tweet_media->media_url, $data_media->{media_url}, 'right media URL';
	}
	ok $tweet->retweets >= $data->{retweets}, "at least $data->{retweets} retweets";
	is $tweet->text, $data->{text}, 'right text';
	is $tweet->user->id, $data->{user_id}, 'right user';
}

foreach my $id (keys %user_data) {
	my $data = $user_data{$id};
	my $name = $data->{screen_name};
	my $user;
	ok(eval { $user = $twitter->get_user(user_id => $id); 1 }, "retrieved user $id") or diag $@;
	is $user->id, $id, 'right user ID';
	is lc $user->screen_name, lc $name, 'right user screen name';
	my $user2;
	ok(eval { $user2 = $twitter->get_user(screen_name => $name); 1 }, "retrieved user $name") or diag $@;
	is $user2->id, $id, 'right user ID';
	is lc $user2->screen_name, lc $name, 'right user screen name';
	
	is $user->created_at, $data->{created_at}, 'right creation timestamp';
	is $user->description, $data->{description}, 'right description';
	ok $user->followers >= $data->{followers}, "at least $data->{followers} followers";
	ok $user->friends >= $data->{friends}, "at least $data->{friends} friends";
	is $user->name, $data->{name}, 'right name';
	ok !($user->protected xor $data->{protected}), 'right protected status';
	ok $user->statuses >= $data->{statuses}, "at least $data->{statuses} statuses";
	is $user->url, $data->{url}, 'right url';
	ok !($user->verified xor $data->{verified}), 'right verified status';
	ok $user->last_tweet->created_at >= $data->{last_tweet_ts}, "last tweet after $data->{last_tweet_ts}";
}

my $search_term = 'defiantly';
my $tweets;
ok(eval { $tweets = $twitter->search_tweets($search_term); 1 }, "searched tweets for \"$search_term\"") or diag $@;
ok scalar(@$tweets), 'retrieved ' . @$tweets . ' search results';
my $tweet_text = $tweets->[0]->text . ($tweets->[0]->source->{retweeted_status}{full_text} // '');
like $tweet_text, qr/defiantly/i, 'search result contains search term';

done_testing;

__DATA__

@@ user
{"contributors_enabled":false,"created_at":"Thu May 23 15:25:28 +0000 2013","default_profile":false,"default_profile_image":false,"description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","entities":{"description":{"urls":[]},"url":{"urls":[{"display_url":"nasa.gov\/station","expanded_url":"http:\/\/www.nasa.gov\/station","indices":[0,23],"url":"https:\/\/t.co\/9Gk2H0gekn"}]}},"favourites_count":8225,"follow_request_sent":null,"followers_count":2213478,"following":null,"friends_count":181,"geo_enabled":false,"has_extended_profile":false,"id":1451773004,"id_str":"1451773004","is_translation_enabled":false,"is_translator":false,"lang":"en","listed_count":9126,"location":"Low Earth Orbit","name":"Intl. Space Station","notifications":null,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1533048257","profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_link_color":"0084B4","profile_location":null,"profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"protected":false,"screen_name":"Space_Station","status":{"contributors":null,"coordinates":null,"created_at":"Tue Jul 31 21:45:22 +0000 2018","display_text_range":[0,140],"entities":{"hashtags":[],"symbols":[],"urls":[],"user_mentions":[{"id":238285207,"id_str":"238285207","indices":[3,11],"name":"NASA Education","screen_name":"NASAedu"},{"id":879153462,"id_str":"879153462","indices":[14,26],"name":"Ricky Arnold","screen_name":"astro_ricky"},{"id":805224600,"id_str":"805224600","indices":[45,55],"name":"Las Cruces Museums","screen_name":"LCmuseums"},{"id":1451773004,"id_str":"1451773004","indices":[110,124],"name":"Intl. Space Station","screen_name":"Space_Station"}]},"favorite_count":0,"favorited":false,"full_text":"RT @NASAedu: .@astro_ricky tells students at @LCmuseums about the two types of plants he's growing aboard the @Space_Station, and also abou…","geo":null,"id":1024410717292847104,"id_str":"1024410717292847104","in_reply_to_screen_name":null,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"is_quote_status":false,"lang":"en","place":null,"retweet_count":31,"retweeted":false,"retweeted_status":{"contributors":null,"coordinates":null,"created_at":"Tue Jul 31 20:43:45 +0000 2018","display_text_range":[0,192],"entities":{"hashtags":[{"indices":[162,176],"text":"STEMonStation"},{"indices":[177,192],"text":"TeacherOnBoard"}],"media":[{"display_url":"pic.twitter.com\/WOEfNJ03ns","expanded_url":"https:\/\/twitter.com\/NASAedu\/status\/1024395212372639744\/video\/1","id":1024395107968012288,"id_str":"1024395107968012288","indices":[193,216],"media_url":"http:\/\/pbs.twimg.com\/media\/DjdiAwAV4AEPji0.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/DjdiAwAV4AEPji0.jpg","sizes":{"large":{"h":720,"resize":"fit","w":1280},"medium":{"h":675,"resize":"fit","w":1200},"small":{"h":383,"resize":"fit","w":680},"thumb":{"h":150,"resize":"crop","w":150}},"type":"photo","url":"https:\/\/t.co\/WOEfNJ03ns"}],"symbols":[],"urls":[],"user_mentions":[{"id":879153462,"id_str":"879153462","indices":[1,13],"name":"Ricky Arnold","screen_name":"astro_ricky"},{"id":805224600,"id_str":"805224600","indices":[32,42],"name":"Las Cruces Museums","screen_name":"LCmuseums"},{"id":1451773004,"id_str":"1451773004","indices":[97,111],"name":"Intl. Space Station","screen_name":"Space_Station"}]},"extended_entities":{"media":[{"additional_media_info":{"call_to_actions":{"visit_site":{"url":"http:\/\/www.nasa.gov"}},"description":"NASA astronaut Ricky Arnold speaks to students in New Mexico about #STEMonStation. For ways to bring space into your classroom, visit www.nasa.gov\/stemonstation. - Clip from NASA TV","embeddable":true,"monetizable":false,"title":"What types of research are you working on?"},"display_url":"pic.twitter.com\/WOEfNJ03ns","expanded_url":"https:\/\/twitter.com\/NASAedu\/status\/1024395212372639744\/video\/1","id":1024395107968012288,"id_str":"1024395107968012288","indices":[193,216],"media_url":"http:\/\/pbs.twimg.com\/media\/DjdiAwAV4AEPji0.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/DjdiAwAV4AEPji0.jpg","sizes":{"large":{"h":720,"resize":"fit","w":1280},"medium":{"h":675,"resize":"fit","w":1200},"small":{"h":383,"resize":"fit","w":680},"thumb":{"h":150,"resize":"crop","w":150}},"type":"video","url":"https:\/\/t.co\/WOEfNJ03ns","video_info":{"aspect_ratio":[16,9],"duration_millis":60100,"variants":[{"bitrate":2176000,"content_type":"video\/mp4","url":"https:\/\/video.twimg.com\/amplify_video\/1024395107968012288\/vid\/1280x720\/zsxyWuQPnJHTzVTu.mp4?tag=8"},{"bitrate":288000,"content_type":"video\/mp4","url":"https:\/\/video.twimg.com\/amplify_video\/1024395107968012288\/vid\/320x180\/E8wOux8YHpAPPKlk.mp4?tag=8"},{"bitrate":832000,"content_type":"video\/mp4","url":"https:\/\/video.twimg.com\/amplify_video\/1024395107968012288\/vid\/640x360\/bwcoUvgGGKVTC3oE.mp4?tag=8"},{"content_type":"application\/x-mpegURL","url":"https:\/\/video.twimg.com\/amplify_video\/1024395107968012288\/pl\/sRet_4QvKfWJVem9.m3u8?tag=8"}]}}]},"favorite_count":98,"favorited":false,"full_text":".@astro_ricky tells students at @LCmuseums about the two types of plants he's growing aboard the @Space_Station, and also about the DNA research he's working on. #STEMonStation #TeacherOnBoard https:\/\/t.co\/WOEfNJ03ns","geo":null,"id":1024395212372639744,"id_str":"1024395212372639744","in_reply_to_screen_name":null,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"is_quote_status":false,"lang":"en","place":null,"possibly_sensitive":false,"retweet_count":31,"retweeted":false,"source":"<a href=\"http:\/\/snappytv.com\" rel=\"nofollow\">SnappyTV.com<\/a>","truncated":false},"source":"<a href=\"http:\/\/twitter.com\" rel=\"nofollow\">Twitter Web Client<\/a>","truncated":false},"statuses_count":9153,"time_zone":null,"translator_type":"none","url":"https:\/\/t.co\/9Gk2H0gekn","utc_offset":null,"verified":true}

@@ tweet
{"contributors":null,"coordinates":null,"created_at":"Fri Oct 23 17:18:19 +0000 2015","display_text_range":[0,139],"entities":{"hashtags":[],"media":[{"display_url":"pic.twitter.com\/HlAnj3eTKk","expanded_url":"https:\/\/twitter.com\/Space_Station\/status\/657606994744860672\/photo\/1","id":657606857159143425,"id_str":"657606857159143425","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","media_url_https":"https:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","sizes":{"large":{"h":259,"resize":"fit","w":400},"medium":{"h":259,"resize":"fit","w":400},"small":{"h":259,"resize":"fit","w":400},"thumb":{"h":150,"resize":"crop","w":150}},"type":"photo","url":"https:\/\/t.co\/HlAnj3eTKk"}],"symbols":[],"urls":[{"display_url":"nasa.gov\/feature\/5-fun-…","expanded_url":"http:\/\/www.nasa.gov\/feature\/5-fun-things-to-do-without-gravity","indices":[92,115],"url":"https:\/\/t.co\/2cdarZPqND"}],"user_mentions":[{"id":1451773004,"id_str":"1451773004","indices":[4,18],"name":"Intl. Space Station","screen_name":"Space_Station"}]},"extended_entities":{"media":[{"display_url":"pic.twitter.com\/HlAnj3eTKk","expanded_url":"https:\/\/twitter.com\/Space_Station\/status\/657606994744860672\/photo\/1","id":657606857159143425,"id_str":"657606857159143425","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","media_url_https":"https:\/\/pbs.twimg.com\/tweet_video_thumb\/CSBJ89LU8AEPKrl.png","sizes":{"large":{"h":259,"resize":"fit","w":400},"medium":{"h":259,"resize":"fit","w":400},"small":{"h":259,"resize":"fit","w":400},"thumb":{"h":150,"resize":"crop","w":150}},"type":"animated_gif","url":"https:\/\/t.co\/HlAnj3eTKk","video_info":{"aspect_ratio":[400,259],"variants":[{"bitrate":0,"content_type":"video\/mp4","url":"https:\/\/video.twimg.com\/tweet_video\/CSBJ89LU8AEPKrl.mp4"}]}}]},"favorite_count":539,"favorited":false,"full_text":"The @Space_Station crew works  ~9.5 hours a day, with ~4 hours free time during the week... https:\/\/t.co\/2cdarZPqND https:\/\/t.co\/HlAnj3eTKk","geo":null,"id":657606994744860672,"id_str":"657606994744860672","in_reply_to_screen_name":null,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"is_quote_status":false,"lang":"en","place":null,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"retweet_count":363,"retweeted":false,"source":"<a href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\">Twitter for Android<\/a>","truncated":false,"user":{"contributors_enabled":false,"created_at":"Thu May 23 15:25:28 +0000 2013","default_profile":false,"default_profile_image":false,"description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","entities":{"description":{"urls":[]},"url":{"urls":[{"display_url":"nasa.gov\/station","expanded_url":"http:\/\/www.nasa.gov\/station","indices":[0,23],"url":"https:\/\/t.co\/9Gk2H0gekn"}]}},"favourites_count":8225,"follow_request_sent":null,"followers_count":2213476,"following":null,"friends_count":181,"geo_enabled":false,"has_extended_profile":false,"id":1451773004,"id_str":"1451773004","is_translation_enabled":false,"is_translator":false,"lang":"en","listed_count":9126,"location":"Low Earth Orbit","name":"Intl. Space Station","notifications":null,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1533048257","profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"protected":false,"screen_name":"Space_Station","statuses_count":9153,"time_zone":null,"translator_type":"none","url":"https:\/\/t.co\/9Gk2H0gekn","utc_offset":null,"verified":true}}

@@ tweet_reply
{"contributors":null,"coordinates":null,"created_at":"Thu Oct 22 22:36:55 +0000 2015","display_text_range":[0,138],"entities":{"hashtags":[],"symbols":[],"urls":[{"display_url":"flickr.com\/photos\/nasa2ex…","expanded_url":"https:\/\/www.flickr.com\/photos\/nasa2explore\/21772465134\/in\/dateposted-public\/","indices":[115,138],"url":"https:\/\/t.co\/wnKeCAdLfg"}],"user_mentions":[{"id":348968125,"id_str":"348968125","indices":[0,11],"name":"Alex von Eckartsberg","screen_name":"starlingLX"},{"id":17897744,"id_str":"17897744","indices":[12,21],"name":"Bob Jacobs","screen_name":"bnjacobs"},{"id":65647594,"id_str":"65647594","indices":[22,38],"name":"Scott Kelly","screen_name":"StationCDRKelly"},{"id":21237045,"id_str":"21237045","indices":[39,46],"name":"Flickr","screen_name":"Flickr"},{"id":65647594,"id_str":"65647594","indices":[88,104],"name":"Scott Kelly","screen_name":"StationCDRKelly"}]},"favorite_count":37,"favorited":false,"full_text":"@starlingLX @bnjacobs @StationCDRKelly @Flickr The higher-res is added but doesn't have @StationCDRKelly's edits...https:\/\/t.co\/wnKeCAdLfg","geo":null,"id":657324783294676992,"id_str":"657324783294676992","in_reply_to_screen_name":"starlingLX","in_reply_to_status_id":657257155855294465,"in_reply_to_status_id_str":"657257155855294465","in_reply_to_user_id":348968125,"in_reply_to_user_id_str":"348968125","is_quote_status":false,"lang":"en","place":null,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"retweet_count":21,"retweeted":false,"source":"<a href=\"http:\/\/twitter.com\" rel=\"nofollow\">Twitter Web Client<\/a>","truncated":false,"user":{"contributors_enabled":false,"created_at":"Thu May 23 15:25:28 +0000 2013","default_profile":false,"default_profile_image":false,"description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","entities":{"description":{"urls":[]},"url":{"urls":[{"display_url":"nasa.gov\/station","expanded_url":"http:\/\/www.nasa.gov\/station","indices":[0,23],"url":"https:\/\/t.co\/9Gk2H0gekn"}]}},"favourites_count":8225,"follow_request_sent":null,"followers_count":2213476,"following":null,"friends_count":181,"geo_enabled":false,"has_extended_profile":false,"id":1451773004,"id_str":"1451773004","is_translation_enabled":false,"is_translator":false,"lang":"en","listed_count":9126,"location":"Low Earth Orbit","name":"Intl. Space Station","notifications":null,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1533048257","profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"protected":false,"screen_name":"Space_Station","statuses_count":9153,"time_zone":null,"translator_type":"none","url":"https:\/\/t.co\/9Gk2H0gekn","utc_offset":null,"verified":true}}

@@ tweet_retweet
{"contributors":null,"coordinates":null,"created_at":"Fri Oct 23 18:40:04 +0000 2015","display_text_range":[0,139],"entities":{"hashtags":[{"indices":[31,40],"text":"Patricia"},{"indices":[94,101],"text":"Mexico"},{"indices":[103,115],"text":"YearInSpace"}],"media":[{"display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"https:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","id":657618738447958017,"id_str":"657618738447958017","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","sizes":{"large":{"h":1065,"resize":"fit","w":1600},"medium":{"h":799,"resize":"fit","w":1200},"small":{"h":453,"resize":"fit","w":680},"thumb":{"h":150,"resize":"crop","w":150}},"source_status_id":657618739492474880,"source_status_id_str":"657618739492474880","source_user_id":65647594,"source_user_id_str":"65647594","type":"photo","url":"https:\/\/t.co\/6LP2xCYcGD"}],"symbols":[],"urls":[],"user_mentions":[{"id":65647594,"id_str":"65647594","indices":[3,19],"name":"Scott Kelly","screen_name":"StationCDRKelly"},{"id":1451773004,"id_str":"1451773004","indices":[61,75],"name":"Intl. Space Station","screen_name":"Space_Station"}]},"extended_entities":{"media":[{"display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"https:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","id":657618738447958017,"id_str":"657618738447958017","indices":[116,139],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","sizes":{"large":{"h":1065,"resize":"fit","w":1600},"medium":{"h":799,"resize":"fit","w":1200},"small":{"h":453,"resize":"fit","w":680},"thumb":{"h":150,"resize":"crop","w":150}},"source_status_id":657618739492474880,"source_status_id_str":"657618739492474880","source_user_id":65647594,"source_user_id_str":"65647594","type":"photo","url":"https:\/\/t.co\/6LP2xCYcGD"}]},"favorite_count":0,"favorited":false,"full_text":"RT @StationCDRKelly: Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https:\/\/t.co\/6LP2xCYcGD","geo":null,"id":657627567948587008,"id_str":"657627567948587008","in_reply_to_screen_name":null,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"is_quote_status":false,"lang":"en","place":null,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"retweet_count":36177,"retweeted":false,"retweeted_status":{"contributors":null,"coordinates":null,"created_at":"Fri Oct 23 18:05:00 +0000 2015","display_text_range":[0,118],"entities":{"hashtags":[{"indices":[10,19],"text":"Patricia"},{"indices":[73,80],"text":"Mexico"},{"indices":[82,94],"text":"YearInSpace"}],"media":[{"display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"https:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","id":657618738447958017,"id_str":"657618738447958017","indices":[95,118],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","sizes":{"large":{"h":1065,"resize":"fit","w":1600},"medium":{"h":799,"resize":"fit","w":1200},"small":{"h":453,"resize":"fit","w":680},"thumb":{"h":150,"resize":"crop","w":150}},"type":"photo","url":"https:\/\/t.co\/6LP2xCYcGD"}],"symbols":[],"urls":[],"user_mentions":[{"id":1451773004,"id_str":"1451773004","indices":[40,54],"name":"Intl. Space Station","screen_name":"Space_Station"}]},"extended_entities":{"media":[{"display_url":"pic.twitter.com\/6LP2xCYcGD","expanded_url":"https:\/\/twitter.com\/StationCDRKelly\/status\/657618739492474880\/photo\/1","id":657618738447958017,"id_str":"657618738447958017","indices":[95,118],"media_url":"http:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/CSBUwibUwAEg0fF.jpg","sizes":{"large":{"h":1065,"resize":"fit","w":1600},"medium":{"h":799,"resize":"fit","w":1200},"small":{"h":453,"resize":"fit","w":680},"thumb":{"h":150,"resize":"crop","w":150}},"type":"photo","url":"https:\/\/t.co\/6LP2xCYcGD"}]},"favorite_count":24610,"favorited":false,"full_text":"Hurricane #Patricia looks menacing from @space_station. Stay safe below, #Mexico. #YearInSpace https:\/\/t.co\/6LP2xCYcGD","geo":null,"id":657618739492474880,"id_str":"657618739492474880","in_reply_to_screen_name":null,"in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"is_quote_status":false,"lang":"en","place":null,"possibly_sensitive":false,"possibly_sensitive_appealable":false,"retweet_count":36177,"retweeted":false,"source":"<a href=\"http:\/\/twitter.com\" rel=\"nofollow\">Twitter Web Client<\/a>","truncated":false,"user":{"contributors_enabled":false,"created_at":"Fri Aug 14 14:31:39 +0000 2009","default_profile":false,"default_profile_image":false,"description":"https:\/\/t.co\/zl81o1KrUc","entities":{"description":{"urls":[{"display_url":"scottkelly.com","expanded_url":"http:\/\/www.scottkelly.com","indices":[0,23],"url":"https:\/\/t.co\/zl81o1KrUc"}]}},"favourites_count":700,"follow_request_sent":null,"followers_count":5765333,"following":null,"friends_count":307,"geo_enabled":true,"has_extended_profile":false,"id":65647594,"id_str":"65647594","is_translation_enabled":false,"is_translator":false,"lang":"en","listed_count":8444,"location":"Houston, TX","name":"Scott Kelly","notifications":null,"profile_background_color":"010505","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/65647594\/1528319980","profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/975030326459170816\/ivHlJOYx_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/975030326459170816\/ivHlJOYx_normal.jpg","profile_link_color":"3B94D9","profile_sidebar_border_color":"838582","profile_sidebar_fill_color":"262626","profile_text_color":"727273","profile_use_background_image":true,"protected":false,"screen_name":"StationCDRKelly","statuses_count":3197,"time_zone":null,"translator_type":"none","url":null,"utc_offset":null,"verified":true}},"source":"<a href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\">Twitter for iPhone<\/a>","truncated":false,"user":{"contributors_enabled":false,"created_at":"Thu May 23 15:25:28 +0000 2013","default_profile":false,"default_profile_image":false,"description":"NASA's page for updates from the International Space Station, the world-class lab orbiting Earth 250 miles above. For the latest research, follow @ISS_Research.","entities":{"description":{"urls":[]},"url":{"urls":[{"display_url":"nasa.gov\/station","expanded_url":"http:\/\/www.nasa.gov\/station","indices":[0,23],"url":"https:\/\/t.co\/9Gk2H0gekn"}]}},"favourites_count":8225,"follow_request_sent":null,"followers_count":2213476,"following":null,"friends_count":181,"geo_enabled":false,"has_extended_profile":false,"id":1451773004,"id_str":"1451773004","is_translation_enabled":false,"is_translator":false,"lang":"en","listed_count":9126,"location":"Low Earth Orbit","name":"Intl. Space Station","notifications":null,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1451773004\/1533048257","profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/822552192875892737\/zO1pmxzw_normal.jpg","profile_link_color":"0084B4","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"protected":false,"screen_name":"Space_Station","statuses_count":9153,"time_zone":null,"translator_type":"none","url":"https:\/\/t.co\/9Gk2H0gekn","utc_offset":null,"verified":true}}

@@ search_tweets
{"statuses":[{"created_at":"Tue Jul 31 22:53:32 +0000 2018","id":1024427872700653569,"id_str":"1024427872700653569","full_text":"addiction is defiantly a mental health issue but don\u2019t speak on it unless you\u2019ve been through it or been put through it. It\u2019s not a black and white issue. it\u2019s hard to forgive and forget solely on that factor.","truncated":false,"display_text_range":[0,209],"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1242862656,"id_str":"1242862656","name":"Mad","screen_name":"Mads_Aleishea","location":"Eugene, OR","description":"live by carpe diem","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":188,"friends_count":74,"listed_count":0,"created_at":"Tue Mar 05 04:51:40 +0000 2013","favourites_count":11801,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":9386,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"BADFCD","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme12\/bg.gif","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme12\/bg.gif","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1024058681103605760\/LAWP_yxf_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1024058681103605760\/LAWP_yxf_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1242862656\/1526194715","profile_link_color":"F58EA8","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"FFF7CC","profile_text_color":"0C3E53","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:53:09 +0000 2018","id":1024427777980690432,"id_str":"1024427777980690432","full_text":"RT @judiadg: Ruh-Roh. Past coming back to haunt. #PaoloZampolli might know about this. https:\/\/t.co\/eGzVFM4ycT","truncated":false,"display_text_range":[0,110],"entities":{"hashtags":[{"text":"PaoloZampolli","indices":[49,63]}],"symbols":[],"user_mentions":[{"screen_name":"judiadg","name":"cassandra","id":281663022,"id_str":"281663022","indices":[3,11]}],"urls":[{"url":"https:\/\/t.co\/eGzVFM4ycT","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[87,110]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/#!\/download\/ipad\" rel=\"nofollow\"\u003eTwitter for iPad\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":281663022,"id_str":"281663022","name":"cassandra","screen_name":"judiadg","location":"United States","description":"#45notmypresident. Truth matters. #resistance Fight racism, bigotry, cruelty, misogyny. Art is a balm for our wounded souls. Dogs are better than people.","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":2463,"friends_count":5003,"listed_count":0,"created_at":"Wed Apr 13 18:30:20 +0000 2011","favourites_count":13373,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":16853,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/780594002177298432\/Ro8dkAi-_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/780594002177298432\/Ro8dkAi-_normal.jpg","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 18:36:56 +0000 2018","id":1024363298165149696,"id_str":"1024363298165149696","full_text":"Ruh-Roh. Past coming back to haunt. #PaoloZampolli might know about this. https:\/\/t.co\/eGzVFM4ycT","truncated":false,"display_text_range":[0,97],"entities":{"hashtags":[{"text":"PaoloZampolli","indices":[36,50]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/eGzVFM4ycT","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[74,97]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/#!\/download\/ipad\" rel=\"nofollow\"\u003eTwitter for iPad\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":281663022,"id_str":"281663022","name":"cassandra","screen_name":"judiadg","location":"United States","description":"#45notmypresident. Truth matters. #resistance Fight racism, bigotry, cruelty, misogyny. Art is a balm for our wounded souls. Dogs are better than people.","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":2463,"friends_count":5003,"listed_count":0,"created_at":"Wed Apr 13 18:30:20 +0000 2011","favourites_count":13373,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":16853,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/780594002177298432\/Ro8dkAi-_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/780594002177298432\/Ro8dkAi-_normal.jpg","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":2,"favorite_count":3,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":2,"favorite_count":0,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},{"created_at":"Tue Jul 31 22:52:20 +0000 2018","id":1024427568987033600,"id_str":"1024427568987033600","full_text":"RT @mog7546: HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantl\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[{"text":"TRUMP","indices":[17,23]},{"text":"GOP","indices":[30,34]},{"text":"Melania","indices":[97,105]}],"symbols":[],"user_mentions":[{"screen_name":"mog7546","name":"Michael O'Grady","id":270921976,"id_str":"270921976","indices":[3,11]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"https:\/\/mobile.twitter.com\" rel=\"nofollow\"\u003eTwitter Lite\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":28722755,"id_str":"28722755","name":"Lisa Wright","screen_name":"PeachyKeen49","location":"","description":"","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":536,"friends_count":202,"listed_count":9,"created_at":"Sat Apr 04 02:47:13 +0000 2009","favourites_count":59853,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":111922,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"0099B9","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme4\/bg.gif","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme4\/bg.gif","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/422849378513338368\/VqprXmZg_normal.jpeg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/422849378513338368\/VqprXmZg_normal.jpeg","profile_link_color":"0099B9","profile_sidebar_border_color":"5ED4DC","profile_sidebar_fill_color":"95E8EC","profile_text_color":"3C3940","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 19:33:13 +0000 2018","id":1024377461663039488,"id_str":"1024377461663039488","full_text":"HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantly stands by his prostitution smear\n\n#MOG\n#MAGA\n#MAGA2018 #TrumpTrain #GOPComplicit #GOPTraitors \nhttps:\/\/t.co\/LLaySvfWeA","truncated":false,"display_text_range":[0,246],"entities":{"hashtags":[{"text":"TRUMP","indices":[4,10]},{"text":"GOP","indices":[17,21]},{"text":"Melania","indices":[84,92]},{"text":"MOG","indices":[162,166]},{"text":"MAGA","indices":[167,172]},{"text":"MAGA2018","indices":[173,182]},{"text":"TrumpTrain","indices":[183,194]},{"text":"GOPComplicit","indices":[195,208]},{"text":"GOPTraitors","indices":[209,221]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/LLaySvfWeA","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/#.W2C5QblLBTN.twitter","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[223,246]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":270921976,"id_str":"270921976","name":"Michael O'Grady","screen_name":"mog7546","location":"CANADA","description":"Anti-Trump, Anti-NRA, Anti-Racist, Anti-homophobic #RESIST","url":"https:\/\/t.co\/mABaaj6bpf","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mABaaj6bpf","expanded_url":"http:\/\/page.is\/michael-7546","display_url":"page.is\/michael-7546","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":18303,"friends_count":13999,"listed_count":168,"created_at":"Wed Mar 23 14:13:55 +0000 2011","favourites_count":125368,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":432799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/270921976\/1508028446","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":22,"favorite_count":30,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":22,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:49:06 +0000 2018","id":1024426758647541762,"id_str":"1024426758647541762","full_text":"DEFIANTLY NOT A HERO!! | VENOM - Official Trailer 2 REACTION: https:\/\/t.co\/iGLkETDj5L via @YouTube","truncated":false,"display_text_range":[0,98],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"YouTube","name":"YouTube","id":10228272,"id_str":"10228272","indices":[90,98]}],"urls":[{"url":"https:\/\/t.co\/iGLkETDj5L","expanded_url":"http:\/\/youtu.be\/rubC8MPUkhY?a","display_url":"youtu.be\/rubC8MPUkhY?a","indices":[62,85]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"https:\/\/www.google.com\/\" rel=\"nofollow\"\u003eGoogle\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1007647289538744320,"id_str":"1007647289538744320","name":"Thaine \"Silver Fox\" Cooney","screen_name":"tc1whatilike","location":"Sun Prairie, WI","description":"YouTuber who makes reaction\/gaming videos. \ud83d\ude00","url":"https:\/\/t.co\/Ygr3Wl5aUb","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/Ygr3Wl5aUb","expanded_url":"https:\/\/www.youtube.com\/user\/TC1WhatILike\/","display_url":"youtube.com\/user\/TC1WhatIL\u2026","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":3,"friends_count":15,"listed_count":0,"created_at":"Fri Jun 15 15:33:29 +0000 2018","favourites_count":6,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":136,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"000000","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1007651058623995904\/XO4Gp14X_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1007651058623995904\/XO4Gp14X_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1007647289538744320\/1529077773","profile_link_color":"E81C4F","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"000000","profile_text_color":"000000","profile_use_background_image":false,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},{"created_at":"Tue Jul 31 22:48:59 +0000 2018","id":1024426729312608256,"id_str":"1024426729312608256","full_text":"RT @kenolin1: Listening to all the tributes to @mitchellreports today. What a remarkable human being. For 40 years she\u2019s courageously spoke\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"kenolin1","name":"Ken Olin","id":164679813,"id_str":"164679813","indices":[3,12]},{"screen_name":"mitchellreports","name":"Andrea Mitchell","id":89820928,"id_str":"89820928","indices":[47,63]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":3484951036,"id_str":"3484951036","name":"Meadllar","screen_name":"meadllar","location":"","description":"self employed - early baby-boomer - concerned citizen","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":676,"friends_count":1504,"listed_count":6,"created_at":"Sat Aug 29 22:52:23 +0000 2015","favourites_count":34292,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":49698,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/849052291332284421\/vCMYue88_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/849052291332284421\/vCMYue88_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/3484951036\/1491264689","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 17:29:46 +0000 2018","id":1024346394499248128,"id_str":"1024346394499248128","full_text":"Listening to all the tributes to @mitchellreports today. What a remarkable human being. For 40 years she\u2019s courageously spoken truth to power. And now, even as Trump seeks to undermine our free press, she defiantly remains a shining testament to intelligence, class and integrity.","truncated":false,"display_text_range":[0,280],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"mitchellreports","name":"Andrea Mitchell","id":89820928,"id_str":"89820928","indices":[33,49]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":164679813,"id_str":"164679813","name":"Ken Olin","screen_name":"kenolin1","location":"los angeles + new york","description":"Director\/Exec Producer @NBCThisIsUs. Opinions here are strictly my own. If yours are different that\u2019s fine, don\u2019t be mad - just don\u2019t follow me. https:\/\/t.co\/ryueFsXQTP","url":null,"entities":{"description":{"urls":[{"url":"https:\/\/t.co\/ryueFsXQTP","expanded_url":"http:\/\/vote.gov","display_url":"vote.gov","indices":[145,168]}]}},"protected":false,"followers_count":76213,"friends_count":260,"listed_count":361,"created_at":"Fri Jul 09 13:38:20 +0000 2010","favourites_count":7840,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":true,"statuses_count":2518,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1022711613650944000\/6yE-8WLU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1022711613650944000\/6yE-8WLU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/164679813\/1532822933","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":148,"favorite_count":1095,"favorited":false,"retweeted":false,"lang":"en"},"is_quote_status":false,"retweet_count":148,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:47:37 +0000 2018","id":1024426382418485248,"id_str":"1024426382418485248","full_text":"@DaiRees1984 Awww thank you so much, it means loads to me!! Going to be so emotional leaving home but hopefully itll all be worth it in the end! I\u2019ll defiantly let you know if I end up in that neck of the woods, would be lovely to see you! X","truncated":false,"display_text_range":[13,241],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"DaiRees1984","name":"Dai Rees","id":2867623215,"id_str":"2867623215","indices":[0,12]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":1024425862270275584,"in_reply_to_status_id_str":"1024425862270275584","in_reply_to_user_id":2867623215,"in_reply_to_user_id_str":"2867623215","in_reply_to_screen_name":"DaiRees1984","user":{"id":1588084015,"id_str":"1588084015","name":"Imo","screen_name":"Imogenrees1","location":"London, England","description":"","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":609,"friends_count":470,"listed_count":0,"created_at":"Fri Jul 12 09:31:51 +0000 2013","favourites_count":2690,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":6885,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1024327203046256641\/eoJCH7v2_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1024327203046256641\/eoJCH7v2_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1588084015\/1533053614","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":1,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:46:04 +0000 2018","id":1024425993166053377,"id_str":"1024425993166053377","full_text":"RT @Mangler112: That is season 6 done from #OITNB defiantly one of the better seasons! Can't believe we have to wait a full year for the ne\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[{"text":"OITNB","indices":[43,49]}],"symbols":[],"user_mentions":[{"screen_name":"Mangler112","name":"Roelof.P","id":177736055,"id_str":"177736055","indices":[3,14]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":387163361,"id_str":"387163361","name":"Diddles","screen_name":"BeauDurnberger","location":"","description":"Instagram: beaudeanxx","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":259,"friends_count":289,"listed_count":1,"created_at":"Sat Oct 08 15:41:24 +0000 2011","favourites_count":8,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":12799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/902199119766315010\/duna76qP_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/902199119766315010\/duna76qP_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/387163361\/1464015544","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 21:26:45 +0000 2018","id":1024406033945509888,"id_str":"1024406033945509888","full_text":"That is season 6 done from #OITNB defiantly one of the better seasons! Can't believe we have to wait a full year for the next one.. https:\/\/t.co\/SOpdhgwW8a","truncated":false,"display_text_range":[0,131],"entities":{"hashtags":[{"text":"OITNB","indices":[27,33]}],"symbols":[],"user_mentions":[],"urls":[],"media":[{"id":1024406025665998849,"id_str":"1024406025665998849","indices":[132,155],"media_url":"http:\/\/pbs.twimg.com\/tweet_video_thumb\/Djdr2lGXcAETOYQ.jpg","media_url_https":"https:\/\/pbs.twimg.com\/tweet_video_thumb\/Djdr2lGXcAETOYQ.jpg","url":"https:\/\/t.co\/SOpdhgwW8a","display_url":"pic.twitter.com\/SOpdhgwW8a","expanded_url":"https:\/\/twitter.com\/Mangler112\/status\/1024406033945509888\/photo\/1","type":"photo","sizes":{"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":498,"h":252,"resize":"fit"},"small":{"w":498,"h":252,"resize":"fit"},"medium":{"w":498,"h":252,"resize":"fit"}}}]},"extended_entities":{"media":[{"id":1024406025665998849,"id_str":"1024406025665998849","indices":[132,155],"media_url":"http:\/\/pbs.twimg.com\/tweet_video_thumb\/Djdr2lGXcAETOYQ.jpg","media_url_https":"https:\/\/pbs.twimg.com\/tweet_video_thumb\/Djdr2lGXcAETOYQ.jpg","url":"https:\/\/t.co\/SOpdhgwW8a","display_url":"pic.twitter.com\/SOpdhgwW8a","expanded_url":"https:\/\/twitter.com\/Mangler112\/status\/1024406033945509888\/photo\/1","type":"animated_gif","sizes":{"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":498,"h":252,"resize":"fit"},"small":{"w":498,"h":252,"resize":"fit"},"medium":{"w":498,"h":252,"resize":"fit"}},"video_info":{"aspect_ratio":[83,42],"variants":[{"bitrate":0,"content_type":"video\/mp4","url":"https:\/\/video.twimg.com\/tweet_video\/Djdr2lGXcAETOYQ.mp4"}]}}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":177736055,"id_str":"177736055","name":"Roelof.P","screen_name":"Mangler112","location":"The Netherlands","description":"\u2606 Film & Series Nerd \u2606 Streamer on @Twitch \u2606 Loving Life \u2606 Completely Crazy \u2606 Opinions Are My Own \u2606","url":"https:\/\/t.co\/t0qcn8CunI","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/t0qcn8CunI","expanded_url":"http:\/\/twitch.tv\/themangler112","display_url":"twitch.tv\/themangler112","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":1029,"friends_count":396,"listed_count":30,"created_at":"Thu Aug 12 23:09:15 +0000 2010","favourites_count":10367,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":100198,"lang":"nl","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"022330","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1022802737619234817\/DgD4Q4sx_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1022802737619234817\/DgD4Q4sx_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/177736055\/1516817545","profile_link_color":"3B94D9","profile_sidebar_border_color":"FFFFFF","profile_sidebar_fill_color":"C0DFEC","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":1,"favorite_count":3,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":1,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:44:31 +0000 2018","id":1024425602865086465,"id_str":"1024425602865086465","full_text":"RT @RawStory: Conservative candidate calls Melania Trump a \u2018hoebag\u2019  https:\/\/t.co\/vBZEjkPmUY","truncated":false,"display_text_range":[0,92],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"RawStory","name":"Raw Story","id":16041234,"id_str":"16041234","indices":[3,12]}],"urls":[{"url":"https:\/\/t.co\/vBZEjkPmUY","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[69,92]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":928183320,"id_str":"928183320","name":"#NotMyPresident","screen_name":"regsixer","location":"","description":"Awesome in Chicago!!! LGBT, #RaiseTheWage, Atheist #VoteBlue, Rabid Libtard with Pride","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":1599,"friends_count":954,"listed_count":170,"created_at":"Mon Nov 05 18:50:51 +0000 2012","favourites_count":7914,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":129211,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/799404433947983872\/qE0BtKRB_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/799404433947983872\/qE0BtKRB_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/928183320\/1399910572","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 20:40:11 +0000 2018","id":1024394314376458246,"id_str":"1024394314376458246","full_text":"Conservative candidate calls Melania Trump a \u2018hoebag\u2019  https:\/\/t.co\/vBZEjkPmUY","truncated":false,"display_text_range":[0,78],"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/vBZEjkPmUY","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[55,78]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/www.hootsuite.com\" rel=\"nofollow\"\u003eHootsuite\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":16041234,"id_str":"16041234","name":"Raw Story","screen_name":"RawStory","location":"Washington, DC","description":"The official Twitter of http:\/\/t.co\/HJOFeYodXw","url":"http:\/\/t.co\/7WT15Zb8IE","entities":{"url":{"urls":[{"url":"http:\/\/t.co\/7WT15Zb8IE","expanded_url":"http:\/\/www.rawstory.com","display_url":"rawstory.com","indices":[0,22]}]},"description":{"urls":[{"url":"http:\/\/t.co\/HJOFeYodXw","expanded_url":"http:\/\/RawStory.com","display_url":"RawStory.com","indices":[24,46]}]}},"protected":false,"followers_count":177848,"friends_count":3710,"listed_count":5186,"created_at":"Fri Aug 29 12:29:21 +0000 2008","favourites_count":8831,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":true,"statuses_count":181272,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"000000","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/879061537423544320\/99vcNM5N_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/879061537423544320\/99vcNM5N_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/16041234\/1497231241","profile_link_color":"000000","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"7D0B0B","profile_text_color":"000000","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":26,"favorite_count":46,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":26,"favorite_count":0,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},{"created_at":"Tue Jul 31 22:43:46 +0000 2018","id":1024425415132114944,"id_str":"1024425415132114944","full_text":"RT @mog7546: HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantl\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[{"text":"TRUMP","indices":[17,23]},{"text":"GOP","indices":[30,34]},{"text":"Melania","indices":[97,105]}],"symbols":[],"user_mentions":[{"screen_name":"mog7546","name":"Michael O'Grady","id":270921976,"id_str":"270921976","indices":[3,11]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1242307801,"id_str":"1242307801","name":"Book um Dano","screen_name":"Dano73Nita","location":"","description":"I book it as I see it; it\u2019s about the truth & the Rule of Law for all \ud83c\uddfa\ud83c\uddf8#TheResistance #BeTheChange #FBR #UnitedWeStand #PurgeTraitors #Vote2018 #EvictTrump","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":3759,"friends_count":3891,"listed_count":1,"created_at":"Mon Mar 04 21:54:18 +0000 2013","favourites_count":12512,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":19721,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/479053369470767104\/rLrZf9Z4_normal.jpeg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/479053369470767104\/rLrZf9Z4_normal.jpeg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/1242307801\/1521089226","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 19:33:13 +0000 2018","id":1024377461663039488,"id_str":"1024377461663039488","full_text":"HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantly stands by his prostitution smear\n\n#MOG\n#MAGA\n#MAGA2018 #TrumpTrain #GOPComplicit #GOPTraitors \nhttps:\/\/t.co\/LLaySvfWeA","truncated":false,"display_text_range":[0,246],"entities":{"hashtags":[{"text":"TRUMP","indices":[4,10]},{"text":"GOP","indices":[17,21]},{"text":"Melania","indices":[84,92]},{"text":"MOG","indices":[162,166]},{"text":"MAGA","indices":[167,172]},{"text":"MAGA2018","indices":[173,182]},{"text":"TrumpTrain","indices":[183,194]},{"text":"GOPComplicit","indices":[195,208]},{"text":"GOPTraitors","indices":[209,221]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/LLaySvfWeA","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/#.W2C5QblLBTN.twitter","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[223,246]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":270921976,"id_str":"270921976","name":"Michael O'Grady","screen_name":"mog7546","location":"CANADA","description":"Anti-Trump, Anti-NRA, Anti-Racist, Anti-homophobic #RESIST","url":"https:\/\/t.co\/mABaaj6bpf","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mABaaj6bpf","expanded_url":"http:\/\/page.is\/michael-7546","display_url":"page.is\/michael-7546","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":18303,"friends_count":13999,"listed_count":168,"created_at":"Wed Mar 23 14:13:55 +0000 2011","favourites_count":125368,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":432799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/270921976\/1508028446","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":22,"favorite_count":30,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":22,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:43:17 +0000 2018","id":1024425293895749633,"id_str":"1024425293895749633","full_text":"RT @mog7546: HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantl\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[{"text":"TRUMP","indices":[17,23]},{"text":"GOP","indices":[30,34]},{"text":"Melania","indices":[97,105]}],"symbols":[],"user_mentions":[{"screen_name":"mog7546","name":"Michael O'Grady","id":270921976,"id_str":"270921976","indices":[3,11]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":796619447209365504,"id_str":"796619447209365504","name":"Edg","screen_name":"Edgutk","location":"","description":"gay athiest, vehemently anti racist, Social Democrat, opinions my own","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":3627,"friends_count":5001,"listed_count":46,"created_at":"Thu Nov 10 07:44:11 +0000 2016","favourites_count":57713,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":65291,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"F5F8FA","profile_background_image_url":null,"profile_background_image_url_https":null,"profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1010151103738961920\/zvGeWaeT_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1010151103738961920\/zvGeWaeT_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/796619447209365504\/1496789168","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 19:33:13 +0000 2018","id":1024377461663039488,"id_str":"1024377461663039488","full_text":"HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantly stands by his prostitution smear\n\n#MOG\n#MAGA\n#MAGA2018 #TrumpTrain #GOPComplicit #GOPTraitors \nhttps:\/\/t.co\/LLaySvfWeA","truncated":false,"display_text_range":[0,246],"entities":{"hashtags":[{"text":"TRUMP","indices":[4,10]},{"text":"GOP","indices":[17,21]},{"text":"Melania","indices":[84,92]},{"text":"MOG","indices":[162,166]},{"text":"MAGA","indices":[167,172]},{"text":"MAGA2018","indices":[173,182]},{"text":"TrumpTrain","indices":[183,194]},{"text":"GOPComplicit","indices":[195,208]},{"text":"GOPTraitors","indices":[209,221]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/LLaySvfWeA","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/#.W2C5QblLBTN.twitter","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[223,246]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":270921976,"id_str":"270921976","name":"Michael O'Grady","screen_name":"mog7546","location":"CANADA","description":"Anti-Trump, Anti-NRA, Anti-Racist, Anti-homophobic #RESIST","url":"https:\/\/t.co\/mABaaj6bpf","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mABaaj6bpf","expanded_url":"http:\/\/page.is\/michael-7546","display_url":"page.is\/michael-7546","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":18303,"friends_count":13999,"listed_count":168,"created_at":"Wed Mar 23 14:13:55 +0000 2011","favourites_count":125368,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":432799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/270921976\/1508028446","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":22,"favorite_count":30,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":22,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:42:24 +0000 2018","id":1024425070880481280,"id_str":"1024425070880481280","full_text":"RT @mog7546: HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantl\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[{"text":"TRUMP","indices":[17,23]},{"text":"GOP","indices":[30,34]},{"text":"Melania","indices":[97,105]}],"symbols":[],"user_mentions":[{"screen_name":"mog7546","name":"Michael O'Grady","id":270921976,"id_str":"270921976","indices":[3,11]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":458318307,"id_str":"458318307","name":"khearts\ud83c\uddfa\ud83c\uddf8\ud83c\udf0a\ud83c\udde8\ud83c\udde6","screen_name":"krazyheartss","location":"Everywhere ","description":"Immortality is VirtueLies are the Gods honest truth The Moment we stop fighting for each other that\u2019s the moment we lose our Humanity #resistance #NeverAgain \ud83c\udf0a","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":1880,"friends_count":2523,"listed_count":0,"created_at":"Sun Jan 08 12:30:55 +0000 2012","favourites_count":30060,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":14974,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"642D8B","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme10\/bg.gif","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme10\/bg.gif","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/980282973609066496\/VKB5-1Lo_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/980282973609066496\/VKB5-1Lo_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/458318307\/1530755098","profile_link_color":"FF0000","profile_sidebar_border_color":"65B0DA","profile_sidebar_fill_color":"7AC3EE","profile_text_color":"3D1957","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 19:33:13 +0000 2018","id":1024377461663039488,"id_str":"1024377461663039488","full_text":"HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantly stands by his prostitution smear\n\n#MOG\n#MAGA\n#MAGA2018 #TrumpTrain #GOPComplicit #GOPTraitors \nhttps:\/\/t.co\/LLaySvfWeA","truncated":false,"display_text_range":[0,246],"entities":{"hashtags":[{"text":"TRUMP","indices":[4,10]},{"text":"GOP","indices":[17,21]},{"text":"Melania","indices":[84,92]},{"text":"MOG","indices":[162,166]},{"text":"MAGA","indices":[167,172]},{"text":"MAGA2018","indices":[173,182]},{"text":"TrumpTrain","indices":[183,194]},{"text":"GOPComplicit","indices":[195,208]},{"text":"GOPTraitors","indices":[209,221]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/LLaySvfWeA","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/#.W2C5QblLBTN.twitter","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[223,246]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":270921976,"id_str":"270921976","name":"Michael O'Grady","screen_name":"mog7546","location":"CANADA","description":"Anti-Trump, Anti-NRA, Anti-Racist, Anti-homophobic #RESIST","url":"https:\/\/t.co\/mABaaj6bpf","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mABaaj6bpf","expanded_url":"http:\/\/page.is\/michael-7546","display_url":"page.is\/michael-7546","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":18303,"friends_count":13999,"listed_count":168,"created_at":"Wed Mar 23 14:13:55 +0000 2011","favourites_count":125368,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":432799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/270921976\/1508028446","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":22,"favorite_count":30,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":22,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:40:37 +0000 2018","id":1024424623755206657,"id_str":"1024424623755206657","full_text":"RT @mog7546: HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantl\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[{"text":"TRUMP","indices":[17,23]},{"text":"GOP","indices":[30,34]},{"text":"Melania","indices":[97,105]}],"symbols":[],"user_mentions":[{"screen_name":"mog7546","name":"Michael O'Grady","id":270921976,"id_str":"270921976","indices":[3,11]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/#!\/download\/ipad\" rel=\"nofollow\"\u003eTwitter for iPad\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":270921976,"id_str":"270921976","name":"Michael O'Grady","screen_name":"mog7546","location":"CANADA","description":"Anti-Trump, Anti-NRA, Anti-Racist, Anti-homophobic #RESIST","url":"https:\/\/t.co\/mABaaj6bpf","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mABaaj6bpf","expanded_url":"http:\/\/page.is\/michael-7546","display_url":"page.is\/michael-7546","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":18303,"friends_count":13999,"listed_count":168,"created_at":"Wed Mar 23 14:13:55 +0000 2011","favourites_count":125368,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":432799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/270921976\/1508028446","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 19:33:13 +0000 2018","id":1024377461663039488,"id_str":"1024377461663039488","full_text":"HEY #TRUMP, SOME #GOP THINK YOUR WIFE IS A PROSTITUTE\n\nConservative candidate calls #Melania Trump a 'HOEBAG' -- then defiantly stands by his prostitution smear\n\n#MOG\n#MAGA\n#MAGA2018 #TrumpTrain #GOPComplicit #GOPTraitors \nhttps:\/\/t.co\/LLaySvfWeA","truncated":false,"display_text_range":[0,246],"entities":{"hashtags":[{"text":"TRUMP","indices":[4,10]},{"text":"GOP","indices":[17,21]},{"text":"Melania","indices":[84,92]},{"text":"MOG","indices":[162,166]},{"text":"MAGA","indices":[167,172]},{"text":"MAGA2018","indices":[173,182]},{"text":"TrumpTrain","indices":[183,194]},{"text":"GOPComplicit","indices":[195,208]},{"text":"GOPTraitors","indices":[209,221]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/LLaySvfWeA","expanded_url":"https:\/\/www.rawstory.com\/2018\/07\/conservative-candidate-calls-melania-trump-hoebag-defiantly-stands-prostitution-smear\/#.W2C5QblLBTN.twitter","display_url":"rawstory.com\/2018\/07\/conser\u2026","indices":[223,246]}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":270921976,"id_str":"270921976","name":"Michael O'Grady","screen_name":"mog7546","location":"CANADA","description":"Anti-Trump, Anti-NRA, Anti-Racist, Anti-homophobic #RESIST","url":"https:\/\/t.co\/mABaaj6bpf","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/mABaaj6bpf","expanded_url":"http:\/\/page.is\/michael-7546","display_url":"page.is\/michael-7546","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":18303,"friends_count":13999,"listed_count":168,"created_at":"Wed Mar 23 14:13:55 +0000 2011","favourites_count":125368,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":432799,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/921934514028396544\/4S2aKoUU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/270921976\/1508028446","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":22,"favorite_count":30,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":22,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:40:11 +0000 2018","id":1024424512387985413,"id_str":"1024424512387985413","full_text":"have y'all ever wondered how its like biasing someone else? like people who like soonyoung is defiantly different from a hansol stan and idk","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[],"symbols":[],"user_mentions":[],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/android\" rel=\"nofollow\"\u003eTwitter for Android\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":893746359982899200,"id_str":"893746359982899200","name":"[17's syah\ud83c\udf3b]","screen_name":"vern0nized","location":"","description":"straight for sol, gay for kyu","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":94,"friends_count":168,"listed_count":0,"created_at":"Sat Aug 05 08:11:52 +0000 2017","favourites_count":3369,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":7615,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"000000","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1022839005686235136\/IKHn0HaU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1022839005686235136\/IKHn0HaU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/893746359982899200\/1533064335","profile_link_color":"E81C4F","profile_sidebar_border_color":"000000","profile_sidebar_fill_color":"000000","profile_text_color":"000000","profile_use_background_image":false,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"regular"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":0,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:39:54 +0000 2018","id":1024424441810485250,"id_str":"1024424441810485250","full_text":"RT @ZaibatsuNews: Conservative candidate calls Melania Trump a \u2018hoebag\u2019 \u2014 then defiantly stands by his prostitution smear https:\/\/t.co\/xkzN\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"ZaibatsuNews","name":"#VoteBlue \ud83c\udf0a","id":93111391,"id_str":"93111391","indices":[3,16]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\" rel=\"nofollow\"\u003eTwitter Web Client\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":17217502,"id_str":"17217502","name":"gritslady","screen_name":"gritslady","location":"Native Texan","description":"Looking to the future, one Sci Fi show\/movie at a time. #TheResistance #NeverAgain","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":4523,"friends_count":4209,"listed_count":71,"created_at":"Thu Nov 06 19:37:35 +0000 2008","favourites_count":1588,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":false,"statuses_count":84324,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"FFFFFF","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme13\/bg.gif","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme13\/bg.gif","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/961008189549375488\/t871dK7Q_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/961008189549375488\/t871dK7Q_normal.jpg","profile_link_color":"1B95E0","profile_sidebar_border_color":"333333","profile_sidebar_fill_color":"FF5288","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 18:09:09 +0000 2018","id":1024356306298126337,"id_str":"1024356306298126337","full_text":"Conservative candidate calls Melania Trump a \u2018hoebag\u2019 \u2014 then defiantly stands by his prostitution smear https:\/\/t.co\/xkzNmAusr0 #p2 #ctl https:\/\/t.co\/KJxkpo0pEX","truncated":false,"display_text_range":[0,136],"entities":{"hashtags":[{"text":"p2","indices":[128,131]},{"text":"ctl","indices":[132,136]}],"symbols":[],"user_mentions":[],"urls":[{"url":"https:\/\/t.co\/xkzNmAusr0","expanded_url":"http:\/\/j.mp\/2LNaYJ3","display_url":"j.mp\/2LNaYJ3","indices":[104,127]}],"media":[{"id":1024356304100372480,"id_str":"1024356304100372480","indices":[137,160],"media_url":"http:\/\/pbs.twimg.com\/media\/Djc-oZ1VsAAnZ68.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/Djc-oZ1VsAAnZ68.jpg","url":"https:\/\/t.co\/KJxkpo0pEX","display_url":"pic.twitter.com\/KJxkpo0pEX","expanded_url":"https:\/\/twitter.com\/ZaibatsuNews\/status\/1024356306298126337\/photo\/1","type":"photo","sizes":{"small":{"w":680,"h":366,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":800,"h":430,"resize":"fit"},"medium":{"w":800,"h":430,"resize":"fit"}}}]},"extended_entities":{"media":[{"id":1024356304100372480,"id_str":"1024356304100372480","indices":[137,160],"media_url":"http:\/\/pbs.twimg.com\/media\/Djc-oZ1VsAAnZ68.jpg","media_url_https":"https:\/\/pbs.twimg.com\/media\/Djc-oZ1VsAAnZ68.jpg","url":"https:\/\/t.co\/KJxkpo0pEX","display_url":"pic.twitter.com\/KJxkpo0pEX","expanded_url":"https:\/\/twitter.com\/ZaibatsuNews\/status\/1024356306298126337\/photo\/1","type":"photo","sizes":{"small":{"w":680,"h":366,"resize":"fit"},"thumb":{"w":150,"h":150,"resize":"crop"},"large":{"w":800,"h":430,"resize":"fit"},"medium":{"w":800,"h":430,"resize":"fit"}}}]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"https:\/\/dlvrit.com\/\" rel=\"nofollow\"\u003edlvr.it\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":93111391,"id_str":"93111391","name":"#VoteBlue \ud83c\udf0a","screen_name":"ZaibatsuNews","location":"Minnesota, USA","description":"Breaking progressive news. Retweets \u2260 endorsements #Basta #Resist #TheResistance #TrumpRussia #BlueWave2018 #UniteBlue #p2 #FlipItBlue #fbr powered by @Zaibatsu","url":"https:\/\/t.co\/XtVd4u92iY","entities":{"url":{"urls":[{"url":"https:\/\/t.co\/XtVd4u92iY","expanded_url":"http:\/\/DifferenceTheory.net","display_url":"DifferenceTheory.net","indices":[0,23]}]},"description":{"urls":[]}},"protected":false,"followers_count":102554,"friends_count":81192,"listed_count":1363,"created_at":"Sat Nov 28 03:36:22 +0000 2009","favourites_count":4064,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":126975,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":true,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/2707738656\/dd233770e3991717b63e7bf9a844765d_normal.jpeg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/2707738656\/dd233770e3991717b63e7bf9a844765d_normal.jpeg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/93111391\/1400832961","profile_link_color":"0084B4","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":true,"default_profile":false,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":8,"favorite_count":16,"favorited":false,"retweeted":false,"possibly_sensitive":false,"lang":"en"},"is_quote_status":false,"retweet_count":8,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"},{"created_at":"Tue Jul 31 22:38:54 +0000 2018","id":1024424192064847880,"id_str":"1024424192064847880","full_text":"RT @kenolin1: Listening to all the tributes to @mitchellreports today. What a remarkable human being. For 40 years she\u2019s courageously spoke\u2026","truncated":false,"display_text_range":[0,140],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"kenolin1","name":"Ken Olin","id":164679813,"id_str":"164679813","indices":[3,12]},{"screen_name":"mitchellreports","name":"Andrea Mitchell","id":89820928,"id_str":"89820928","indices":[47,63]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":1681678512,"id_str":"1681678512","name":"Laura","screen_name":"wkulaura86","location":"","description":"","url":null,"entities":{"description":{"urls":[]}},"protected":false,"followers_count":36,"friends_count":444,"listed_count":0,"created_at":"Sun Aug 18 21:52:05 +0000 2013","favourites_count":856,"utc_offset":null,"time_zone":null,"geo_enabled":false,"verified":false,"statuses_count":293,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/378800000320244329\/dd8dae1d81445cf00754f485b88f97de_normal.jpeg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/378800000320244329\/dd8dae1d81445cf00754f485b88f97de_normal.jpeg","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"retweeted_status":{"created_at":"Tue Jul 31 17:29:46 +0000 2018","id":1024346394499248128,"id_str":"1024346394499248128","full_text":"Listening to all the tributes to @mitchellreports today. What a remarkable human being. For 40 years she\u2019s courageously spoken truth to power. And now, even as Trump seeks to undermine our free press, she defiantly remains a shining testament to intelligence, class and integrity.","truncated":false,"display_text_range":[0,280],"entities":{"hashtags":[],"symbols":[],"user_mentions":[{"screen_name":"mitchellreports","name":"Andrea Mitchell","id":89820928,"id_str":"89820928","indices":[33,49]}],"urls":[]},"metadata":{"iso_language_code":"en","result_type":"recent"},"source":"\u003ca href=\"http:\/\/twitter.com\/download\/iphone\" rel=\"nofollow\"\u003eTwitter for iPhone\u003c\/a\u003e","in_reply_to_status_id":null,"in_reply_to_status_id_str":null,"in_reply_to_user_id":null,"in_reply_to_user_id_str":null,"in_reply_to_screen_name":null,"user":{"id":164679813,"id_str":"164679813","name":"Ken Olin","screen_name":"kenolin1","location":"los angeles + new york","description":"Director\/Exec Producer @NBCThisIsUs. Opinions here are strictly my own. If yours are different that\u2019s fine, don\u2019t be mad - just don\u2019t follow me. https:\/\/t.co\/ryueFsXQTP","url":null,"entities":{"description":{"urls":[{"url":"https:\/\/t.co\/ryueFsXQTP","expanded_url":"http:\/\/vote.gov","display_url":"vote.gov","indices":[145,168]}]}},"protected":false,"followers_count":76213,"friends_count":260,"listed_count":361,"created_at":"Fri Jul 09 13:38:20 +0000 2010","favourites_count":7840,"utc_offset":null,"time_zone":null,"geo_enabled":true,"verified":true,"statuses_count":2518,"lang":"en","contributors_enabled":false,"is_translator":false,"is_translation_enabled":false,"profile_background_color":"C0DEED","profile_background_image_url":"http:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_image_url_https":"https:\/\/abs.twimg.com\/images\/themes\/theme1\/bg.png","profile_background_tile":false,"profile_image_url":"http:\/\/pbs.twimg.com\/profile_images\/1022711613650944000\/6yE-8WLU_normal.jpg","profile_image_url_https":"https:\/\/pbs.twimg.com\/profile_images\/1022711613650944000\/6yE-8WLU_normal.jpg","profile_banner_url":"https:\/\/pbs.twimg.com\/profile_banners\/164679813\/1532822933","profile_link_color":"1DA1F2","profile_sidebar_border_color":"C0DEED","profile_sidebar_fill_color":"DDEEF6","profile_text_color":"333333","profile_use_background_image":true,"has_extended_profile":false,"default_profile":true,"default_profile_image":false,"following":null,"follow_request_sent":null,"notifications":null,"translator_type":"none"},"geo":null,"coordinates":null,"place":null,"contributors":null,"is_quote_status":false,"retweet_count":148,"favorite_count":1095,"favorited":false,"retweeted":false,"lang":"en"},"is_quote_status":false,"retweet_count":148,"favorite_count":0,"favorited":false,"retweeted":false,"lang":"en"}],"search_metadata":{"completed_in":0.054,"max_id":1024427872700653569,"max_id_str":"1024427872700653569","next_results":"?max_id=1024424192064847879&q=defiantly&include_entities=1","query":"defiantly","refresh_url":"?since_id=1024427872700653569&q=defiantly&include_entities=1","count":15,"since_id":0,"since_id_str":"0"}}
