package Acme::CPANModules::UUID;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2020-12-22'; # DATE
our $DIST = 'Acme-CPANModules-UUID'; # DIST
our $VERSION = '0.001'; # VERSION

use strict;
use warnings;

our $LIST = {
    summary => 'Modules that can generate immutable universally unique identifier (UUIDs)',
    description => <<'_',

UUIDs are 128-bit numbers that can be used as permanent IDs or keys in
databases. There are 5 "versions" of UUID, each might be more suitable than
others in specific cases. Version 1 (v1) UUIDs are generated from a time and a
node ID (usually the MAC address); version 2 (v2) UUIDs from an identifier
(group/user ID), a time, and a node ID; version 4 (v4) UUIDs from a
random/pseudo-random number; version 3 (v3) UUIDs from hashing a namespace using
MD5; version 5 (v5) from hashing a namespace using SHA-1.

<pm:Data::UUID> should be your first choice,

_
    entry_features => {
        v4_secure_random => {summary => 'Whether the module uses cryptographically secure pseudo-random number generator for v4 UUIDs'},
    },
    entries => [
        {
            module => 'Data::UUID',
            description => <<'_',

This module creates v1 and v2 UUIDs. Depending on the OS, for MAC address, it
usually uses a hash of hostname instead. This module is XS, so performance is
good. If you cannot use an XS module, try <pm:UUID::Tiny> instead.

The benchmark code creates 1000 v1 string UUIDs.

_
            bench_code_template => 'my $u = Data::UUID->new; $u->create for 1..1000',
            features => {
                is_xs => 1,
                is_pp => 0,
                create_v1 => 1,
                create_v2 => 1,
                create_v3 => 0,
                create_v4 => 0,
                create_v5 => 0,
            },
        },

        {
            module => 'UUID::Tiny',
            description => <<'_',

This module should be your go-to choice if you cannot use an XS module.

To create a cryptographically secure random (v4) UUIDs, use
<pm:UUID::Tiny::Patch::UseMRS>.

The benchmark code creates 1000 v1 string UUIDs.

See also: <pm:Types::UUID> which is a type library that uses Data::UUID as the
backend.

_
            bench_code_template => 'UUID::Tiny::create_uuid() for 1..1000',
            features => {
                is_xs => 0,
                is_pp => 1,
                create_v1 => 1,
                create_v2 => 0,
                create_v3 => 1,
                create_v4 => 1,
                v4_secure_random => 0,
                create_v5 => 1,
            },
        },

        {
            module => 'UUID::Random',
            description => <<'_',

This module simply uses 32 calls to Perl's C<rand()> to construct each random
hexadecimal digits of the UUID. Not really recommended, but it's dead simple.

To create a cryptographically secure random UUIDs, use
<pm:UUID::Random::Patch::UseMRS>.

The benchmark code creates 1000 v1 string UUIDs.

_
            bench_code_template => 'UUID::Random::generate() for 1..1000',
            features => {
                is_xs => 0,
                is_pp => 1,
                create_v1 => 0,
                create_v2 => 0,
                create_v3 => 0,
                create_v4 => 1,
                v4_secure_random => 0,
                create_v5 => 0,
            },
        },
    ],
};

1;
# ABSTRACT: Modules that can generate immutable universally unique identifier (UUIDs)

__END__

=pod

=encoding UTF-8

=head1 NAME

Acme::CPANModules::UUID - Modules that can generate immutable universally unique identifier (UUIDs)

=head1 VERSION

This document describes version 0.001 of Acme::CPANModules::UUID (from Perl distribution Acme-CPANModules-UUID), released on 2020-12-22.

=head1 DESCRIPTION

UUIDs are 128-bit numbers that can be used as permanent IDs or keys in
databases. There are 5 "versions" of UUID, each might be more suitable than
others in specific cases. Version 1 (v1) UUIDs are generated from a time and a
node ID (usually the MAC address); version 2 (v2) UUIDs from an identifier
(group/user ID), a time, and a node ID; version 4 (v4) UUIDs from a
random/pseudo-random number; version 3 (v3) UUIDs from hashing a namespace using
MD5; version 5 (v5) from hashing a namespace using SHA-1.

L<Data::UUID> should be your first choice,

=head1 ACME::CPANMODULES FEATURE COMPARISON MATRIX

 +--------------+-----------+-----------+-----------+-----------+-----------+-------+-------+----------------------+
 | module       | create_v1 | create_v2 | create_v3 | create_v4 | create_v5 | is_pp | is_xs | v4_secure_random *1) |
 +--------------+-----------+-----------+-----------+-----------+-----------+-------+-------+----------------------+
 | Data::UUID   | yes       | yes       | no        | no        | no        | no    | yes   | N/A                  |
 | UUID::Tiny   | yes       | no        | yes       | yes       | yes       | yes   | no    | no                   |
 | UUID::Random | no        | no        | no        | yes       | no        | yes   | no    | no                   |
 +--------------+-----------+-----------+-----------+-----------+-----------+-------+-------+----------------------+


Notes:

=over

=item 1. v4_secure_random: Whether the module uses cryptographically secure pseudo-random number generator for v4 UUIDs

=back

=head1 ACME::MODULES ENTRIES

=over

=item * L<Data::UUID>

This module creates v1 and v2 UUIDs. Depending on the OS, for MAC address, it
usually uses a hash of hostname instead. This module is XS, so performance is
good. If you cannot use an XS module, try L<UUID::Tiny> instead.

The benchmark code creates 1000 v1 string UUIDs.


=item * L<UUID::Tiny>

This module should be your go-to choice if you cannot use an XS module.

To create a cryptographically secure random (v4) UUIDs, use
L<UUID::Tiny::Patch::UseMRS>.

The benchmark code creates 1000 v1 string UUIDs.

See also: L<Types::UUID> which is a type library that uses Data::UUID as the
backend.


=item * L<UUID::Random>

This module simply uses 32 calls to Perl's C<rand()> to construct each random
hexadecimal digits of the UUID. Not really recommended, but it's dead simple.

To create a cryptographically secure random UUIDs, use
L<UUID::Random::Patch::UseMRS>.

The benchmark code creates 1000 v1 string UUIDs.


=back

=head1 FAQ

=head2 What are ways to use this module?

Aside from reading it, you can install all the listed modules using
L<cpanmodules>:

    % cpanmodules ls-entries UUID | cpanm -n

or L<Acme::CM::Get>:

    % perl -MAcme::CM::Get=UUID -E'say $_->{module} for @{ $LIST->{entries} }' | cpanm -n

This module contains benchmark instructions. You can run a benchmark
for some/all the modules listed in this Acme::CPANModules module using
L<bencher>:

    % bencher --cpanmodules-module UUID

This module also helps L<lcpan> produce a more meaningful result for C<lcpan
related-mods> when it comes to finding related modules for the modules listed
in this Acme::CPANModules module.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Acme-CPANModules-UUID>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Acme-CPANModules-UUID>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://github.com/perlancar/perl-Acme-CPANModules-UUID/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Acme::CPANModules> - about the Acme::CPANModules namespace

L<cpanmodules> - CLI tool to let you browse/view the lists

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
