#!/usr/bin/env perl


# PODNAME: get_hcloud_info.pl
# ABSTRACT: get API definitions from docs.hetzner.cloud

use v5.24;

use Mojo::Base -strict, -signatures;

use Mojo::File qw(path curfile);
use Mojo::JSON qw(decode_json encode_json);
use Mojo::UserAgent;
use Mojo::Util qw(dumper decamelize);
use Mojo::URL;
use Data::Printer;
use Data::Dumper::Perltidy;
use JSON::XS;

$Data::Dumper::Sortkeys = 1;
$Data::Dumper::Trailingcomma = 1;
#$Data::Dumper::Purity = 1;
$Data::Dumper::Deepcopy = 1;

my $ua   = Mojo::UserAgent->new;
my $url  = 'https://dns.hetzner.com';
my $base = 'https://dns.hetzner.com/api/v1';

my $tx        = $ua->get( $url . '/api-docs' );
my $dom       = $tx->res->dom;
my ($scripts) = $dom->find('script')->grep( sub {
    my $src = $_->attr('src');
    return if !$src;
    return 1 if $src =~ m{main\.\w+\.chunk.js};
    return;
});

my $js_tx  = $ua->get($url . $scripts->first->attr('src'));
my ($json) = $js_tx->res->body =~ m{JSON\.parse\('(.*?\}{4,})'\)};
$json =~ s{\\'}{'}g;
$json =~ s{\\\\}{\\}g;

my $data  = decode_json $json;

# create schema class
my $path = curfile->dirname->child(
    qw/.. lib DNS Hetzner/,
    "Schema.pm"
);

my $schema_package = _get_schema_package( $path, $data );
my $fh = $path->open('>');
$fh->print($schema_package);
$fh->close;

my @paths = keys $data->{paths}->%*;

my $components = $data->{components};
delete $components->{securitySchemes};
delete $components->{schemas}->{Pagination};
delete $components->{schemas}->{Meta};

my %classes;
APIPATH:
for my $api_path ( @paths ) {
    my $parts = path( $api_path )->to_array;
    shift $parts->@*;

    my $name  = shift $parts->@*;
    my $class = ucfirst $name;

    my $subtree = $data->{paths}->{$api_path};
    $subtree->{path} = $api_path;
    $subtree->{parts} = $parts;

    $classes{$class} //= {
        endpoint => $name,
    };

    push $classes{$class}->{defs}->@*, $subtree;
}

for my $class ( keys %classes ) {
    say "Build class $class.pm";
    my $def = $classes{$class};

    my $code = _get_code( $class, $def, $components );

    my $path = curfile->dirname->child(
        qw/.. lib DNS Hetzner API/,
        "$class.pm"
    );

    my $fh = $path->open('>');
    $fh->print( $code );
    $fh->close;
}

sub _get_code ($class, $definitions, $components) {
    my $endpoint    = $definitions->{endpoint};
    my $subs        = '';
    my $methods_pod = '';

    for my $subdef ( $definitions->{defs}->@* ) {
        my $uri  = join '/', '', $subdef->{parts}->@*;
        $uri =~ s/\{(.*?)\}/:$1/g;

        METHOD:
        for my $method ( sort keys $subdef->%* ) {
            my ($method_def) = $subdef->{$method};

            next METHOD if 'HASH' ne ref $method_def;

            my $operation_id = delete $method_def->{operationId};
            delete @{ $method_def }{qw/responses x-code-samples summary tags/};
        
            my $description = delete $method_def->{description};
            my ($sub, $method_name) = _get_sub( $operation_id, $method, $endpoint, $uri );
            my $pod                 = _get_pod( $method_name, $description, $class, $endpoint );

            $subs        .= $sub;
            $methods_pod .= $pod;
        }
    }

    my $pod = sprintf q~
=head1 SYNOPSIS

    use DNS::Hetzner;

    my $api_key = '1234abc';
    my $dns     = DNS::Hetzner->new(
        token => $api_key,
    );

    $dns->records->create(
    );

=head1 ATTRIBUTES

=over 4

=item * endpoint

=back

=head1 METHODS

%s
~, $methods_pod;

    my $code = sprintf q~package DNS::Hetzner::API::%s;

# ABSTRACT: %s

# ---
# This class is auto-generated by bin/get_hetzner_info.pl
# ---

use v5.24;

use Moo;
use Types::Standard qw(:all);

use Mojo::Base -strict, -signatures;

extends 'DNS::Hetzner::APIBase';

with 'MooX::Singleton';

use DNS::Hetzner::Schema;
use Carp;

has endpoint  => ( is => 'ro', isa => Str, default => sub { '%s' } );
%s

1;

__END__

=pod

%s
    ~,
    $class, $class, $endpoint, $subs, $pod;

    return $code;
}

sub _get_sub ( $operation_id, $method, $class, $uri ) {
    my $method_name = decamelize $operation_id;

    $method_name = 'list' if $method_name eq 'get_' . $class;
    
    $class =~ s{s\z}{};
    $method_name =~ s{_${class}s?}{};

    my $sub = sprintf q~
sub %s ($self, %params) {
    $self->_do( '%s', \%params, '%s', { type => '%s' } );
}
~, $method_name, $operation_id, $uri, $method;

    return ($sub, $method_name);
}

sub _get_pod ( $method, $description, $object, $endpoint, $params = '') {
    my $pod = sprintf q~

=head2 %s

%s

    $dns->%s->%s(%s);
~,
        $method, $description, $endpoint, $method, $params;

    return $pod;
}

sub _get_schema_package ( $path, $data ) {
    my ($code) = split /__DATA__/, $path->slurp;
    $code .= sprintf q!
__DATA__
@@ openapi.json
%s
!, JSON::XS->new->utf8(1)->pretty(1)->encode( $data );

    return $code;
}

__END__

=pod

=encoding utf-8

=head1 NAME

get_hcloud_info.pl - get API definitions from docs.hetzner.cloud

=head1 VERSION

version 0.03

=head1 AUTHOR

Renee Baecker <reneeb@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2020 by Renee Baecker.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut
