#! perl

package HarfBuzz::Shaper;

use 5.010001;
use strict;
use warnings;
use Carp;
use Encode;

our $VERSION = '0.01';

require XSLoader;
XSLoader::load('HarfBuzz::Shaper', $VERSION);

sub new {
    my ( $pkg, $opts ) = @_;

    $opts //= {};

    my $self = bless {} => $pkg;
    $self->{buf} = hb_buffer_create();

    if ( $opts->{font} ) {
	$self->set_font( delete $opts->{font} );
    }
    if ( $opts->{size} ) {
	$self->set_size( delete $opts->{size} );
    }

    return $self;
}

sub set_font {
    my ( $self, $fontfile, $size ) = @_;

    my $blob = hb_blob_create_from_file($fontfile);
    my $face = $self->{"face_$fontfile"} //= hb_face_create( $blob, 0 );
    $self->{font} = $self->{"font_$fontfile"} //= do {
	# hb_font_create should default to OT.
	my $font = hb_font_create( $face );
	hb_ot_font_set_funcs($font);
	$font;
    };
    $self->set_size($size) if $size;

    $self;
}

sub set_size {
    my ( $self, $size ) = @_;

    $self->{size} = $size;

    $self;
}

sub set_text {
    my ( $self, @text ) = @_;

    $self->{text} = join( "", @text );

    $self;
}

sub shaper {
    my ( $self ) = @_;

    croak("HarfBuzz shape() without font")     unless $self->{font};
    croak("HarfBuzz shape() without fontsize") unless $self->{size};
    croak("HarfBuzz shape() without text")     unless defined $self->{text};

    hb_buffer_clear_contents( $self->{buf} );
    hb_buffer_add_utf8( $self->{buf}, $self->{text} );

    hb_buffer_guess_segment_properties( $self->{buf} );

    # Set the font point size for correct hinting.
    hb_font_set_ptem( $self->{font}, $self->{size} );
    # Set a scaling for precision (all info are ints!).
    my $scale = 1000;
    hb_font_set_scale( $self->{font}, $scale, $scale );

    my $info = _hb_shaper( $self->{font}, $self->{buf} );

    foreach my $i ( @$info ) {
	$i->{$_} *= $self->{size} / $scale for qw( ax ay dx dy );
    }

    return $info;
}

1;

__END__

=head1 NAME

HarfBuzz::Shaper - Use HarfBuzz for text shaping

=head1 SYNOPSIS

  use HarfBuzz::Shaper;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for HarfBuzz::Shaper, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head1 METHODS

=head2 new


=head1 SEE ALSO

=head1 AUTHOR

Johan Vromans, E<lt>jv@cpan.org<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2020 by Johan Vromans

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.28.2 or,
at your option, any later version of Perl 5 you may have available.


=cut
