#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2016 -- leonerd@leonerd.org.uk

package Net::Prometheus::ProcessCollector;

use strict;
use warnings;

our $VERSION = '0.04';

=head1 NAME

C<Net::Prometheus::ProcessCollector> - obtain a process collector for the OS

=head1 SYNOPSIS

   use Net::Prometheus::ProcessCollector;

   my $collector = Net::Prometheus::ProcessCollector->new;

=head1 DESCRIPTION

This module-loading package provides a method that attempts to load a process
collector appropriate for the host OS it is running on.

The following OS-specific modules are provided with this distribution:

=over 2

=item *

L<Net::Prometheus::ProcessCollector::linux>

=back

Other OSes may be supported by 3rd-party CPAN modules by following this naming
pattern based on the value of the C<$^O> variable on the OS concerned.

=cut

=head1 MAGIC CONSTRUCTORS

=cut

=head2 new

   $collector = Net::Prometheus::ProcessCollector->new( @args )

Attempts to construct a new process collector for the OS named by C<$^O>,
passing in any extra arguments into the C<new> constructor for the specific
class.

If no perl module is found under the appropriate file name, C<undef> is
returned. If any other error occurs while loading or constructing the
instance, the exception is thrown as normal.

=cut

sub new
{
   my $class = shift;
   $class->for_OS( $^O, @_ );
}

=head2 for_OS

   $collector = Net::Prometheus::ProcessCollector->for_OS( $os, @args )

Attempts to construct a new process collector for the named OS. Except under
especially-exceptional circumstances, you don't want to call this method.
Call L</new> instead.

=cut

sub for_OS
{
   shift; # class
   my ( $os, @args ) = @_;

   my $pkg = __PACKAGE__ . "::$os";

   ( my $file = "$pkg.pm" ) =~ s{::}{/}g;
   if( !eval { require $file } ) {
      return if $@ =~ m/^Can't locate \Q$file\E in \@INC/;
      die $@;
   }

   return $pkg->new( @args );
}

0x55AA;
