## no critic: ValuesAndExpressions::ProhibitCommaSeparatedStatements BuiltinFunctions::RequireBlockMap

package Test::Sah::Schema;

use 5.010001;
use strict 'subs', 'vars';
use warnings;
use Log::ger;
use Log::ger::App;

use File::Spec;
use Test::Builder;
use Test::More ();

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2022-07-17'; # DATE
our $DIST = 'Test-Sah-Schema'; # DIST
our $VERSION = '0.011'; # VERSION

my $Test = Test::Builder->new;

sub import {
    my $self = shift;
    my $caller = caller;
    no strict 'refs';
    *{$caller.'::sah_schema_modules_ok'}      = \&sah_schema_modules_ok;

    $Test->exported_to($caller);
    $Test->plan(@_);
}

sub _set_option_defaults {
    my $opts = shift;
    $opts->{test_schema_examples} //= 1;
}

sub sah_schema_module_ok {
    require Data::Sah::Normalize;

    my $module = shift;
    my %opts   = (@_ && (ref $_[0] eq "HASH")) ? %{(shift)} : ();
    my $msg    = @_ ? shift : "Sah schema in $module";
    my $res;
    my $ok = 1;

    _set_option_defaults(\%opts);

    my $modulep = $module; $modulep =~ s!::!/!g; $modulep .= ".pm";
    require $modulep;
    my $sch = ${"$module\::schema"};

    (my $sch_name = $module) =~ s/\ASah::Schema:://;

    $Test->subtest(
        $msg,
        sub {
          TEST_NORMALIZED: {
                require Data::Dump;
                require Text::Diff;
                my $nsch = Data::Sah::Normalize::normalize_schema($sch);
                my $nsch_with_extras = [@$nsch, {}]; # extras part still accepted because most of my schema modules have it

                my $sch_dmp  = Data::Dump::dump($sch);
                my $nsch_dmp = Data::Dump::dump($nsch);
                my $nsch_with_extras_dmp = Data::Dump::dump($nsch_with_extras);

                if ($sch_dmp eq $nsch_dmp) {
                    $Test->ok(1, "Schema is normalized");
                } elsif ($sch_dmp eq $nsch_with_extras_dmp) {
                    $Test->carp("\e[31mSchema still contains extras part, please remove it to avoid this warning\e[0m");
                    $Test->ok(1, "Schema is normalized but still contains extras part");
                } else {
                    my $diff = Text::Diff::diff(\$sch_dmp, \$nsch_dmp);
                    $Test->diag("Schema difference with normalized version: $diff");
                    $Test->ok(0, "Schema is not normalized");
                    return 0;
                }
            }

          TEST_EXAMPLES: {
                last unless $opts{test_schema_examples};
                last unless $sch->[1]{examples};
                require Data::Sah;

                my $vdr = Data::Sah::gen_validator($sch, {return_type=>'str_errmsg+val'});

                my $wsub;
                if (eval { require Perinci::Sub::Wrapper; 1 }) {
                    # we also want to test whether code generated by coercion
                    # rules, prefilters, etc used by the schema behave well when
                    # used with Perinci::Sub::Wrapper (PSW), e.g. they do not
                    # use 'return' to exit early from the wrapper subroutine.
                    # PSW is used when we use Perinci::Access::Perl, which in
                    # turn is used in applications like Perinci::CmdLine.
                    my $wres = Perinci::Sub::Wrapper::wrap_sub(
                        sub => sub { my %args = @_; [200, "OK", $args{'arg'}] },
                        meta => { v=>1.1, args=>{arg=>{schema=>[$sch_name,req=>1]}} },
                        validate_args => 1,
                    );
                    die "Can't wrap subroutine with Perinci::Sub::Wrapper: $wres->[0] - $wres->[1]"
                        unless $wres->[0] == 200;
                    $wsub = $wres->[2]{sub};
                } else {
                    $Test->note("Skipping test with Perinci::Sub::Wrapper: $@");
                }

                my $i = 0;
                for my $eg (@{ $sch->[1]{examples} }) {
                    $i++;
                    # non-defhash example is allowed, this means the example
                    # just specifies valid value. we normalize to hash form.
                    unless (ref $eg eq 'HASH') {
                        $eg = {value=>$eg, valid=>1};
                    }

                    next unless $eg->{test} // 1;
                    $Test->subtest(
                        "example #$i" .
                            ($eg->{name} ? " ($eg->{name})" :
                             ($eg->{summary} ? " ($eg->{summary})" : "")),
                        sub {
                            my $value =
                                exists $eg->{value} ? $eg->{value} :
                                exists $eg->{data}  ? $eg->{data} : die "BUG in example #$i: Please specify 'value' or 'data'";

                            my ($vdr_errmsg, $vdr_res)  = @{ $vdr->($value) };
                            my $wsub_res;
                            $wsub_res = $wsub->(arg => $value) if $wsub;

                            if ($eg->{valid}) {
                                if ($vdr_errmsg) {
                                    $Test->ok(0, "Value should be valid, but isn't ($vdr_errmsg)");
                                    $ok = 0;
                                    return;
                                } else {
                                    $Test->ok(1, "Value should be valid");
                                }

                                my $validated_value =
                                    exists $eg->{validated_value} ? $eg->{validated_value} :
                                    exists $eg->{res} ? $eg->{res} :
                                    exists $eg->{value} ? $eg->{value} : $eg->{data};
                                Test::More::is_deeply($vdr_res, $validated_value, 'Validated value matches') or do {
                                    $Test->diag($Test->explain($vdr_res));
                                    $ok = 0;
                                };

                                if ($wsub) {
                                    Test::More::is_deeply($wsub_res, [200, "OK", $validated_value], "Wrapped sub should return successful result") or do {
                                        $Test->diag($Test->explain($wsub_res));
                                        $ok = 0;
                                    };
                                }

                            } else {
                                if (!$vdr_errmsg) {
                                    $Test->ok(0, "Value shouldn't be valid, but is");
                                    $ok = 0;
                                    return;
                                } else {
                                    $Test->ok(1, "Value should not be valid");

                                    if ($wsub) {
                                        $Test->ok($wsub_res->[0]==400, "Wrapped sub should return enveloped result with status=400");
                                    }
                                }
                            }
                        }
                    );
                } # for example
            } # TEST_EXAMPLES
            $ok;
        } # subtest
    ) or $ok = 0;

    $ok;
}

# BEGIN copy-pasted from Test::Pod::Coverage, with a bit modification

sub all_modules {
    my @starters = @_ ? @_ : _starting_points();
    my %starters = map {$_,1} @starters;

    my @queue = @starters;

    my @modules;
    while ( @queue ) {
        my $file = shift @queue;
        if ( -d $file ) {
            local *DH;
            opendir DH, $file or next;
            my @newfiles = readdir DH;
            closedir DH;

            @newfiles = File::Spec->no_upwards( @newfiles );
            @newfiles = grep { $_ ne "CVS" && $_ ne ".svn" } @newfiles;

            push @queue, map "$file/$_", @newfiles;
        }
        if ( -f $file ) {
            next unless $file =~ /\.pm$/;

            my @parts = File::Spec->splitdir( $file );
            shift @parts if @parts && exists $starters{$parts[0]};
            shift @parts if @parts && $parts[0] eq "lib";
            $parts[-1] =~ s/\.pm$// if @parts;

            # Untaint the parts
            for ( @parts ) {
                if ( /^([a-zA-Z0-9_\.\-]*)$/ && ($_ eq $1) ) {
                    $_ = $1;  # Untaint the original
                }
                else {
                    die qq{Invalid and untaintable filename "$file"!};
                }
            }
            my $module = join( "::", grep {length} @parts );
            push( @modules, $module );
        }
    } # while

    return @modules;
}

sub _starting_points {
    return 'blib' if -e 'blib';
    return 'lib';
}

# END copy-pasted from Test::Pod::Coverage

sub sah_schema_modules_ok {
    my $opts = (@_ && (ref $_[0] eq "HASH")) ? shift : {};
    my $msg  = shift;
    my $ok = 1;

    _set_option_defaults($opts);

    my @starters = _starting_points();
    local @INC = (@starters, @INC);

    $Test->plan(tests => 1);

    my @include_modules;
    {
        my $val = delete $opts->{include_modules};
        last unless $val;
        for my $mod (@$val) {
            $mod = "Sah::Schema::$mod" unless $mod =~ /^Sah::Schema::/;
            push @include_modules, $mod;
        }
    }
    my @exclude_modules;
    {
        my $val = delete $opts->{exclude_modules};
        last unless $val;
        for my $mod (@$val) {
            $mod = "Sah::Schema::$mod" unless $mod =~ /^Sah::Schema::/;
            push @exclude_modules, $mod;
        }
    }

    my @all_modules = all_modules(@starters);
    if (@all_modules) {
        $Test->subtest(
            "Sah schema modules in dist",
            sub {
                for my $module (@all_modules) {
                    next unless $module =~ /\ASah::Schema::/;
                    if (@include_modules) {
                        next unless grep { $module eq $_ } @include_modules;
                    }
                    if (@exclude_modules) {
                        next if grep { $module eq $_ } @exclude_modules;
                    }

                    log_info "Processing module %s ...", $module;
                    my $thismsg = defined $msg ? $msg :
                        "Sah schema module in $module";
                    my $thisok = sah_schema_module_ok(
                        $module, $opts, $thismsg)
                        or $ok = 0;
                }
            }
        ) or $ok = 0;
    } else {
        $Test->ok(1, "No modules found.");
    }
    $ok;
}

1;
# ABSTRACT: Test Sah::Schema::* modules in distribution

__END__

=pod

=encoding UTF-8

=head1 NAME

Test::Sah::Schema - Test Sah::Schema::* modules in distribution

=head1 VERSION

This document describes version 0.011 of Test::Sah::Schema (from Perl distribution Test-Sah-Schema), released on 2022-07-17.

=head1 SYNOPSIS

To check a single Sah::Schema::* module:

 use Test::Sah::Schema tests=>1;
 sah_schema_module_ok("Sah::Schema::Foo", {opt => ...}, $msg);

To check all Sah::Schema::* modules in a distro:

 # save in release-sah-schema.t, put in distro's t/ subdirectory
 use Test::More;
 plan skip_all => "Not release testing" unless $ENV{RELEASE_TESTING};
 eval "use Test::Sah::Schema";
 plan skip_all => "Test::Sah::Schema required for testing Sah::Schema modules" if $@;
 sah_schema_modules_ok({opt => ...}, $msg);

=head1 DESCRIPTION

This module performs various checks on Sah::Schema::* modules. It is recommended
that you include something like C<release-sah-schema.t> in your distribution if
you add metadata to your code. If you use L<Dist::Zilla> to build your
distribution, there is L<Dist::Zilla::Plugin::Sah::Schemas> to make it easy to
do so.

=for Pod::Coverage ^(all_modules)$

=head1 ACKNOWLEDGEMENTS

Some code taken from L<Test::Pod::Coverage> by Andy Lester.

=head1 FUNCTIONS

All these functions are exported by default.

=head2 sah_schema_module_ok($module [, \%opts ] [, $msg])

Load C<$module>, get its C<$schema>, and perform test on it.

Available options:

=over

=item * test_schema_examples => BOOL (default: 1)

Whether to test examples in schema.

=back

=head2 sah_schema_modules_ok([ \%opts ] [, $msg])

Look for modules in directory C<lib> (or C<blib> instead, if it exists), and
C<run sah_schema_module_ok()> on each of them.

Options are the same as in C<sah_schema_module_ok()>, plus:

=over

=item * include_modules

=item * exclude_modules

=back

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Test-Sah-Schema>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Test-Sah-Schema>.

=head1 SEE ALSO

L<test-sah-schema>, a command-line interface for C<sah_schema_modules_ok()>.

L<Test::Sah> to use Sah schema to test data.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla plugin and/or Pod::Weaver::Plugin. Any additional steps required
beyond that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022, 2020 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Test-Sah-Schema>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
