package Linux::Info::FileStats;
use strict;
use warnings;
use Carp qw(croak);

our $VERSION = '2.15'; # VERSION

# ABSTRACT: Collect linux file statistics.


sub new {
    my $class = shift;
    my $opts  = ref( $_[0] ) ? shift : {@_};

    my %self = (
        files => {
            path     => '/proc',
            file_nr  => 'sys/fs/file-nr',
            inode_nr => 'sys/fs/inode-nr',
            dentries => 'sys/fs/dentry-state',
        }
    );

    foreach my $file ( keys %{ $opts->{files} } ) {
        $self{files}{$file} = $opts->{files}->{$file};
    }

    return bless \%self, $class;
}

sub get {
    my $self  = shift;
    my $class = ref $self;
    my $file  = $self->{files};
    my $stats = {};

    $self->{stats} = $stats;
    $self->_get_file_nr;
    $self->_get_inode_nr;
    $self->_get_dentries;

    return $stats;
}

sub _get_file_nr {
    my $self  = shift;
    my $class = ref $self;
    my $file  = $self->{files};
    my $stats = $self->{stats};

    my $filename =
      $file->{path} ? "$file->{path}/$file->{file_nr}" : $file->{file_nr};
    open my $fh, '<', $filename
      or croak "$class: unable to open $filename ($!)";
    @$stats{qw(fhalloc fhfree fhmax)} = ( split /\s+/, <$fh> )[ 0 .. 2 ];
    close($fh);
}

sub _get_inode_nr {
    my $self  = shift;
    my $class = ref $self;
    my $file  = $self->{files};
    my $stats = $self->{stats};

    my $filename =
      $file->{path} ? "$file->{path}/$file->{inode_nr}" : $file->{inode_nr};
    open my $fh, '<', $filename
      or croak "$class: unable to open $filename ($!)";
    @$stats{qw(inalloc infree)} = ( split /\s+/, <$fh> )[ 0 .. 1 ];
    $stats->{inmax} = $stats->{inalloc} + $stats->{infree};
    close($fh);
}

sub _get_dentries {
    my $self  = shift;
    my $class = ref $self;
    my $file  = $self->{files};
    my $stats = $self->{stats};

    my $filename =
      $file->{path} ? "$file->{path}/$file->{dentries}" : $file->{dentries};
    open my $fh, '<', $filename
      or croak "$class: unable to open $filename ($!)";
    @$stats{qw(dentries unused agelimit wantpages)} =
      ( split /\s+/, <$fh> )[ 0 .. 3 ];
    close($fh);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Linux::Info::FileStats - Collect linux file statistics.

=head1 VERSION

version 2.15

=head1 SYNOPSIS

    use Linux::Info::FileStats;

    my $lxs  = Linux::Info::FileStats->new;
    my $stat = $lxs->get;

=head1 DESCRIPTION

Linux::Info::FileStats gathers file statistics from the virtual F</proc> filesystem (procfs).

For more information read the documentation of the front-end module L<Linux::Info>.

=head1 FILE STATISTICS

Generated by F</proc/sys/fs/file-nr>, F</proc/sys/fs/inode-nr> and F</proc/sys/fs/dentry-state>.

    fhalloc    -  Number of allocated file handles.
    fhfree     -  Number of free file handles.
    fhmax      -  Number of maximum file handles.
    inalloc    -  Number of allocated inodes.
    infree     -  Number of free inodes.
    inmax      -  Number of maximum inodes.
    dentries   -  Dirty directory cache entries.
    unused     -  Free diretory cache size.
    agelimit   -  Time in seconds the dirty cache entries can be reclaimed.
    wantpages  -  Pages that are requested by the system when memory is short.

If youre looking for information about inodes usage in a mouint point, most probably you want to take a look at L<Linux::Info::DiskUsage> module.
This module gets information about inodes from /proc as L<Linux::Info::DiskUsage> does, but documentation about this specific part of /proc is obscure,
so you're on your own to find out which module to use.

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

    my $lxs = Linux::Info::FileStats->new;

It's possible to set the path to the proc filesystem.

     Linux::Info::FileStats->new(
        files => {
            # This is the default
            path     => '/proc',
            file_nr  => 'sys/fs/file-nr',
            inode_nr => 'sys/fs/inode-nr',
            dentries => 'sys/fs/dentry-state',
        }
    );

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

    my $stat = $lxs->get;

=head1 EXPORTS

Nothing.

=head1 SEE ALSO

=over

=item *

B<proc(5)>

=item *

L<Linux::Info>

=item *

L<Linux::Info::DiskUsage>

=back

=head1 AUTHOR

Alceu Rodrigues de Freitas Junior <glasswalk3r@yahoo.com.br>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by Alceu Rodrigues de Freitas Junior.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
