package Chart::Plotly::Trace::Volume::Slices::X;
use Moose;
use MooseX::ExtraArgs;
use Moose::Util::TypeConstraints qw(enum union);
if ( !defined Moose::Util::TypeConstraints::find_type_constraint('PDL') ) {
    Moose::Util::TypeConstraints::type('PDL');
}

our $VERSION = '0.039';    # VERSION

# ABSTRACT: This attribute is one of the possible options for the trace volume.

sub TO_JSON {
    my $self       = shift;
    my $extra_args = $self->extra_args // {};
    my $meta       = $self->meta;
    my %hash       = %$self;
    for my $name ( sort keys %hash ) {
        my $attr = $meta->get_attribute($name);
        if ( defined $attr ) {
            my $value = $hash{$name};
            my $type  = $attr->type_constraint;
            if ( $type && $type->equals('Bool') ) {
                $hash{$name} = $value ? \1 : \0;
            }
        }
    }
    %hash = ( %hash, %$extra_args );
    delete $hash{'extra_args'};
    if ( $self->can('type') && ( !defined $hash{'type'} ) ) {
        $hash{type} = $self->type();
    }
    return \%hash;
}

has fill => (
    is  => "rw",
    isa => "Num",
    documentation =>
      "Sets the fill ratio of the `slices`. The default fill value of the `slices` is 1 meaning that they are entirely shaded. On the other hand Applying a `fill` ratio less than one would allow the creation of openings parallel to the edges.",
);

has locations => (
    is  => "rw",
    isa => "ArrayRef|PDL",
    documentation =>
      "Specifies the location(s) of slices on the axis. When not specified slices would be created for all points of the axis x except start and end.",
);

has locationssrc => ( is            => "rw",
                      isa           => "Str",
                      documentation => "Sets the source reference on plot.ly for  locations .",
);

has show => ( is            => "rw",
              isa           => "Bool",
              documentation => "Determines whether or not slice planes about the x dimension are drawn.",
);

__PACKAGE__->meta->make_immutable();
1;

__END__

=pod

=encoding utf-8

=head1 NAME

Chart::Plotly::Trace::Volume::Slices::X - This attribute is one of the possible options for the trace volume.

=head1 VERSION

version 0.039

=head1 SYNOPSIS

 use Chart::Plotly;
 use Chart::Plotly::Trace::Volume;
 use Chart::Plotly::Plot;
 
 # Example from https://github.com/plotly/plotly.js/blob/e23b2c39173f215438198bdcd5ad4d9395626237/test/image/mocks/gl3d_volume_airflow.json
 
 my $volume = Chart::Plotly::Trace::Volume->new(
         "flatshading"=>JSON::true,
         "lighting"=>{
             "facenormalsepsilon"=>0
         },
         "colorscale"=>[[ 0.0, "rgb(0,0,255)" ], [ 0.5, "rgb(255,255,255)" ], [ 0.75, "rgb(255,255,0)" ], [ 0.875, "rgb(255,0,0)" ], [ 1.0, "rgb(127,127,127)" ]],
         "reversescale"=>JSON::true,
         "opacityscale"=>[
             [0, 1.0],
             [0.2, 0.9],
             [0.3, 0.5],
             [0.4, 0.4],
             [0.5, 0.5],
             [1, 1.0]
         ],
         "opacity"=>0.05,
         "surface"=>{ "show"=>JSON::true, "fill"=>1.0, "pattern"=>"all", "count"=>25 },
         "spaceframe"=>{ "show"=>JSON::true, "fill"=>1.0 },
         "slices"=>{
             "x"=>{ "show"=>JSON::false },
             "y"=>{ "show"=>JSON::false },
             "z"=>{ "show"=>JSON::false }
         },
         "caps"=>{
             "x"=>{ "show"=>JSON::true, "fill"=>1.0 },
             "y"=>{ "show"=>JSON::true, "fill"=>1.0 },
             "z"=>{ "show"=>JSON::true, "fill"=>1.0 }
         },
         "contour"=>{
             "show"=>JSON::true,
             "width"=>4
         },
         "value"=>[
 
             752.121, 706.793, 669.056, 635.331, 601.977, 565.938, 525.114, 478.455, 425.890, 368.155, 306.583, 242.868, 178.835, 116.194, 56.315, 0.000,
             721.584, 678.474, 643.231, 612.121, 581.358, 547.784, 509.245, 464.680, 414.033, 358.075, 298.177, 236.080, 173.653, 112.653, 54.491, 0.000,
             693.354, 652.250, 619.207, 590.397, 561.935, 530.584, 494.140, 451.523, 402.685, 348.416, 290.124, 229.584, 168.706, 109.285, 52.765, 0.000,
             668.740, 629.294, 597.960, 570.924, 544.289, 514.776, 480.131, 439.241, 392.047, 339.346, 282.562, 223.498, 164.091, 106.165, 51.180, 0.000,
             649.336, 611.040, 580.683, 554.640, 529.132, 500.891, 467.618, 428.143, 382.365, 331.062, 275.658, 217.963, 159.927, 103.384, 49.793, 0.000,
             636.803, 598.991, 568.653, 542.571, 517.254, 489.530, 457.058, 418.581, 373.918, 323.795, 269.603, 213.141, 156.349, 101.046, 48.664, 0.000,
             632.456, 594.367, 562.970, 535.661, 509.429, 481.304, 448.928, 410.930, 367.008, 317.790, 264.604, 209.206, 153.499, 99.260, 47.856, 0.000,
             636.803, 597.691, 564.222, 534.535, 506.254, 476.737, 443.666, 405.550, 361.927, 313.290, 260.864, 206.327, 151.514, 98.124, 47.422, 0.000,
             649.336, 608.584, 572.276, 539.301, 507.992, 476.154, 441.593, 402.727, 358.924, 310.506, 258.558, 204.646, 150.502, 97.709, 47.394, 0.000,
             668.740, 625.915, 586.325, 549.513, 514.499, 479.599, 442.844, 402.628, 358.161, 309.580, 257.805, 204.254, 150.522, 98.042, 47.773, 0.000,
             693.354, 648.198, 605.169, 564.321, 525.259, 486.821, 447.338, 405.257, 359.679, 310.563, 258.645, 205.174, 151.574, 99.101, 48.533, 0.000,
             721.584, 673.971, 627.537, 582.702, 539.536, 497.343, 454.793, 410.461, 363.394, 313.401, 261.034, 207.353, 153.595, 100.820, 49.621, 0.000,
             752.121, 702.012, 652.304, 603.673, 556.528, 510.566, 464.793, 417.959, 369.113, 317.949, 264.848, 210.678, 156.476, 103.103, 50.977, 0.000,
             783.987, 731.395, 678.583, 626.399, 575.494, 525.880, 476.866, 427.396, 376.567, 323.996, 269.910, 214.993, 160.080, 105.840, 52.538, 0.000,
             816.497, 761.460, 705.715, 650.226, 595.813, 542.730, 490.543, 438.396, 385.457, 331.298, 276.017, 220.123, 164.264, 108.928, 54.248, 0.000,
             849.182, 791.753, 733.237, 674.672, 616.998, 560.649, 505.409, 450.603, 395.486, 339.608, 282.962, 225.899, 168.893, 112.273, 56.061, 0.000,
 
             721.584, 678.474, 643.231, 612.121, 581.358, 547.784, 509.245, 464.680, 414.033, 358.075, 298.177, 236.080, 173.653, 112.653, 54.491, 0.000,
             686.589, 646.086, 613.857, 585.923, 558.277, 527.621, 491.736, 449.557, 401.061, 347.064, 288.994, 228.650, 167.960, 108.743, 52.463, 0.000,
             653.357, 615.301, 585.871, 560.880, 536.137, 508.216, 474.840, 434.935, 388.501, 336.396, 280.097, 221.457, 162.457, 104.971, 50.512, 0.000,
             623.482, 587.545, 560.440, 537.892, 515.607, 490.066, 458.929, 421.098, 376.579, 326.255, 271.641, 214.631, 157.253, 101.422, 48.690, 0.000,
             599.175, 564.785, 539.166, 518.180, 497.585, 473.822, 444.483, 408.410, 365.579, 316.873, 263.820, 208.339, 152.487, 98.206, 47.063, 0.000,
             583.039, 549.351, 523.968, 503.221, 483.160, 460.281, 432.088, 397.313, 355.849, 308.530, 256.867, 202.779, 148.329, 95.456, 45.714, 0.000,
             577.350, 543.327, 516.660, 494.500, 473.488, 450.313, 422.398, 388.308, 347.781, 301.546, 251.052, 198.181, 144.969, 93.322, 44.732, 0.000,
             583.039, 547.663, 518.278, 493.067, 469.521, 444.712, 416.049, 381.898, 341.779, 296.252, 246.650, 194.776, 142.600, 91.949, 44.199, 0.000,
             599.175, 561.668, 528.583, 499.109, 471.696, 443.993, 413.530, 378.508, 338.202, 292.948, 243.913, 192.772, 141.383, 91.443, 44.164, 0.000,
             623.482, 583.383, 546.180, 511.866, 479.771, 448.227, 415.052, 378.389, 337.288, 291.845, 243.014, 192.303, 141.407, 91.849, 44.631, 0.000,
             653.357, 610.463, 569.150, 529.958, 492.914, 457.015, 420.486, 381.548, 339.103, 293.016, 244.017, 193.403, 142.672, 93.131, 45.555, 0.000,
             686.589, 640.860, 595.657, 551.864, 509.997, 469.619, 429.402, 387.752, 343.519, 296.384, 246.851, 195.994, 145.083, 95.188, 46.859, 0.000,
             721.584, 673.060, 624.252, 576.252, 529.892, 485.174, 441.187, 396.586, 350.250, 301.733, 251.337, 199.908, 148.477, 97.877, 48.453, 0.000,
             757.290, 706.047, 653.914, 602.099, 551.638, 502.850, 455.182, 407.549, 358.915, 308.762, 257.222, 204.924, 152.663, 101.049, 50.254, 0.000,
             793.052, 739.179, 683.966, 628.686, 574.499, 521.952, 470.778, 420.138, 369.109, 317.141, 264.229, 210.806, 157.449, 104.567, 52.190, 0.000,
             828.478, 772.063, 713.978, 655.527, 597.946, 541.939, 487.468, 433.905, 380.450, 326.550, 272.092, 217.336, 162.668, 108.319, 54.210, 0.000,
 
             693.354, 652.250, 619.207, 590.397, 561.935, 530.584, 494.140, 451.523, 402.685, 348.416, 290.124, 229.584, 168.706, 109.285, 52.765, 0.000,
             653.357, 615.301, 585.871, 560.880, 536.137, 508.216, 474.840, 434.935, 388.501, 336.396, 280.097, 221.457, 162.457, 104.971, 50.512, 0.000,
             614.104, 579.049, 553.178, 531.946, 510.854, 486.293, 455.920, 418.669, 374.590, 324.605, 270.262, 213.486, 156.331, 100.742, 48.305, 0.000,
             577.350, 545.055, 522.403, 504.569, 486.806, 465.345, 437.773, 403.026, 361.187, 313.235, 260.779, 205.808, 150.441, 96.688, 46.197, 0.000,
             546.024, 515.904, 495.601, 480.269, 465.076, 446.138, 420.954, 388.419, 348.615, 302.547, 251.867, 198.609, 144.946, 92.936, 44.268, 0.000,
             524.284, 495.263, 475.674, 461.169, 447.154, 429.714, 406.205, 375.394, 337.294, 292.880, 243.809, 192.135, 140.057, 89.655, 42.626, 0.000,
             516.398, 486.970, 465.806, 449.710, 434.813, 417.330, 394.426, 364.620, 327.738, 284.648, 236.951, 186.682, 136.029, 87.054, 41.402, 0.000,
             524.284, 492.954, 468.009, 447.801, 429.663, 410.245, 386.571, 356.822, 320.515, 278.310, 231.679, 182.580, 133.141, 85.349, 40.725, 0.000,
             546.024, 511.799, 481.804, 455.799, 432.494, 409.329, 383.421, 352.648, 316.157, 274.306, 228.360, 180.135, 131.638, 84.715, 40.680, 0.000,
             577.350, 539.821, 504.564, 472.285, 442.862, 414.704, 385.327, 352.500, 315.038, 272.959, 227.263, 179.559, 131.669, 85.225, 41.275, 0.000,
             614.104, 573.225, 533.079, 494.893, 459.330, 425.686, 392.073, 356.392, 317.259, 274.389, 228.486, 180.907, 133.229, 86.819, 42.431, 0.000,
             653.357, 609.235, 564.731, 521.315, 480.111, 441.084, 402.963, 363.948, 322.620, 278.469, 231.921, 184.054, 136.167, 89.332, 44.023, 0.000,
             693.354, 646.143, 597.767, 549.800, 503.608, 459.610, 417.064, 374.532, 330.681, 284.870, 237.291, 188.742, 140.233, 92.548, 45.920, 0.000,
             733.133, 682.986, 631.134, 579.183, 528.616, 480.149, 433.445, 387.417, 340.884, 293.152, 244.224, 194.646, 145.151, 96.257, 48.009, 0.000,
             772.195, 719.254, 664.238, 608.747, 554.320, 501.855, 451.321, 401.929, 352.673, 302.855, 252.338, 201.446, 150.665, 100.285, 50.206, 0.000,
             810.303, 754.695, 696.758, 638.074, 580.195, 524.138, 470.091, 417.514, 365.562, 313.568, 261.288, 208.865, 156.567, 104.500, 52.455, 0.000,
 
             668.740, 629.294, 597.960, 570.924, 544.289, 514.776, 480.131, 439.241, 392.047, 339.346, 282.562, 223.498, 164.091, 106.165, 51.180, 0.000,
             623.482, 587.545, 560.440, 537.892, 515.607, 490.066, 458.929, 421.098, 376.579, 326.255, 271.641, 214.631, 157.253, 101.422, 48.690, 0.000,
             577.350, 545.055, 522.403, 504.569, 486.806, 465.345, 437.773, 403.026, 361.187, 313.235, 260.779, 205.808, 150.441, 96.688, 46.197, 0.000,
             531.830, 503.158, 484.964, 471.833, 458.551, 441.110, 417.039, 385.313, 346.100, 300.472, 250.132, 197.159, 143.764, 92.048, 43.753, 0.000,
             490.276, 464.804, 450.439, 441.369, 432.028, 418.194, 397.324, 368.404, 331.661, 288.243, 239.931, 188.885, 137.393, 87.639, 41.445, 0.000,
             459.150, 435.587, 423.060, 416.102, 409.180, 397.896, 379.522, 352.944, 318.362, 276.941, 230.506, 181.269, 131.577, 83.668, 39.407, 0.000,
             447.214, 423.180, 408.755, 400.183, 392.767, 382.037, 364.869, 339.813, 306.864, 267.094, 222.300, 174.699, 126.656, 80.420, 37.831, 0.000,
             459.150, 432.177, 411.999, 397.463, 385.715, 372.697, 354.833, 330.073, 297.972, 259.344, 215.850, 169.640, 123.038, 78.232, 36.934, 0.000,
             490.276, 459.156, 431.676, 408.724, 389.608, 371.473, 350.738, 324.766, 292.510, 254.359, 211.716, 166.570, 121.122, 77.405, 36.874, 0.000,
             531.830, 496.468, 462.234, 430.975, 403.544, 378.599, 353.220, 324.576, 291.093, 252.666, 210.336, 165.841, 121.161, 78.070, 37.663, 0.000,
             577.350, 538.042, 498.174, 459.890, 424.812, 392.797, 361.887, 329.529, 293.898, 254.463, 211.874, 167.544, 123.150, 80.121, 39.159, 0.000,
             623.482, 580.557, 536.015, 492.021, 450.495, 412.027, 375.533, 338.982, 300.583, 259.540, 216.149, 171.471, 126.827, 83.269, 41.145, 0.000,
             668.740, 622.480, 573.953, 525.260, 478.393, 434.348, 392.680, 351.905, 310.434, 267.363, 222.711, 177.200, 131.790, 87.176, 43.426, 0.000,
             712.543, 663.178, 611.149, 558.470, 507.122, 458.306, 412.017, 367.229, 322.614, 277.262, 230.998, 184.245, 137.634, 91.548, 45.858, 0.000,
             754.719, 702.437, 647.251, 591.089, 535.886, 482.951, 432.564, 384.058, 336.356, 288.599, 240.475, 192.168, 144.021, 96.170, 48.347, 0.000,
             795.271, 740.230, 682.142, 622.859, 564.259, 507.701, 453.657, 401.729, 351.057, 300.849, 250.708, 200.622, 150.705, 100.899, 50.840, 0.000,
 
             649.336, 611.040, 580.683, 554.640, 529.132, 500.891, 467.618, 428.143, 382.365, 331.062, 275.658, 217.963, 159.927, 103.384, 49.793, 0.000,
             599.175, 564.785, 539.166, 518.180, 497.585, 473.822, 444.483, 408.410, 365.579, 316.873, 263.820, 208.339, 152.487, 98.206, 47.063, 0.000,
             546.024, 515.904, 495.601, 480.269, 465.076, 446.138, 420.954, 388.419, 348.615, 302.547, 251.867, 198.609, 144.946, 92.936, 44.268, 0.000,
             490.276, 464.804, 450.439, 441.369, 432.028, 418.194, 397.324, 368.404, 331.661, 288.243, 239.931, 188.885, 137.393, 87.639, 41.445, 0.000,
             434.237, 413.569, 405.431, 402.848, 399.458, 390.737, 374.140, 348.782, 315.044, 274.223, 228.233, 179.353, 129.988, 82.442, 38.669, 0.000,
             386.097, 369.198, 365.704, 368.127, 369.580, 365.213, 352.385, 330.253, 299.294, 260.912, 217.128, 170.322, 123.001, 77.571, 36.094, 0.000,
             365.148, 347.905, 342.546, 344.188, 346.544, 344.142, 333.675, 313.927, 285.227, 248.952, 207.154, 162.268, 116.862, 73.403, 33.985, 0.000,
             386.097, 363.536, 347.993, 339.876, 336.087, 331.103, 320.301, 301.359, 273.977, 239.236, 199.062, 155.853, 112.175, 70.470, 32.721, 0.000,
             434.237, 405.460, 378.742, 357.282, 341.907, 329.353, 314.684, 294.308, 266.869, 232.812, 193.730, 151.849, 109.619, 69.329, 32.635, 0.000,
             490.276, 456.251, 421.297, 388.950, 361.847, 339.406, 318.101, 294.053, 264.998, 230.594, 191.922, 150.884, 109.672, 70.248, 33.753, 0.000,
             546.024, 507.593, 466.709, 426.598, 390.243, 358.564, 329.755, 300.644, 268.692, 232.947, 193.937, 153.130, 112.322, 73.008, 35.769, 0.000,
             599.175, 556.893, 511.405, 465.593, 422.324, 383.122, 347.374, 312.871, 277.319, 239.485, 199.443, 158.199, 117.079, 77.069, 38.300, 0.000,
             649.336, 603.582, 554.256, 503.966, 455.376, 410.213, 368.551, 328.983, 289.647, 249.284, 207.662, 165.366, 123.261, 81.884, 41.059, 0.000,
             696.664, 647.715, 595.028, 541.000, 488.106, 438.117, 391.493, 347.395, 304.384, 261.296, 217.715, 173.885, 130.272, 87.059, 43.882, 0.000,
             741.450, 689.520, 633.793, 576.500, 519.959, 465.925, 415.077, 366.966, 320.496, 274.636, 228.865, 183.165, 137.687, 92.352, 46.683, 0.000,
             783.987, 729.251, 670.715, 610.480, 550.733, 493.192, 438.666, 386.969, 337.272, 288.670, 240.584, 192.804, 145.242, 97.629, 49.423, 0.000,
 
             636.803, 598.991, 568.653, 542.571, 517.254, 489.530, 457.058, 418.581, 373.918, 323.795, 269.603, 213.141, 156.349, 101.046, 48.664, 0.000,
             583.039, 549.351, 523.968, 503.221, 483.160, 460.281, 432.088, 397.313, 355.849, 308.530, 256.867, 202.779, 148.329, 95.456, 45.714, 0.000,
             524.284, 495.263, 475.674, 461.169, 447.154, 429.714, 406.205, 375.394, 337.294, 292.880, 243.809, 192.135, 140.057, 89.655, 42.626, 0.000,
             459.150, 435.587, 423.060, 416.102, 409.180, 397.896, 379.522, 352.944, 318.362, 276.941, 230.506, 181.269, 131.577, 83.668, 39.407, 0.000,
             386.097, 369.198, 365.704, 368.127, 369.580, 365.213, 352.385, 330.253, 299.294, 260.912, 217.128, 170.322, 123.001, 77.571, 36.094, 0.000,
             307.052, 298.178, 305.874, 319.291, 329.947, 332.826, 325.637, 307.946, 280.572, 245.180, 203.997, 159.571, 114.567, 71.558, 32.807, 0.000,
             258.199, 251.101, 260.927, 279.034, 295.426, 303.699, 301.117, 287.261, 263.097, 230.454, 191.708, 149.544, 106.757, 66.059, 29.862, 0.000,
             307.052, 287.069, 272.724, 270.729, 277.776, 283.930, 282.291, 270.388, 248.404, 217.917, 181.256, 141.140, 100.429, 61.892, 27.919, 0.000,
             386.097, 357.596, 327.071, 302.042, 287.801, 281.135, 273.935, 260.427, 238.675, 209.251, 174.054, 135.640, 96.795, 60.178, 27.780, 0.000,
             459.150, 425.107, 386.739, 349.464, 318.871, 296.711, 279.055, 260.058, 236.042, 206.172, 171.542, 134.278, 96.871, 61.563, 29.517, 0.000,
             524.284, 485.828, 442.417, 398.219, 357.787, 323.928, 295.733, 269.396, 241.208, 209.437, 174.340, 137.424, 100.634, 65.516, 32.372, 0.000,
             583.039, 540.753, 493.406, 444.450, 397.660, 355.798, 319.234, 285.881, 252.849, 218.247, 181.761, 144.266, 107.042, 70.917, 35.637, 0.000,
             636.803, 591.061, 540.330, 487.617, 436.180, 388.569, 345.654, 306.377, 268.681, 230.872, 192.347, 153.461, 114.889, 76.900, 38.957, 0.000,
             686.589, 637.662, 583.881, 527.948, 472.769, 420.696, 372.801, 328.618, 286.705, 245.642, 204.704, 163.865, 123.336, 83.000, 42.192, 0.000,
             733.133, 681.231, 624.628, 565.804, 507.410, 451.638, 399.642, 351.307, 305.617, 261.391, 217.861, 174.742, 131.912, 89.004, 45.299, 0.000,
             776.976, 722.270, 663.017, 601.526, 540.253, 481.264, 425.741, 373.787, 324.678, 277.422, 231.242, 185.681, 140.384, 94.827, 48.267, 0.000,
 
             632.456, 594.367, 562.970, 535.661, 509.429, 481.304, 448.928, 410.930, 367.008, 317.790, 264.604, 209.206, 153.499, 99.260, 47.856, 0.000,
             577.350, 543.327, 516.660, 494.500, 473.488, 450.313, 422.398, 388.308, 347.781, 301.546, 251.052, 198.181, 144.969, 93.322, 44.732, 0.000,
             516.398, 486.970, 465.806, 449.710, 434.813, 417.330, 394.426, 364.620, 327.738, 284.648, 236.951, 186.682, 136.029, 87.054, 41.402, 0.000,
             447.214, 423.180, 408.755, 400.183, 392.767, 382.037, 364.869, 339.813, 306.864, 267.094, 222.300, 174.699, 126.656, 80.420, 37.831, 0.000,
             365.148, 347.905, 342.546, 344.188, 346.544, 344.142, 333.675, 313.927, 285.227, 248.952, 207.154, 162.268, 116.862, 73.403, 33.985, 0.000,
             258.199, 251.101, 260.927, 279.034, 295.426, 303.699, 301.117, 287.261, 263.097, 230.454, 191.708, 149.544, 106.757, 66.059, 29.862, 0.000,
             0.000, 82.347, 150.810, 203.978, 241.169, 262.432, 268.512, 260.794, 241.222, 212.199, 176.462, 136.962, 96.718, 58.690, 25.637, 0.000,
             258.199, 231.289, 193.918, 179.625, 203.628, 228.690, 240.168, 237.122, 221.381, 195.543, 162.559, 125.566, 87.768, 52.325, 22.234, 0.000,
             365.148, 333.851, 292.558, 251.358, 226.380, 223.228, 225.983, 221.711, 207.099, 183.108, 152.207, 117.444, 82.082, 49.361, 21.955, 0.000,
             447.214, 411.698, 367.560, 320.743, 279.516, 251.279, 234.828, 221.112, 203.017, 178.438, 148.390, 115.320, 82.208, 51.776, 25.084, 0.000,
             516.398, 477.024, 430.017, 379.839, 332.023, 291.763, 260.793, 235.646, 210.931, 183.386, 152.633, 120.152, 88.074, 57.909, 29.280, 0.000,
             577.350, 534.430, 484.602, 431.568, 379.637, 332.765, 292.927, 258.942, 227.567, 195.999, 163.257, 129.922, 97.100, 65.239, 33.435, 0.000,
             632.456, 586.245, 533.682, 477.997, 422.822, 371.430, 325.689, 285.323, 248.377, 212.733, 177.280, 141.982, 107.155, 72.608, 37.321, 0.000,
             683.130, 633.839, 578.639, 520.431, 462.416, 407.446, 357.270, 312.019, 270.467, 231.010, 192.559, 154.703, 117.252, 79.666, 40.928, 0.000,
             730.297, 678.102, 620.360, 559.730, 499.105, 441.058, 387.235, 337.977, 292.485, 249.504, 207.998, 167.339, 127.030, 86.342, 44.288, 0.000,
             774.597, 719.649, 659.456, 596.488, 533.411, 472.591, 415.589, 362.865, 313.886, 267.631, 223.119, 179.599, 136.386, 92.649, 47.439, 0.000,
 
             636.803, 597.691, 564.222, 534.535, 506.254, 476.737, 443.666, 405.550, 361.927, 313.290, 260.864, 206.327, 151.514, 98.124, 47.422, 0.000,
             583.039, 547.663, 518.278, 493.067, 469.521, 444.712, 416.049, 381.898, 341.779, 296.252, 246.650, 194.776, 142.600, 91.949, 44.199, 0.000,
             524.284, 492.954, 468.009, 447.801, 429.663, 410.245, 386.571, 356.822, 320.515, 278.310, 231.679, 182.580, 133.141, 85.349, 40.725, 0.000,
             459.150, 432.177, 411.999, 397.463, 385.715, 372.697, 354.833, 330.073, 297.972, 259.344, 215.850, 169.640, 123.038, 78.232, 36.934, 0.000,
             386.097, 363.536, 347.993, 339.876, 336.087, 331.103, 320.301, 301.359, 273.977, 239.236, 199.062, 155.853, 112.175, 70.470, 32.721, 0.000,
             307.052, 287.069, 272.724, 270.729, 277.776, 283.930, 282.291, 270.388, 248.404, 217.917, 181.256, 141.140, 100.429, 61.892, 27.919, 0.000,
             258.199, 231.289, 193.918, 179.625, 203.628, 228.690, 240.168, 237.122, 221.381, 195.543, 162.559, 125.566, 87.768, 52.325, 22.234, 0.000,
             307.052, 274.496, 219.253, 136.867, 86.374, 164.219, 195.339, 202.967, 194.052, 173.042, 143.748, 109.791, 74.721, 42.052, 15.435, 0.000,
             386.097, 351.345, 301.128, 239.709, 176.148, 147.642, 164.759, 175.650, 171.028, 153.742, 127.634, 96.567, 64.425, 35.386, 14.542, 0.000,
             459.150, 421.435, 371.994, 315.381, 258.142, 210.914, 184.938, 174.439, 163.510, 145.512, 120.883, 92.613, 64.683, 40.971, 21.363, 0.000,
             524.284, 483.381, 432.809, 376.651, 320.025, 269.030, 229.096, 200.595, 177.631, 154.210, 128.354, 101.257, 75.215, 51.210, 27.200, 0.000,
             583.039, 538.983, 486.561, 429.405, 371.797, 318.215, 272.223, 234.647, 202.992, 173.697, 144.753, 116.091, 88.277, 60.889, 32.103, 0.000,
             636.803, 589.708, 535.151, 476.410, 417.210, 361.220, 311.212, 268.057, 230.480, 196.225, 163.602, 131.945, 100.906, 69.569, 36.384, 0.000,
             686.589, 636.586, 579.792, 519.203, 458.151, 399.813, 346.539, 299.199, 257.072, 218.575, 182.261, 147.200, 112.615, 77.412, 40.226, 0.000,
             733.133, 680.349, 621.297, 558.743, 495.726, 435.087, 378.915, 328.067, 282.115, 239.857, 200.011, 161.533, 123.446, 84.593, 43.738, 0.000,
             776.976, 721.530, 660.236, 595.673, 530.649, 467.761, 408.913, 354.948, 305.610, 259.930, 216.744, 174.966, 133.522, 91.244, 46.993, 0.000,
 
             649.336, 608.584, 572.276, 539.301, 507.992, 476.154, 441.593, 402.727, 358.924, 310.506, 258.558, 204.646, 150.502, 97.709, 47.394, 0.000,
             599.175, 561.668, 528.583, 499.109, 471.696, 443.993, 413.530, 378.508, 338.202, 292.948, 243.913, 192.772, 141.383, 91.443, 44.164, 0.000,
             546.024, 511.799, 481.804, 455.799, 432.494, 409.329, 383.421, 352.648, 316.157, 274.306, 228.360, 180.135, 131.638, 84.715, 40.680, 0.000,
             490.276, 459.156, 431.676, 408.724, 389.608, 371.473, 350.738, 324.766, 292.510, 254.359, 211.716, 166.570, 121.122, 77.405, 36.874, 0.000,
             434.237, 405.460, 378.742, 357.282, 341.907, 329.353, 314.684, 294.308, 266.869, 232.812, 193.730, 151.849, 109.619, 69.329, 32.635, 0.000,
             386.097, 357.596, 327.071, 302.042, 287.801, 281.135, 273.935, 260.427, 238.675, 209.251, 174.054, 135.640, 96.795, 60.178, 27.780, 0.000,
             365.148, 333.851, 292.558, 251.358, 226.380, 223.228, 225.983, 221.711, 207.099, 183.108, 152.207, 117.444, 82.082, 49.361, 21.955, 0.000,
             386.097, 351.345, 301.128, 239.709, 176.148, 147.642, 164.759, 175.650, 171.028, 153.742, 127.634, 96.567, 64.425, 35.386, 14.542, 0.000,
             434.237, 396.834, 344.800, 281.198, 207.946, 122.208, 64.932, 119.908, 130.858, 121.895, 100.934, 73.141, 42.567, 9.607, 13.446, 0.000,
             490.276, 450.271, 397.876, 336.648, 270.421, 203.849, 145.702, 115.947, 111.112, 102.507, 84.892, 62.468, 43.440, 33.494, 21.047, 0.000,
             546.024, 503.280, 449.988, 389.756, 326.730, 265.733, 212.408, 171.937, 144.268, 122.828, 102.370, 82.704, 65.191, 48.021, 27.050, 0.000,
             599.175, 553.640, 498.912, 438.434, 376.133, 316.241, 262.837, 218.668, 183.570, 154.674, 129.061, 105.374, 82.703, 59.084, 32.011, 0.000,
             649.336, 601.033, 544.544, 483.090, 420.298, 359.875, 305.068, 257.812, 218.023, 183.937, 153.475, 125.078, 97.327, 68.381, 36.321, 0.000,
             696.664, 645.655, 587.231, 524.395, 460.491, 398.823, 342.135, 291.987, 248.363, 209.992, 175.188, 142.397, 110.091, 76.557, 40.179, 0.000,
             741.450, 687.815, 627.370, 562.925, 497.577, 434.320, 375.567, 322.659, 275.634, 233.485, 194.759, 157.956, 121.550, 83.941, 43.702, 0.000,
             783.987, 727.811, 665.312, 599.133, 532.161, 467.144, 406.258, 350.706, 300.560, 254.980, 212.663, 172.176, 132.032, 90.726, 46.964, 0.000,
 
             668.740, 625.915, 586.325, 549.513, 514.499, 479.599, 442.844, 402.628, 358.161, 309.580, 257.805, 204.254, 150.522, 98.042, 47.773, 0.000,
             623.482, 583.383, 546.180, 511.866, 479.771, 448.227, 415.052, 378.389, 337.288, 291.845, 243.014, 192.303, 141.407, 91.849, 44.631, 0.000,
             577.350, 539.821, 504.564, 472.285, 442.862, 414.704, 385.327, 352.500, 315.038, 272.959, 227.263, 179.559, 131.669, 85.225, 41.275, 0.000,
             531.830, 496.468, 462.234, 430.975, 403.544, 378.599, 353.220, 324.576, 291.093, 252.666, 210.336, 165.841, 121.161, 78.070, 37.663, 0.000,
             490.276, 456.251, 421.297, 388.950, 361.847, 339.406, 318.101, 294.053, 264.998, 230.594, 191.922, 150.884, 109.672, 70.248, 33.753, 0.000,
             459.150, 425.107, 386.739, 349.464, 318.871, 296.711, 279.055, 260.058, 236.042, 206.172, 171.542, 134.278, 96.871, 61.563, 29.517, 0.000,
             447.214, 411.698, 367.560, 320.743, 279.516, 251.279, 234.828, 221.112, 203.017, 178.438, 148.390, 115.320, 82.208, 51.776, 25.084, 0.000,
             459.150, 421.435, 371.994, 315.381, 258.142, 210.914, 184.938, 174.439, 163.510, 145.512, 120.883, 92.613, 64.683, 40.971, 21.363, 0.000,
             490.276, 450.271, 397.876, 336.648, 270.421, 203.849, 145.702, 115.947, 111.112, 102.507, 84.892, 62.468, 43.440, 33.494, 21.047, 0.000,
             531.830, 489.496, 435.549, 373.379, 306.503, 238.406, 172.409, 111.534, 58.383, 15.048, 16.970, 36.807, 44.264, 39.797, 24.491, 0.000,
             577.350, 532.597, 477.366, 415.024, 349.226, 283.832, 222.898, 170.644, 130.765, 104.058, 87.257, 76.017, 65.440, 50.624, 28.914, 0.000,
             623.482, 576.240, 519.588, 456.800, 391.503, 327.511, 268.604, 218.043, 177.594, 146.608, 122.555, 102.388, 82.826, 60.544, 33.191, 0.000,
             668.740, 618.988, 560.725, 497.087, 431.548, 367.683, 308.810, 257.432, 214.544, 179.334, 149.756, 123.330, 97.402, 69.338, 37.146, 0.000,
             712.543, 660.296, 600.279, 535.467, 469.148, 404.621, 344.806, 291.726, 246.033, 206.947, 172.725, 141.223, 110.143, 77.244, 40.795, 0.000,
             754.719, 700.014, 638.151, 571.944, 504.489, 438.838, 377.593, 322.465, 273.937, 231.287, 192.980, 157.100, 121.589, 84.465, 44.184, 0.000,
             795.271, 738.162, 674.400, 606.653, 537.838, 470.789, 407.863, 350.556, 299.255, 253.299, 211.302, 171.517, 132.063, 91.142, 47.354, 0.000,
 
             693.354, 648.198, 605.169, 564.321, 525.259, 486.821, 447.338, 405.257, 359.679, 310.563, 258.645, 205.174, 151.574, 99.101, 48.533, 0.000,
             653.357, 610.463, 569.150, 529.958, 492.914, 457.015, 420.486, 381.548, 339.103, 293.016, 244.017, 193.403, 142.672, 93.131, 45.555, 0.000,
             614.104, 573.225, 533.079, 494.893, 459.330, 425.686, 392.073, 356.392, 317.259, 274.389, 228.486, 180.907, 133.229, 86.819, 42.431, 0.000,
             577.350, 538.042, 498.174, 459.890, 424.812, 392.797, 361.887, 329.529, 293.898, 254.463, 211.874, 167.544, 123.150, 80.121, 39.159, 0.000,
             546.024, 507.593, 466.709, 426.598, 390.243, 358.564, 329.755, 300.644, 268.692, 232.947, 193.937, 153.130, 112.322, 73.008, 35.769, 0.000,
             524.284, 485.828, 442.417, 398.219, 357.787, 323.928, 295.733, 269.396, 241.208, 209.437, 174.340, 137.424, 100.634, 65.516, 32.372, 0.000,
             516.398, 477.024, 430.017, 379.839, 332.023, 291.763, 260.793, 235.646, 210.931, 183.386, 152.633, 120.152, 88.074, 57.909, 29.280, 0.000,
             524.284, 483.381, 432.809, 376.651, 320.025, 269.030, 229.096, 200.595, 177.631, 154.210, 128.354, 101.257, 75.215, 51.210, 27.200, 0.000,
             546.024, 503.280, 449.988, 389.756, 326.730, 265.733, 212.408, 171.937, 144.268, 122.828, 102.370, 82.704, 65.191, 48.021, 27.050, 0.000,
             577.350, 532.597, 477.366, 415.024, 349.226, 283.832, 222.898, 170.644, 130.765, 104.058, 87.257, 76.017, 65.440, 50.624, 28.914, 0.000,
             614.104, 567.217, 510.410, 446.936, 380.324, 314.229, 252.321, 198.135, 154.728, 123.739, 103.748, 89.773, 75.700, 57.096, 31.919, 0.000,
             653.357, 604.249, 545.927, 481.553, 414.580, 348.584, 287.059, 233.123, 188.982, 155.133, 129.757, 109.058, 88.578, 64.674, 35.300, 0.000,
             693.354, 641.974, 582.080, 516.720, 449.245, 383.082, 321.465, 267.039, 221.323, 184.211, 153.890, 127.337, 101.108, 72.200, 38.700, 0.000,
             733.133, 679.463, 617.912, 551.403, 483.152, 416.398, 354.090, 298.469, 250.617, 210.176, 175.467, 143.944, 112.761, 79.358, 41.991, 0.000,
             772.195, 716.241, 652.963, 585.152, 515.875, 448.178, 384.747, 327.514, 277.290, 233.620, 194.963, 159.096, 123.557, 86.101, 45.136, 0.000,
             810.303, 752.090, 687.034, 617.800, 547.305, 478.411, 413.581, 354.511, 301.841, 255.083, 212.819, 173.059, 133.609, 92.454, 48.133, 0.000,
 
             721.584, 673.971, 627.537, 582.702, 539.536, 497.343, 454.793, 410.461, 363.394, 313.401, 261.034, 207.353, 153.595, 100.820, 49.621, 0.000,
             686.589, 640.860, 595.657, 551.864, 509.997, 469.619, 429.402, 387.752, 343.519, 296.384, 246.851, 195.994, 145.083, 95.188, 46.859, 0.000,
             653.357, 609.235, 564.731, 521.315, 480.111, 441.084, 402.963, 363.948, 322.620, 278.469, 231.921, 184.054, 136.167, 89.332, 44.023, 0.000,
             623.482, 580.557, 536.015, 492.021, 450.495, 412.027, 375.533, 338.982, 300.583, 259.540, 216.149, 171.471, 126.827, 83.269, 41.145, 0.000,
             599.175, 556.893, 511.405, 465.593, 422.324, 383.122, 347.374, 312.871, 277.319, 239.485, 199.443, 158.199, 117.079, 77.069, 38.300, 0.000,
             583.039, 540.753, 493.406, 444.450, 397.660, 355.798, 319.234, 285.881, 252.849, 218.247, 181.761, 144.266, 107.042, 70.917, 35.637, 0.000,
             577.350, 534.430, 484.602, 431.568, 379.637, 332.765, 292.927, 258.942, 227.567, 195.999, 163.257, 129.922, 97.100, 65.239, 33.435, 0.000,
             583.039, 538.983, 486.561, 429.405, 371.797, 318.215, 272.223, 234.647, 202.992, 173.697, 144.753, 116.091, 88.277, 60.889, 32.103, 0.000,
             599.175, 553.640, 498.912, 438.434, 376.133, 316.241, 262.837, 218.668, 183.570, 154.674, 129.061, 105.374, 82.703, 59.084, 32.011, 0.000,
             623.482, 576.240, 519.588, 456.800, 391.503, 327.511, 268.604, 218.043, 177.594, 146.608, 122.555, 102.388, 82.826, 60.544, 33.191, 0.000,
             653.357, 604.249, 545.927, 481.553, 414.580, 348.584, 287.059, 233.123, 188.982, 155.133, 129.757, 109.058, 88.578, 64.674, 35.300, 0.000,
             686.589, 635.504, 575.616, 509.992, 441.982, 375.023, 312.422, 257.051, 210.887, 174.345, 145.737, 121.506, 97.477, 70.257, 37.919, 0.000,
             721.584, 668.446, 606.998, 540.209, 471.328, 403.666, 340.335, 283.914, 235.995, 196.676, 164.287, 135.733, 107.436, 76.384, 40.751, 0.000,
             757.290, 702.057, 639.012, 571.020, 501.219, 432.773, 368.587, 310.944, 261.096, 218.901, 182.757, 149.969, 117.466, 82.591, 43.630, 0.000,
             793.052, 735.707, 671.018, 601.744, 530.901, 461.500, 396.236, 337.132, 285.182, 240.104, 200.387, 163.647, 127.200, 88.673, 46.475, 0.000,
             828.478, 769.020, 702.649, 632.011, 560.000, 489.472, 422.931, 362.183, 308.033, 260.124, 217.040, 176.637, 136.523, 94.554, 49.247, 0.000,
 
             752.121, 702.012, 652.304, 603.673, 556.528, 510.566, 464.793, 417.959, 369.113, 317.949, 264.848, 210.678, 156.476, 103.103, 50.977, 0.000,
             721.584, 673.060, 624.252, 576.252, 529.892, 485.174, 441.187, 396.586, 350.250, 301.733, 251.337, 199.908, 148.477, 97.877, 48.453, 0.000,
             693.354, 646.143, 597.767, 549.800, 503.608, 459.610, 417.064, 374.532, 330.681, 284.870, 237.291, 188.742, 140.233, 92.548, 45.920, 0.000,
             668.740, 622.480, 573.953, 525.260, 478.393, 434.348, 392.680, 351.905, 310.434, 267.363, 222.711, 177.200, 131.790, 87.176, 43.426, 0.000,
             649.336, 603.582, 554.256, 503.966, 455.376, 410.213, 368.551, 328.983, 289.647, 249.284, 207.662, 165.366, 123.261, 81.884, 41.059, 0.000,
             636.803, 591.061, 540.330, 487.617, 436.180, 388.569, 345.654, 306.377, 268.681, 230.872, 192.347, 153.461, 114.889, 76.900, 38.957, 0.000,
             632.456, 586.245, 533.682, 477.997, 422.822, 371.430, 325.689, 285.323, 248.377, 212.733, 177.280, 141.982, 107.155, 72.608, 37.321, 0.000,
             636.803, 589.708, 535.151, 476.410, 417.210, 361.220, 311.212, 268.057, 230.480, 196.225, 163.602, 131.945, 100.906, 69.569, 36.384, 0.000,
             649.336, 601.033, 544.544, 483.090, 420.298, 359.875, 305.068, 257.812, 218.023, 183.937, 153.475, 125.078, 97.327, 68.381, 36.321, 0.000,
             668.740, 618.988, 560.725, 497.087, 431.548, 367.683, 308.810, 257.432, 214.544, 179.334, 149.756, 123.330, 97.402, 69.338, 37.146, 0.000,
             693.354, 641.974, 582.080, 516.720, 449.245, 383.082, 321.465, 267.039, 221.323, 184.211, 153.890, 127.337, 101.108, 72.200, 38.700, 0.000,
             721.584, 668.446, 606.998, 540.209, 471.328, 403.666, 340.335, 283.914, 235.995, 196.676, 164.287, 135.733, 107.436, 76.384, 40.751, 0.000,
             752.121, 697.133, 634.152, 566.067, 496.035, 427.253, 362.696, 304.783, 254.962, 213.263, 178.087, 146.520, 115.208, 81.315, 43.092, 0.000,
             783.987, 727.087, 662.561, 593.209, 522.095, 452.297, 386.633, 327.329, 275.624, 231.425, 193.189, 158.258, 123.594, 86.586, 45.574, 0.000,
             816.497, 757.647, 691.546, 620.905, 548.683, 477.834, 411.016, 350.254, 296.593, 249.833, 208.499, 170.174, 132.123, 91.957, 48.106, 0.000,
             849.182, 788.365, 720.660, 648.684, 575.292, 503.310, 435.235, 372.913, 317.226, 267.897, 223.526, 181.906, 140.560, 97.295, 50.633, 0.000,
 
             783.987, 731.395, 678.583, 626.399, 575.494, 525.880, 476.866, 427.396, 376.567, 323.996, 269.910, 214.993, 160.080, 105.840, 52.538, 0.000,
             757.290, 706.047, 653.914, 602.099, 551.638, 502.850, 455.182, 407.549, 358.915, 308.762, 257.222, 204.924, 152.663, 101.049, 50.254, 0.000,
             733.133, 682.986, 631.134, 579.183, 528.616, 480.149, 433.445, 387.417, 340.884, 293.152, 244.224, 194.646, 145.151, 96.257, 48.009, 0.000,
             712.543, 663.178, 611.149, 558.470, 507.122, 458.306, 412.017, 367.229, 322.614, 277.262, 230.998, 184.245, 137.634, 91.548, 45.858, 0.000,
             696.664, 647.715, 595.028, 541.000, 488.106, 438.117, 391.493, 347.395, 304.384, 261.296, 217.715, 173.885, 130.272, 87.059, 43.882, 0.000,
             686.589, 637.662, 583.881, 527.948, 472.769, 420.696, 372.801, 328.618, 286.705, 245.642, 204.704, 163.865, 123.336, 83.000, 42.192, 0.000,
             683.130, 633.839, 578.639, 520.431, 462.416, 407.446, 357.270, 312.019, 270.467, 231.010, 192.559, 154.703, 117.252, 79.666, 40.928, 0.000,
             686.589, 636.586, 579.792, 519.203, 458.151, 399.813, 346.539, 299.199, 257.072, 218.575, 182.261, 147.200, 112.615, 77.412, 40.226, 0.000,
             696.664, 645.655, 587.231, 524.395, 460.491, 398.823, 342.135, 291.987, 248.363, 209.992, 175.188, 142.397, 110.091, 76.557, 40.179, 0.000,
             712.543, 660.296, 600.279, 535.467, 469.148, 404.621, 344.806, 291.726, 246.033, 206.947, 172.725, 141.223, 110.143, 77.244, 40.795, 0.000,
             733.133, 679.463, 617.912, 551.403, 483.152, 416.398, 354.090, 298.469, 250.617, 210.176, 175.467, 143.944, 112.761, 79.358, 41.991, 0.000,
             757.290, 702.057, 639.012, 571.020, 501.219, 432.773, 368.587, 310.944, 261.096, 218.901, 182.757, 149.969, 117.466, 82.591, 43.630, 0.000,
             783.987, 727.087, 662.561, 593.209, 522.095, 452.297, 386.633, 327.329, 275.624, 231.425, 193.189, 158.258, 123.594, 86.586, 45.574, 0.000,
             812.383, 753.739, 687.724, 617.067, 544.750, 473.759, 406.804, 345.992, 292.468, 246.102, 205.403, 167.846, 130.554, 91.040, 47.707, 0.000,
             841.829, 781.390, 713.865, 641.909, 568.419, 496.278, 428.074, 365.777, 310.407, 261.773, 218.442, 178.052, 137.926, 95.732, 49.942, 0.000,
             871.847, 809.579, 740.521, 667.247, 592.566, 519.256, 449.776, 385.956, 328.693, 277.742, 231.728, 188.455, 145.445, 100.520, 52.224, 0.000,
 
             816.497, 761.460, 705.715, 650.226, 595.813, 542.730, 490.543, 438.396, 385.457, 331.298, 276.017, 220.123, 164.264, 108.928, 54.248, 0.000,
             793.052, 739.179, 683.966, 628.686, 574.499, 521.952, 470.778, 420.138, 369.109, 317.141, 264.229, 210.806, 157.449, 104.567, 52.190, 0.000,
             772.195, 719.254, 664.238, 608.747, 554.320, 501.855, 451.321, 401.929, 352.673, 302.855, 252.338, 201.446, 150.665, 100.285, 50.206, 0.000,
             754.719, 702.437, 647.251, 591.089, 535.886, 482.951, 432.564, 384.058, 336.356, 288.599, 240.475, 192.168, 144.021, 96.170, 48.347, 0.000,
             741.450, 689.520, 633.793, 576.500, 519.959, 465.925, 415.077, 366.966, 320.496, 274.636, 228.865, 183.165, 137.687, 92.352, 46.683, 0.000,
             733.133, 681.231, 624.628, 565.804, 507.410, 451.638, 399.642, 351.307, 305.617, 261.391, 217.861, 174.742, 131.912, 89.004, 45.299, 0.000,
             730.297, 678.102, 620.360, 559.730, 499.105, 441.058, 387.235, 337.977, 292.485, 249.504, 207.998, 167.339, 127.030, 86.342, 44.288, 0.000,
             733.133, 680.349, 621.297, 558.743, 495.726, 435.087, 378.915, 328.067, 282.115, 239.857, 200.011, 161.533, 123.446, 84.593, 43.738, 0.000,
             741.450, 687.815, 627.370, 562.925, 497.577, 434.320, 375.567, 322.659, 275.634, 233.485, 194.759, 157.956, 121.550, 83.941, 43.702, 0.000,
             754.719, 700.014, 638.151, 571.944, 504.489, 438.838, 377.593, 322.465, 273.937, 231.287, 192.980, 157.100, 121.589, 84.465, 44.184, 0.000,
             772.195, 716.241, 652.963, 585.152, 515.875, 448.178, 384.747, 327.514, 277.290, 233.620, 194.963, 159.096, 123.557, 86.101, 45.136, 0.000,
             793.052, 735.707, 671.018, 601.744, 530.901, 461.500, 396.236, 337.132, 285.182, 240.104, 200.387, 163.647, 127.200, 88.673, 46.475, 0.000,
             816.497, 757.647, 691.546, 620.905, 548.683, 477.834, 411.016, 350.254, 296.593, 249.833, 208.499, 170.174, 132.123, 91.957, 48.106, 0.000,
             841.829, 781.390, 713.865, 641.909, 568.419, 496.278, 428.074, 365.777, 310.407, 261.773, 218.442, 178.052, 137.926, 95.732, 49.942, 0.000,
             868.474, 806.382, 737.414, 664.162, 589.453, 516.092, 446.577, 382.790, 325.684, 275.048, 229.491, 186.753, 144.276, 99.822, 51.913, 0.000,
             895.976, 832.187, 761.754, 687.201, 611.283, 536.717, 465.903, 400.622, 341.743, 289.024, 241.123, 195.897, 150.928, 104.091, 53.963, 0.000,
 
             849.182, 791.753, 733.237, 674.672, 616.998, 560.649, 505.409, 450.603, 395.486, 339.608, 282.962, 225.899, 168.893, 112.273, 56.061, 0.000,
             828.478, 772.063, 713.978, 655.527, 597.946, 541.939, 487.468, 433.905, 380.450, 326.550, 272.092, 217.336, 162.668, 108.319, 54.210, 0.000,
             810.303, 754.695, 696.758, 638.074, 580.195, 524.138, 470.091, 417.514, 365.562, 313.568, 261.288, 208.865, 156.567, 104.500, 52.455, 0.000,
             795.271, 740.230, 682.142, 622.859, 564.259, 507.701, 453.657, 401.729, 351.057, 300.849, 250.708, 200.622, 150.705, 100.899, 50.840, 0.000,
             783.987, 729.251, 670.715, 610.480, 550.733, 493.192, 438.666, 386.969, 337.272, 288.670, 240.584, 192.804, 145.242, 97.629, 49.423, 0.000,
             776.976, 722.270, 663.017, 601.526, 540.253, 481.264, 425.741, 373.787, 324.678, 277.422, 231.242, 185.681, 140.384, 94.827, 48.267, 0.000,
             774.597, 719.649, 659.456, 596.488, 533.411, 472.591, 415.589, 362.865, 313.886, 267.631, 223.119, 179.599, 136.386, 92.649, 47.439, 0.000,
             776.976, 721.530, 660.236, 595.673, 530.649, 467.761, 408.913, 354.948, 305.610, 259.930, 216.744, 174.966, 133.522, 91.244, 46.993, 0.000,
             783.987, 727.811, 665.312, 599.133, 532.161, 467.144, 406.258, 350.706, 300.560, 254.980, 212.663, 172.176, 132.032, 90.726, 46.964, 0.000,
             795.271, 738.162, 674.400, 606.653, 537.838, 470.789, 407.863, 350.556, 299.255, 253.299, 211.302, 171.517, 132.063, 91.142, 47.354, 0.000,
             810.303, 752.090, 687.034, 617.800, 547.305, 478.411, 413.581, 354.511, 301.841, 255.083, 212.819, 173.059, 133.609, 92.454, 48.133, 0.000,
             828.478, 769.020, 702.649, 632.011, 560.000, 489.472, 422.931, 362.183, 308.033, 260.124, 217.040, 176.637, 136.523, 94.554, 49.247, 0.000,
             849.182, 788.365, 720.660, 648.684, 575.292, 503.310, 435.235, 372.913, 317.226, 267.897, 223.526, 181.906, 140.560, 97.295, 50.633, 0.000,
             871.847, 809.579, 740.521, 667.247, 592.566, 519.256, 449.776, 385.956, 328.693, 277.742, 231.728, 188.455, 145.445, 100.520, 52.224, 0.000,
             895.976, 832.187, 761.754, 687.201, 611.283, 536.717, 465.903, 400.622, 341.743, 289.024, 241.123, 195.897, 150.928, 104.091, 53.963, 0.000,
             921.156, 855.793, 783.961, 708.132, 630.997, 555.205, 483.086, 416.348, 355.812, 301.224, 251.281, 203.915, 156.801, 107.891, 55.803, 0.000
         ],
         "x"=>[
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
 
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
 
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
 
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
 
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
 
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
 
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
 
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
 
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
 
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
 
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
 
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
 
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
 
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
 
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
 
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000
         ],
         "y"=>[
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000,
 
             0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
             8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000, 8.000,
             16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000, 16.000,
             24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000, 24.000,
             32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000, 32.000,
             40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000, 40.000,
             48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000, 48.000,
             56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000, 56.000,
             64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000, 64.000,
             72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000, 72.000,
             80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000, 80.000,
             88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000, 88.000,
             96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000, 96.000,
             104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000, 104.000,
             112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000, 112.000,
             120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000, 120.000
         ],
         "z"=>[
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
 
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000,
             0.000, 0.533, 2.133, 4.800, 8.533, 13.333, 19.200, 26.133, 34.133, 43.200, 53.333, 64.533, 76.800, 90.133, 104.533, 120.000
         ]
     );
 
 my $plot = Chart::Plotly::Plot->new(
     traces => [ $volume ],
     layout => {
         "title"=>{
             "text"=>"volume airflow model"
         },
         "width"=>1000,
         "height"=>750,
         "scene"=>{
             "aspectratio"=>{
                 "x"=>1,
                 "y"=>1,
                 "z"=>1
             },
             "xaxis"=>{ "nticks"=>12 },
             "yaxis"=>{ "nticks"=>12 },
             "zaxis"=>{ "nticks"=>12 },
             "camera"=>{
                 "eye"=>{
                     "x"=>1.75,
                     "y"=>0.2,
                     "z"=>0.35
                 },
                 "center"=>{
                     "x"=>0,
                     "y"=>0,
                     "z"=>0
                 }
             }
         }
     }
 );
 
 Chart::Plotly::show_plot($plot);

=head1 DESCRIPTION

This attribute is part of the possible options for the trace volume.

This file has been autogenerated from the official plotly.js source.

If you like Plotly, please support them: L<https://plot.ly/> 
Open source announcement: L<https://plot.ly/javascript/open-source-announcement/>

Full reference: L<https://plot.ly/javascript/reference/#volume>

=head1 DISCLAIMER

This is an unofficial Plotly Perl module. Currently I'm not affiliated in any way with Plotly. 
But I think plotly.js is a great library and I want to use it with perl.

=head1 METHODS

=head2 TO_JSON

Serialize the trace to JSON. This method should be called only by L<JSON> serializer.

=head1 ATTRIBUTES

=over

=item * fill

Sets the fill ratio of the `slices`. The default fill value of the `slices` is 1 meaning that they are entirely shaded. On the other hand Applying a `fill` ratio less than one would allow the creation of openings parallel to the edges.

=item * locations

Specifies the location(s) of slices on the axis. When not specified slices would be created for all points of the axis x except start and end.

=item * locationssrc

Sets the source reference on plot.ly for  locations .

=item * show

Determines whether or not slice planes about the x dimension are drawn.

=back

=head1 AUTHOR

Pablo Rodríguez González <pablo.rodriguez.gonzalez@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2020 by Pablo Rodríguez González.

This is free software, licensed under:

  The MIT (X11) License

=cut
