# Vote::Count

## A Toolkit for determining the outcome of Ranked Choice and other Alternative Balloting Strategies.

Provides a Toolkit for implementing multiple voting systems, allowing a wide range of method options. This library allows the creation of election resolution methods matching a set of Election Rules that are written in an organization's governing rules, and not requiring the bylaws to specify the rules of the software that will be used for the election, especially important given that many of the other libraries available do not provide a bylaws compatible explanation of their process.

This is also extremely useful to researchers who may want to study multiple methods and variations of methods.

# Synopsis

```perl
use 5.022; # Minimum Perl, or any later Perl.
use feature qw /postderef signatures/;

use Vote::Count;
use Vote::Count::Method::CondorcetDropping;
use Vote::Count::ReadBallots 'read_ballots';


# example uses biggerset1 from the distribution test data.
my $ballotset = read_ballots 't/data/biggerset1.txt' ;
my $CondorcetElection =
  Vote::Count::Method::CondorcetDropping->new(
    'BallotSet' => $ballotset ,
    'DropStyle' => 'all',
    'DropRule'  => 'topcount',
  );
# ChoicesAfterFloor a hashref of choices meeting the
# ApprovalFloor which defaulted to 5%.
my $ChoicesAfterFloor = $CondorcetElection->ApprovalFloor();
# Apply the ChoicesAfterFloor to the Election.
$CondorcetElection->Active( $ChoicesAfterFloor );
# Get Smith Set and the Election with it as the Active List.
my $SmithSet = $CondorcetElection->Matrix()->SmithSet() ;
$CondorcetElection->logt(
  "Dominant Set Is: " . join( ', ', keys( $SmithSet->%* )));
my $Winner = $CondorcetElection->RunCondorcetDropping( $SmithSet );

# Create an object for IRV, use the same Floor as Condorcet
my $IRVElection = Vote::Count::Method::IRV->new(
  'BallotSet' => $ballotset,
  'Active' => $ChoicesAfterFloor );
# Get a RankCount Object for the
my $Plurality = $IRVElection->TopCount();
# In case of ties RankCount objects return top as an array, log the result.
my $PluralityWinner = $Plurality->Leader();
$IRVElection->logv( "Plurality Results", $Plurality->RankTable);
if ( $PluralityWinner->{'winner'}) {
  $IRVElection->logt( "Plurality Winner: ", $PluralityWinner->{'winner'} )
} else {
  $IRVElection->logt(
    "Plurality Tie: " . join( ', ', $PluralityWinner->{'tied'}->@*) )
}
my $IRVResult = $IRVElection->RunIRV();

# Now print the logs and winning information.
note $CondorcetElection->logv();
note $IRVElection->logv();
note '******************';
note "Plurality Winner: $PluralityWinner->{'winner'}";
note "IRV Winner: $IRVResult->{'winner'}";
note "Winner: $Winner";

```

# Preview Release

This module is not ready for production.

# Overview

## Brief Review of Voting Methods

Several alternatives have been proposed to the simple vote for a single choice method that has been used in most elections for a single member. In addition a number of different methods have been used for multiple members. For alternative single member voting, the three common alternatives are *Approval* (voters indicate all choices that they approve of), *Ranked Choice* (Voters rank the choices), and *Score* also known as *Range* (A Ranked Choice Ballot where the number of rankings is limited but voters may rank more than 1 choice at each rank).

*Vote for One* ballots may be resolved by one of two methods: *Majority* and *Plurality*. Majority vote requires a majority of votes to win (but frequently produces no winner), and Plurality which selects the choice with the most votes.

Numerous methods have been proposed and tried for *Ranked Choice Ballots*. To compare these methods a number of criteria have been developed. While Mathematicians treat these criteria as absolutes, Policy oriented people may prefer to see them as a range. Furthermore Policy oriented people see the majority of the criteria proposed by Mathematicians as subcomponents of a single group, Consistency. In addition Policy oriented People will add a criteria for Complexity, which is generally not considered by the Mathematical Community.



### The Criteria for Resolving Ranked Choice (including Score) Ballots

#### Later Harm

Marking a later Choice on a Ballot should not cause a Voter's higher ranked choice to lose.

#### Condorcet Criteria

*Condorcet Loser* states that a method should not choose a winner that would be defeated in a direct matchup to all other choices.

*Condorcet Winner* states a choice which defeats all others in direct matchups should be the Winner.

#### Consistency Criteria

The Consistency Criteria collectively state that changes to non-winning choices should not change the winner. To illustrate inconsistency: suppose every morning we vote on a flavor of Ice Cream and Chocolate always wins; one morning the three voters who always vote 1:RockyRoad 2:Chocolate simply vote for Chocolate, consistency is violated if Chocolate loses on that day.

#### Complexity

Is it feasible to count 100 ballots by hand? How difficult is it to understand the process (can the average middle school student understand it at all)? How many steps?

#### Resolvability

*Majority* meets all of the above criteria, however, unless votes are restricted to two choices, it will frequently fail to provide a winner, or even a tie. No method can be completely impervious to ties. Methods that are not Resolveable are frequently combined with other methods, *Instant Runoff Voting* is a compound method with *Majority*, and all usable *Condorcet* Methods combine seeking a Condorcet Winner with some other process for when there isn't one.

### Popular Ranked Choice Resolution Systems

#### Instant Runoff Voting (IRV also known as Alternative Vote)

Seeks a Majority Winner. If there is none the lowest choice is eliminated until there is a Majority Winner or all remaining choices have the same number of votes.

* Easy to Hand Count and Easy to Understand
* Meets Later Harm
* Fails Condorcet Winner (but meets Condorcet Loser)
* Fails many Consistency Criteria (The example given for Consistency can happen with IRV)

#### Borda Count

Scores choices on a ballot based on their position. Borda supporters often disagree about the weighting rule to use in the scoring. Iterative Borda Methods (Baldwin, Nansen) eliminate the lowest choice and recalculate the Borda score ignoring choices for eliminated ones (if your second choice is eliminated your third choice is promoted).

* Easy to Understand
* Fails Later Harm
* Fails Condorcet Winner
* Fails many Consistency Criteria (and is vulnerable to manipulation if voters coordinate their votes)

#### Condorcet

Technically this family of methods should be called Condorcet Pairwise, because any method that meets both Condorcet Criteria is technically a Condorcet Method. However, in discussion and throughout this software collection the term Condorcet will refer to any method which uses a Matrix of Wins and Losses derived from direct pairing of choices and seeks to identify a Condorcet Winner.

The basic Condorcet Method will frequently fail to identify a winner. One possibility is a Loop (Condorcet's Paradox) Where A > B, B > C, and C > A. Another possibility is a knot (not an accepted term, but one which may be used in this documentation). A Loop can be considered a tie, and a knot can be considered something less than a tie. To make Condorcet resolveable a second method is used to resolve Loops and Knots.

* Complexity Varies among sub-methods
* Fails Later Harm
* Meets both Condorcet Criteria
* When a Condorcet Winner is present consistency is met. Submethods vary in consistency when there is no Condorcet Winner, but are generally more consistent than IRV or Borda.

### Score Voting Systems

Most Methods for Ranked Choice Ballots can be used for Score Ballots, either directly or by translating the ballots. The majority of proposals for Score Voting simply implement the *Borda Count*. *STAR*, creates a virtual runoff between the top two *Borda Count* Choices.

Advocates of Score Voting claim that it is a better expression of voter preference (which is purely a matter of opinion and cannot be proved or disproved), but it does create more potential for ties in the resolution process than RC does (which is a reason to assert RC is better). When Advocates say it is mathematically better, they are clueless because underneath the method they're usually promoting is still *Borda Count* subject to all the failings of that method.

