package LWP::UserAgent::Plugin::Retry;

our $DATE = '2020-08-13'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;
use Log::ger;

use Time::HiRes qw(sleep);

sub after_request {
    my ($self, $r) = @_;

    $r->{config}{max_attempts} //=
        $ENV{LWP_USERAGENT_PLUGIN_RETRY_MAX_ATTEMPTS} // 3;
    $r->{config}{delay}        //=
        $ENV{LWP_USERAGENT_PLUGIN_RETRY_DELAY}        // 2;

    my $should_retry = 0;
    if (($r->{config}{retry_client_errors} // 0) && $r->{response}->code =~ /\A4/) { $should_retry++ }
    if (($r->{config}{retry_server_errors} // 1) && $r->{response}->code =~ /\A5/) { $should_retry++ }
    return -1 unless $should_retry;

    $r->{retries} //= 0;
    return 0 if $r->{config}{max_attempts} &&
        $r->{retries} >= $r->{config}{max_attempts};
    $r->{retries}++;
    my ($ua, $request) = @{ $r->{argv} };
    log_trace "Failed requesting %s (%s - %s), retrying in %.1f second(s) (%d of %d) ...",
        $request->uri,
        $r->{response}->code,
        $r->{response}->message,
        $r->{config}{delay},
        $r->{retries},
        $r->{config}{max_attempts};
    sleep $r->{config}{delay};
    98; # repeat request()
}

1;
# ABSTRACT: Retry failed requests

__END__

=pod

=encoding UTF-8

=head1 NAME

LWP::UserAgent::Plugin::Retry - Retry failed requests

=head1 VERSION

This document describes version 0.001 of LWP::UserAgent::Plugin::Retry (from Perl distribution LWP-UserAgent-Plugin-Retry), released on 2020-08-13.

=head1 SYNOPSIS

 use LWP::UserAgent::Plugin 'Retry' => {
     max_attempts => 3, # optional, default 3
     delay        => 2, # optional, default 2
     retry_if     => qr/^[45]/, # optional, default is only 5xx errors are retried
 };

 my $res  = HTTP::Tiny::Plugin->new->get("http://www.example.com/");

=head1 DESCRIPTION

This plugin retries failed response. By default only retries 5xx failures, as
4xx are considered to be client's fault (but you can configure it with
L</retry_if>).

=for Pod::Coverage .+

=head1 CONFIGURATION

=head2 max_attempts

Int.

=head2 delay

Float.

=head2 retry_client_errors

Bool, default 0. Whether 4xx errors should be retried.

=head2 retry_server_errors

Bool, default 1. Whether 5xx errors should be retried.

=head1 ENVIRONMENT

=head2 LWP_USERAGENT_PLUGIN_RETRY_MAX_ATTEMPTS

Int.

=head2 LWP_USERAGENT_PLUGIN_RETRY_DELAY

Int.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/LWP-UserAgent-Plugin-Retry>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-LWP-UserAgent-Plugin-Retry>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=LWP-UserAgent-Plugin-Retry>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<LWP::UserAgent::Plugin::CustomRetry>

L<LWP::UserAgent::Plugin>

Existing non-plugin solutions: L<LWP::UserAgent::Determined>,
L<LWP::UserAgent::ExponentialBackoff>.

L<HTTP::Tiny::Plugin::Retry>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
