# -*- Perl -*-
#
# Music::Guidonian - a means of melodic phrase generation based on the
# "Guidonian Hand" that is credited to Guido of Arezzo

package Music::Guidonian;
our $VERSION = '0.01';

use 5.24.0;
use warnings;
use Carp 'croak';
use List::Util 'shuffle';
use List::UtilsBy 'nsort_by';
use Moo;
use namespace::clean;

use constant { INDEX => 0, CHOICE => 1, FIRST => 0, DONE => -1, DIRTY => -1 };

use parent qw(Exporter);
our @EXPORT_OK = qw(intervalize_scale_nums);

has key2pitch => ( is => 'rw' );

# perldoc Moo
sub BUILD {
    my ( $self, $args ) = @_;

    if ( exists $args->{key2pitch} and exists $args->{key_set} ) {
        croak "cannot specify both key2pitch and key_set";

    } elsif ( exists $args->{key2pitch} ) {
        croak "key2pitch must be a hash reference with keys"
          unless defined $args->{key2pitch}
          and ref $args->{key2pitch} eq 'HASH'
          and keys $args->{key2pitch}->%*;

    } elsif ( exists $args->{key_set} ) {
        my $set = $args->{key_set};
        croak "key_set must be a hash reference with keys"
          unless defined $set
          and ref $set eq 'HASH'
          and keys $set->%*;

        croak "intervals must be an array with elements"
          unless defined $set->{intervals}
          and ref $set->{intervals} eq 'ARRAY'
          and $set->{intervals}->@*;
        croak "keys must be an array with elements"
          unless defined $set->{keys}
          and ref $set->{keys} eq 'ARRAY'
          and $set->{keys}->@*;
        croak "min must be an integer"
          unless defined $set->{min} and $set->{min} =~ m/^(?a)-?\d+$/;
        croak "max must be an integer"
          unless defined $set->{max} and $set->{max} =~ m/^(?a)-?\d+$/;

        croak "min must be less than max" if $set->{min} >= $set->{max};

        my $curinterval = 0;
        my $curkey      = 0;
        my %key2pitch;
        my $pitch = $set->{min};

        while (1) {
            push @{ $key2pitch{ $set->{keys}->[$curkey] } }, $pitch;
            $pitch += $set->{intervals}->[$curinterval];
            last if $pitch > $set->{max};
            $curinterval = ++$curinterval % $set->{intervals}->@*;
            $curkey      = ++$curkey % $set->{keys}->@*;
        }
        $self->key2pitch( \%key2pitch );

        # may want to preserve this for reference or cloning?
        delete $args->{key_set};

    } else {
        croak "need key2pitch or key_set";
    }
}

########################################################################
#
# METHODS

# TODO probably need user-supplied callbacks for shuffle/weighted choice
# in the event the caller needs something different there
sub iterator {
    my ( $self, $sequence ) = @_;
    croak "sequence is not an array reference"
      unless defined $sequence and ref $sequence eq 'ARRAY';
    croak "sequence is too short" if @$sequence < 2;

    my $key2pitch = $self->key2pitch;
    croak "no key2pitch map is set"
      unless defined $key2pitch
      and ref $key2pitch eq 'HASH'
      and keys %$key2pitch;

    # the possibilities are either scalars (integer pitch numbers, a
    # static choice) or an [ INDEX, CHOICE ] array reference where the
    # CHOICE is an array reference of possible integer pitch numbers
    my @possible;
    for my $i ( 0 .. $#$sequence ) {
        my $s = $sequence->[$i];
        croak "sequence element is undefined ($i)" unless defined $s;
        if ( $s =~ m/^(?a)-?\d+$/ ) {
            push @possible, $s;
        } else {
            my $choices = $key2pitch->{$s} // '';
            croak "choices are not an array reference for '$s'"
              unless ref $choices eq 'ARRAY';
            my $length = $choices->@*;
            croak "no choices for '$s' at index $i" if $length == 0;
            if ( $length == 1 ) {
                push @possible, $choices->[0];
                next;
            }
            if ( $i > 0 ) {
                $choices = _weight_choices( _previous_pitch( $possible[ $i - 1 ] ), $choices );
            } else {
                $choices = [ shuffle @$choices ];
            }
            push @possible, [ FIRST, $choices ];    # INDEX, CHOICE
        }
    }

    # edge case: there is only one iteration due to a lack of choices.
    # fail so that the iterator is not complicated to handle that
    my $refcount = 0;
    for my $p (@possible) { $refcount++ if ref $p eq 'ARRAY' }
    croak "no choices in @possible" if $refcount == 0;

    return sub {
        return [] unless @possible;

        my @phrase;
        for my $p (@possible) {
            if ( ref $p eq 'ARRAY' ) {
                push @phrase, 0 + $p->[CHOICE][ $p->[INDEX] ];
            } else {
                push @phrase, 0 + $p;
            }
        }

        my $dirty = 0;
        for my $i ( reverse DONE .. $#possible ) {
            if ( $i == DONE ) {
                @possible = ();
                $dirty    = 0;
                last;
            } elsif ( ref $possible[$i] eq 'ARRAY' ) {
                if ( ++$possible[$i][INDEX] >= $possible[$i][CHOICE]->@* ) {
                    $possible[$i][INDEX] = DIRTY;
                    $dirty = 1;
                } else {
                    # nothing more to update (this time)
                    last;
                }
            }
        }
        if ($dirty) {
            for my $i ( 0 .. $#possible ) {
                if ( ref $possible[$i] eq 'ARRAY' and $possible[$i][INDEX] == DIRTY ) {
                    $possible[$i][INDEX]  = FIRST;
                    $possible[$i][CHOICE] = _weight_choices( _previous_pitch( $possible[ $i - 1 ] ),
                        $possible[$i][CHOICE] );
                }
            }
        }

        return \@phrase;
    };
}

sub _previous_pitch {
    my ($possible) = @_;
    return ref $possible eq 'ARRAY' ? $possible->[CHOICE][FIRST] : $possible;
}

sub _weight_choices {
    my ( $previous, $possible ) = @_;
    my @choices = nsort_by { abs( $previous - $_ ) } $possible->@*;
    return \@choices;
}

########################################################################
#
# FUNCTIONS

# convert Music::Scales "get_scale_nums" to the interval for each step,
# making various assumptions (or lack of sanity tests) along the way
# (pretty sure I've written this same code elsewhere...)
sub intervalize_scale_nums {
    my ( $scale, $max_interval ) = @_;
    $max_interval ||= 12;    # assume Western 12-tone system
    my @intervals;
    my $previous = 0;
    for my $s (@$scale) {
        next if $s == 0;
        push @intervals, $s - $previous;
        $previous = $s;
    }
    push @intervals, $max_interval - $previous;
    return \@intervals;
}

1;
__END__

=head1 NAME

Music::Guidonian - a "Guidonian Hand" melodic phrase generator

=head1 SYNOPSIS

  my $mg = Music::Guidonian->new(
    key_set => {
      intervals => [ 2, 2, 1, 2, 2, 2, 1 ],
      keys      => [qw(a e i o u)],         
      min       => 48,
      max       => 72
    }
  );

  my @text   = qw(Lo rem ip sum do lor sit);
  my @vowels = map { m/([aeiou])/; $1 } @text;

  my $iter   = $mg->iterator(\@vowels);

  $iter->();        # [71,67,69,72,71,71,69] (maybe)

=head1 DESCRIPTION

"Guido of Arezzo" is credited with the creation of the "Guidonian
Hand" which grew into among other things a method to aid with the
creation of new music. This implementation is based off of a
description of the process found in "Musimathics" (Volume 1, Chapter
9). In brief, pitches in a given ambitus are mapped to particular
letters, usually vowels using some sequence of intervals (a scale).
Then, given a sequence of those particular letters, a sequence of
pitch numbers is returned for each call to an iterator function until
no more possibilities remain, or, more likely for longer phrases, the
caller gives up having found something suitable or otherwise having
aborted early. Pitch numbers may be included in the input sequence to
lock those positions to the given pitches.

Pitches are integers, typically MIDI numbers. These may need to be
confined to a particular range (the ambitus) of values. Keys could be
any scalar value but typically will be the vowels of a text phrase that
is to be set to music. The caller may need to manually or otherwise
process the text phrase to extract the vowels, and then after obtaining
results from the B<iterator> convert the pitch numbers returned into
L<MIDI> or musical notes for a scorewriter.

=head2 What is that synopsis code even doing?

The synopsis code should result in the keys (vowels) being mapped to
pitches as follows;

   a  e  i  o  u  a  e  i  o  u  a  e  i  o  u
  48 50 52 53 55 57 59 60 62 64 65 67 69 71 72
   C  D  E  F  G  A  B  C  D  E  F  G  A  B  C

the iterator function works (eventually, and assuming no bugs) through
all possible combinations given that there are multiple choices for each
vowel: the "o" of "Lorem" maps to 53 or 62 or 71, and then the "e", etc.
Obviously longer phrases will suffer from what has been called the
"combinatorial explosion" (see "The Lighthill debate on Artificial
Intelligence").

=head2 Caveat

Various calls will throw exceptions when something is awry with
the input.

=head3 Caveat

Various calls may accept bad data that will not generate known
exceptions.

=head1 CONSTRUCTOR

The B<new> method requires either that the B<key2pitch> attribute is
set, or that B<key_set> containing B<intervals>, B<keys>, B<min>, and
B<max> is set so that B<key2pitch> can be constructed from those values.

L<Music::Scales> can be used to obtain suitable B<intervals> for
different known scales. These will need to be converted with the
B<intervalize_scale_nums> function. L<Music::AtonalUtil> is another way
to obtain pitch set intervals.

=head1 ATTRIBUTES

=over 4

=item B<key2pitch>

This attribute must be set for the B<iterator> method to be able to
generate choices from a given I<sequence> of keys. Example keys for a
Latin phrase would typically be the vowels C<a e i o u>. These vowels
must map to one or more integer pitch numbers.

  Music::Guidonian->new(
    key2pitch => { i => [60, 67], a => [62, 69], ... } );
  );

=back

=head1 METHODS

=over 4

=item B<iterator> I<sequence>

This method accepts an array reference that is a I<sequence> of B<key>
values or integer pitch numbers. A function is returned. Each call of
the function will return an array reference containing a list of integer
pitch numbers. When there are no more combinations an empty array
reference is returned.

=back

=head1 FUNCTIONS

These are not exported by default. They are mostly for internal use.

=over 4

=item B<intervalize_scale_nums> I<scale> [ I<max-interval> ]

Converts the output of C<get_scale_nums> of L<Music::Scales> into
an interval form usable by this module.

  use Music::Guidonian 'intervalize_scale_nums';
  use Music::Scales 'get_scale_nums';
  ...
    intervals => intervalize_scale_nums([get_scale_nums('major')])

=item B<_previous_pitch>

Internal. Looks up the previous pitch for a given pitch in a
I<sequence>.

=item B<_weight_choices>

Internal. Orders the choices such that pitches closest to the previous
pitch are returned first, as is common in melodic lines.

=item B<BUILD>

Internal. This is a L<Moo> utility function used by the
L</"CONSTRUCTOR">.

=back

=head1 BUGS

None known.

=head1 SEE ALSO

L<MIDI>, L<Music::AtonalUtil>, L<Music::Scales>, L<Music::VoiceGen>

"Musimathics: the mathematical foundations of music". Gareth Loy.
Mit Press. 2011.

=head1 COPYRIGHT AND LICENSE

Copyright 2021 Jeremy Mates

This program is distributed under the (Revised) BSD License:
L<https://opensource.org/licenses/BSD-3-Clause>

=cut
