use 5.008008;
use strict;
use warnings;

use List::Util ();
use Scalar::Util ();

package Hydrogen;

our $AUTHORITY = 'cpan:TOBYINK';
our $VERSION   = '0.004';

use Exporter::Shiny qw( croak );

sub croak {
    my ( $message, @args ) = @_;
    if ( @args ) {
        require Data::Dumper;
        local $Data::Dumper::Terse  = 1;
        local $Data::Dumper::Indent = 0;
        $message = sprintf $message, map {
            ref($_) ? Data::Dumper::Dumper($_) : defined($_) ? $_ : '(undef)'
        } @args;
    }
    require Carp;
    @_ = $message;
    goto \&Carp::croak;
}

1;

__END__

=pod

=encoding utf-8

=head1 NAME

Hydrogen - utilities for the simplest elements of Perl

=head1 SYNOPSIS

Normal version of the function:

    use feature 'say';
    use Hydrogen::HashRef { prefix => 'hhr_' }, qw( get set );
    
    my %hash;
    hhr_set( \%hash, Alice => 123 );
    hhr_set( \%hash, Bob   => 456 );
    
    say $hash{Alice};                ## ==> 123
    say hhr_get( \%hash, 'Bob' );    ## ==> 456

Version of the function which uses prototypes:

    use feature 'say';
    use Hydrogen::Hash { prefix => 'hh_' }, qw( get set );
    
    my %hash;
    hh_set( %hash, Alice => 123 );
    hh_set( %hash, Bob   => 456 );
    
    say $hash{Alice};                ## ==> 123
    say hh_get( %hash, 'Bob' );      ## ==> 456

Currying:

    use feature 'say';
    use Hydrogen::Curry::HashRef qw( curry_get curry_set );
    
    my %hash;
    my $setter = curry_set( \%hash );
    my $getter = curry_get( \%hash );
    
    $setter->( Alice => 123 );
    $setter->( Bob   => 456 );
    
    say $hash{Alice};                ## ==> 123
    say $getter->( 'Bob' );          ## ==> 456

=head1 DESCRIPTION

L<Hydrogen> provides a standard library for doing really simple things in
Perl. And I mean I<really> simple things.

Things which are often Perl builtin functions, operators, and even just
part of Perl syntax like accessing keys within hashes.

What is the point in having functions to do these simple things? Well, you
can make a coderef pointing to C<< \&Hydrogen::Number::add >> but you can't
make a coderef pointing to Perl's C<< += >> operator!

=head1 THE HYDROGEN LIBRARY

=over

=item *

L<Hydrogen::Array>

=item *

L<Hydrogen::ArrayRef>

=item *

L<Hydrogen::Bool>

=item *

L<Hydrogen::Code>

=item *

L<Hydrogen::CodeRef>

=item *

L<Hydrogen::Counter>

=item *

L<Hydrogen::Hash>

=item *

L<Hydrogen::HashRef>

=item *

L<Hydrogen::Number>

=item *

L<Hydrogen::Scalar>

=item *

L<Hydrogen::String>

=item *

L<Hydrogen::Curry::ArrayRef>

=item *

L<Hydrogen::Curry::Bool>

=item *

L<Hydrogen::Curry::CodeRef>

=item *

L<Hydrogen::Curry::Counter>

=item *

L<Hydrogen::Curry::HashRef>

=item *

L<Hydrogen::Curry::Number>

=item *

L<Hydrogen::Curry::Scalar>

=item *

L<Hydrogen::Curry::String>

=back

=head1 BUGS

Please report any bugs to
L<http://github.com/tobyink/p5-hydrogen/issues>.

=head1 SEE ALSO

This standard library is autogenerated from L<Sub::HandlesVia> which provides
the same functionality as methods which objects can delegate to attributes.

=head1 AUTHOR

Toby Inkster E<lt>tobyink@cpan.orgE<gt>.

=head1 COPYRIGHT AND LICENCE

This software is copyright (c) 2022 by Toby Inkster.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 DISCLAIMER OF WARRANTIES

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

