package VM::HetznerCloud::Schema;

# ABSTRACT: OpenAPI schema for the cloud API

use v5.24;

use Mojo::Base -strict, -signatures;

use JSON::Validator;
use JSON::Validator::Formats;
use List::Util qw(uniq);
use Mojo::JSON qw(decode_json);
use Mojo::Loader qw(data_section);
use Mojo::Util qw(camelize);

use constant IV_SIZE => eval 'require Config;$Config::Config{ivsize}'; ## no critic

our $VERSION = '0.0.1'; # VERSION

sub validate ( $class, $operation, $params = {} ) {
    my ($spec, $validator) = _get_params($operation);

    my %check_params = map {
        my $camelized = camelize $_;
        $camelized =~ s{Id$}{ID};

        $spec->{param_names}->{$camelized} ?
            ($camelized => $params->{$_}) :
            ($_ => $params->{$_});
    } keys $params->%*;

    my @errors = $validator->validate(
        \%check_params,
    );

    if ( @errors ) {
        return ( undef, @errors );
    }

    my %request_params;

    for my $param_name ( sort keys $spec->{param_names}->%* ) {
        my $key = $spec->{param_names}->{$param_name};

        next if !$check_params{$param_name};

        $request_params{$key}->{$param_name} = $check_params{$param_name};
    }

    return \%request_params;
}

sub _get_subnmame ( $parts, $method ) {
    my $all          = ( $parts->@* && $parts->[-1] =~ m{\{} ) ? 0 : 1;
    my @static_parts = grep { $_ !~ m/\{/ } $parts->@*;

    my $suffix = @static_parts ? ( join '_', '', @static_parts ) : '';

    if ( $all ) {
        return 'list' . $suffix   if $method eq 'get';
        return 'create' . $suffix if $method eq 'post';
    }

    return $method . $suffix;
}

sub _get_params ($operation) {
    state %operation_params;

    my $op_data = $operation_params{$operation};
    return $op_data->@* if $op_data;

    my $api_spec = data_section(__PACKAGE__, 'openapi.json');
    $api_spec    =~ s{/components/schemas}{}g;

    my $data    = decode_json( $api_spec );
    my $schemas = $data->{components}->{schemas};

    my %paths = $data->{paths}->%*;

    my $op;

    for my $path ( keys %paths ) {

        METHOD:
        for my $method_name ( keys $paths{$path}->%* ) {
            next METHOD if 'HASH' ne ref $paths{$path}->{$method_name};

            my $method = $paths{$path}->{$method_name};
            if ( $method->{operationId} && $method->{operationId} eq $operation ) {
                $op = $method;
            }
        }
    }

    return {} if !$op;

    my $params = $op->{parameters};

    my %properties;
    my @required;
    my %param_names;

    PARAM:
    for my $param ( $params->@* ) {
        next PARAM if $param->{name} eq 'Auth-API-Token';

        my $name = $param->{name};

        if ( $param->{required} ) {
            push @required, $name;
        }

        $param_names{$name} = $param->{in};
        $properties{$name}  = $param;
    }

    my ($content_type, $body_required);
    if ( $op->{requestBody} ) {
       my $body       = $op->{requestBody}->{content};
       $body_required = $op->{requestBody}->{required};

       ($content_type) = sort keys $body->%*;

       if ( $content_type eq 'application/json' ) {
           my $schema = $body->{$content_type}->{schema};
           my $prop   = $schema->{properties} || {};
           for my $property ( keys $prop->%* ) {
               $properties{$property}  = delete $prop->{$property};
               $param_names{$property} = 'body';
           }

           if ( $schema->{'$ref'} ) {  # '$ref' is not a typo. The key is named '$ref'!
               $properties{'$ref'} = $schema->{'$ref'};
           }
       }
       elsif ( $content_type eq 'text/plain' ) {
           $properties{text}  = { type => "string" };
           $param_names{text} = 'body';
       }
    }

    my $spec = {
        type          => 'object',
        required      => \@required,
        body_required => $body_required,
        properties    => \%properties,
        param_names   => \%param_names,
        content_type  => $content_type,
        $schemas->%*,
    };

    my $validator = JSON::Validator->new->schema($spec);

    $validator->formats->{uint32} = sub {
        my ($sub) = JSON::Validator::Formats->can('_match_number');

        $sub->( unint32 => $_[0], 'L' );
    };

    $validator->formats->{uint64} = sub {
        my ($sub) = JSON::Validator::Formats->can('_match_number');

        $sub->( unint32 => $_[0], IV_SIZE >= 8 ? 'Q' : '' );
    };

    if ( $spec->{properties}->{'$ref'} ) {
        my @ref = $spec->{properties}->{'$ref'};
        while ( my $ref = shift @ref ) {
            $ref =~ s{^#}{};

            my $data = $validator->get( $ref );
            if ( $data->{properties} ) {
                for my $property ( keys $data->{properties}->%* ) {
                    next if $data->{properties}->{$property}->{readOnly};

                    $spec->{param_names}->{$property} = 'body';
                }
            }

            if ( $data->{allOf} ) {
                push @ref, map{ $_->{'$ref'} } $data->{allOf}->@*;
                if ( $data->{required} ) {
                    push $spec->{required}->@*, $data->{required}->@*;
                }
            }
        }
    }

    $spec->{required}->@* = uniq $spec->{required}->@*;

    $operation_params{$operation} = [ $spec, $validator ];
    return $operation_params{$operation}->@*;
}

1;

=pod

=encoding UTF-8

=head1 NAME

VM::HetznerCloud::Schema - OpenAPI schema for the cloud API

=head1 VERSION

version 0.0.1

=head1 SYNOPSIS

=head1 METHODS

=head2 validate

=head1 AUTHOR

Renee Baecker <reneeb@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Renee Baecker.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut

__DATA__
@@ paths.json
{
   "/actions" : {
      "get" : {
         "description" : "Returns all Action objects. You can `sort` the results by using the sort URI parameter, and filter them with the `status` parameter.",
         "parameters" : [
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified IDs",
               "in" : "query",
               "name" : "id",
               "required" : false,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions",
         "tags" : [
            "Actions"
         ]
      }
   },
   "/actions/{id}" : {
      "get" : {
         "description" : "Returns a specific Action object.",
         "parameters" : [
            {
               "description" : "ID of the Resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key in the reply has this structure"
            }
         },
         "summary" : "Get an Action",
         "tags" : [
            "Actions"
         ]
      }
   },
   "/certificates" : {
      "get" : {
         "description" : "Returns all Certificate objects.",
         "parameters" : [
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "name",
                     "name:asc",
                     "name:desc",
                     "created",
                     "created:asc",
                     "created:desc"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by their name. The response will only contain the resources matching the specified name",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by labels. The response will only contain resources matching the label selector.",
               "in" : "query",
               "name" : "label_selector",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times. The response will only contain Certificates matching the type.",
               "in" : "query",
               "name" : "type",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "uploaded",
                     "managed"
                  ],
                  "title" : "ParameterType",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "certificates" : [
                           {
                              "certificate" : "-----BEGIN CERTIFICATE-----\n...",
                              "created" : "2019-01-08T12:10:00+00:00",
                              "domain_names" : [
                                 "example.com",
                                 "webmail.example.com",
                                 "www.example.com"
                              ],
                              "fingerprint" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                              "id" : 897,
                              "labels" : {
                                 "env" : "dev"
                              },
                              "name" : "my website cert",
                              "not_valid_after" : "2019-07-08T09:59:59+00:00",
                              "not_valid_before" : "2019-01-08T10:00:00+00:00",
                              "status" : null,
                              "type" : "uploaded",
                              "used_by" : [
                                 {
                                    "id" : 4711,
                                    "type" : "load_balancer"
                                 }
                              ]
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "certificates" : {
                              "items" : {
                                 "properties" : {
                                    "certificate" : {
                                       "description" : "Certificate and chain in PEM format, in order so that each record directly certifies the one preceding",
                                       "example" : "-----BEGIN CERTIFICATE-----\n...",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "created" : {
                                       "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "domain_names" : {
                                       "description" : "Domains and subdomains covered by the Certificate",
                                       "example" : [
                                          "example.com",
                                          "webmail.example.com",
                                          "www.example.com"
                                       ],
                                       "items" : {
                                          "type" : "string"
                                       },
                                       "type" : "array"
                                    },
                                    "fingerprint" : {
                                       "description" : "SHA256 fingerprint of the Certificate",
                                       "example" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "labels" : {
                                       "additionalProperties" : {
                                          "type" : "string"
                                       },
                                       "description" : "User-defined labels (key-value pairs)",
                                       "type" : "object"
                                    },
                                    "name" : {
                                       "description" : "Name of the Resource. Must be unique per Project.",
                                       "example" : "my-resource",
                                       "type" : "string"
                                    },
                                    "not_valid_after" : {
                                       "description" : "Point in time when the Certificate stops being valid (in ISO-8601 format)",
                                       "example" : "2019-07-08T09:59:59+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "not_valid_before" : {
                                       "description" : "Point in time when the Certificate becomes valid (in ISO-8601 format)",
                                       "example" : "2019-01-08T10:00:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Current status of a type `managed` Certificate, always *null* for type `uploaded` Certificates",
                                       "nullable" : true,
                                       "properties" : {
                                          "error" : {
                                             "description" : "If issuance or renewal reports `failed`, this property contains information about what happened",
                                             "example" : null,
                                             "nullable" : true,
                                             "properties" : {
                                                "code" : {
                                                   "type" : "string"
                                                },
                                                "message" : {
                                                   "type" : "string"
                                                }
                                             },
                                             "type" : "object"
                                          },
                                          "issuance" : {
                                             "description" : "Status of the issuance process of the Certificate",
                                             "enum" : [
                                                "pending",
                                                "completed",
                                                "failed"
                                             ],
                                             "example" : "valid",
                                             "type" : "string"
                                          },
                                          "renewal" : {
                                             "description" : "Status of the renewal process of the Certificate.",
                                             "enum" : [
                                                "scheduled",
                                                "pending",
                                                "failed",
                                                "unavailable"
                                             ],
                                             "example" : "scheduled",
                                             "type" : "string"
                                          }
                                       },
                                       "type" : "object"
                                    },
                                    "type" : {
                                       "description" : "Type of the Certificate",
                                       "enum" : [
                                          "uploaded",
                                          "managed"
                                       ],
                                       "example" : "uploaded",
                                       "type" : "string"
                                    },
                                    "used_by" : {
                                       "description" : "Resources currently using the Certificate",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of resource referenced",
                                                "example" : 4711,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "load_balancer",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "labels",
                                    "certificate",
                                    "created",
                                    "not_valid_before",
                                    "not_valid_after",
                                    "domain_names",
                                    "fingerprint",
                                    "used_by"
                                 ],
                                 "title" : "Certificate",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "certificates"
                        ],
                        "title" : "CertificatesResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `certificates` key contains an array of Certificate objects"
            }
         },
         "summary" : "Get all Certificates",
         "tags" : [
            "Certificates"
         ]
      },
      "post" : {
         "description" : "Creates a new Certificate.\n\nThe default type **uploaded** allows for uploading your existing `certificate` and `private_key` in PEM format. You have to monitor its expiration date and handle renewal yourself.\n\nIn contrast, type **managed** requests a new Certificate from *Let's Encrypt* for the specified `domain_names`. Only domains managed by *Hetzner DNS* are supported. We handle renewal and timely alert the project owner via email if problems occur.\n\nFor type `managed` Certificates the `action` key of the response contains the Action that allows for tracking the issuance process. For type `uploaded` Certificates the `action` is always null.\n",
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "examples" : {
                     "managed" : {
                        "summary" : "Creating a type `managed` Certificate",
                        "value" : {
                           "domain_names" : [
                              "example.com",
                              "webmail.example.com",
                              "www.example.com"
                           ],
                           "name" : "my website cert",
                           "type" : "managed"
                        }
                     },
                     "uploaded" : {
                        "summary" : "Creating a type `uploaded` Certificate",
                        "value" : {
                           "certificate" : "-----BEGIN CERTIFICATE-----\n...",
                           "name" : "my website cert",
                           "private_key" : "-----BEGIN PRIVATE KEY-----\n...",
                           "type" : "uploaded"
                        }
                     }
                  },
                  "schema" : {
                     "properties" : {
                        "certificate" : {
                           "description" : "Certificate and chain in PEM format, in order so that each record directly certifies the one preceding. Required for type `uploaded` Certificates.",
                           "example" : "-----BEGIN CERTIFICATE-----\n...",
                           "type" : "string"
                        },
                        "domain_names" : {
                           "description" : "Domains and subdomains that should be contained in the Certificate issued by *Let's Encrypt*. Required for type `managed` Certificates.",
                           "example" : null,
                           "items" : {
                              "type" : "string"
                           },
                           "type" : "array"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "Name of the Certificate",
                           "example" : "my website cert",
                           "type" : "string"
                        },
                        "private_key" : {
                           "description" : "Certificate key in PEM format. Required for type `uploaded` Certificates.",
                           "example" : "-----BEGIN PRIVATE KEY-----\n...",
                           "type" : "string"
                        },
                        "type" : {
                           "description" : "Choose between uploading a Certificate in PEM format or requesting a managed *Let's Encrypt* Certificate. If omitted defaults to `uploaded`.",
                           "enum" : [
                              "uploaded",
                              "managed"
                           ],
                           "example" : "uploaded",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "name"
                     ],
                     "title" : "CreateCertificateRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "examples" : {
                        "managed" : {
                           "summary" : "Response when creating a type `managed` Certificate",
                           "value" : {
                              "action" : {
                                 "command" : "create_certificate",
                                 "error" : {
                                    "code" : "action_failed",
                                    "message" : "Action failed"
                                 },
                                 "finished" : null,
                                 "id" : 13,
                                 "progress" : 0,
                                 "resources" : [
                                    {
                                       "id" : 879,
                                       "type" : "certificate"
                                    }
                                 ],
                                 "started" : "2019-01-08T12:10:00+00:00",
                                 "status" : "running"
                              },
                              "certificate" : {
                                 "certificate" : null,
                                 "created" : "2019-01-08T12:10:00+00:00",
                                 "domain_names" : [
                                    "example.com",
                                    "webmail.example.com",
                                    "www.example.com"
                                 ],
                                 "fingerprint" : null,
                                 "id" : 897,
                                 "labels" : {
                                    "env" : "dev"
                                 },
                                 "name" : "my website cert",
                                 "not_valid_after" : null,
                                 "not_valid_before" : null,
                                 "status" : {
                                    "error" : null,
                                    "issuance" : "pending",
                                    "renewal" : "unavailable"
                                 },
                                 "type" : "managed",
                                 "used_by" : [
                                    {
                                       "id" : 4711,
                                       "type" : "load_balancer"
                                    }
                                 ]
                              }
                           }
                        },
                        "uploaded" : {
                           "summary" : "Response when creating a type `uploaded` Certificate",
                           "value" : {
                              "action" : null,
                              "certificate" : {
                                 "certificate" : "-----BEGIN CERTIFICATE-----\n...",
                                 "created" : "2019-01-08T12:10:00+00:00",
                                 "domain_names" : [
                                    "example.com",
                                    "webmail.example.com",
                                    "www.example.com"
                                 ],
                                 "fingerprint" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                                 "id" : 897,
                                 "labels" : {
                                    "env" : "dev"
                                 },
                                 "name" : "my website cert",
                                 "not_valid_after" : "2019-07-08T09:59:59+00:00",
                                 "not_valid_before" : "2019-01-08T10:00:00+00:00",
                                 "status" : null,
                                 "type" : "uploaded",
                                 "used_by" : [
                                    {
                                       "id" : 4711,
                                       "type" : "load_balancer"
                                    }
                                 ]
                              }
                           }
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "nullable" : true,
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "NullableAction",
                              "type" : "object"
                           },
                           "certificate" : {
                              "properties" : {
                                 "certificate" : {
                                    "description" : "Certificate and chain in PEM format, in order so that each record directly certifies the one preceding",
                                    "example" : "-----BEGIN CERTIFICATE-----\n...",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "domain_names" : {
                                    "description" : "Domains and subdomains covered by the Certificate",
                                    "example" : [
                                       "example.com",
                                       "webmail.example.com",
                                       "www.example.com"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 },
                                 "fingerprint" : {
                                    "description" : "SHA256 fingerprint of the Certificate",
                                    "example" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "not_valid_after" : {
                                    "description" : "Point in time when the Certificate stops being valid (in ISO-8601 format)",
                                    "example" : "2019-07-08T09:59:59+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "not_valid_before" : {
                                    "description" : "Point in time when the Certificate becomes valid (in ISO-8601 format)",
                                    "example" : "2019-01-08T10:00:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Current status of a type `managed` Certificate, always *null* for type `uploaded` Certificates",
                                    "nullable" : true,
                                    "properties" : {
                                       "error" : {
                                          "description" : "If issuance or renewal reports `failed`, this property contains information about what happened",
                                          "example" : null,
                                          "nullable" : true,
                                          "properties" : {
                                             "code" : {
                                                "type" : "string"
                                             },
                                             "message" : {
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "issuance" : {
                                          "description" : "Status of the issuance process of the Certificate",
                                          "enum" : [
                                             "pending",
                                             "completed",
                                             "failed"
                                          ],
                                          "example" : "valid",
                                          "type" : "string"
                                       },
                                       "renewal" : {
                                          "description" : "Status of the renewal process of the Certificate.",
                                          "enum" : [
                                             "scheduled",
                                             "pending",
                                             "failed",
                                             "unavailable"
                                          ],
                                          "example" : "scheduled",
                                          "type" : "string"
                                       }
                                    },
                                    "type" : "object"
                                 },
                                 "type" : {
                                    "description" : "Type of the Certificate",
                                    "enum" : [
                                       "uploaded",
                                       "managed"
                                    ],
                                    "example" : "uploaded",
                                    "type" : "string"
                                 },
                                 "used_by" : {
                                    "description" : "Resources currently using the Certificate",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of resource referenced",
                                             "example" : 4711,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "load_balancer",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "labels",
                                 "certificate",
                                 "created",
                                 "not_valid_before",
                                 "not_valid_after",
                                 "domain_names",
                                 "fingerprint",
                                 "used_by"
                              ],
                              "title" : "Certificate",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "certificate"
                        ],
                        "title" : "CreateCertificateResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `certificate` key contains the Certificate that was just created. For type `managed` Certificates the `action` key contains the Action that allows for tracking the issuance process. For type `uploaded` Certificates the `action` is always null."
            }
         },
         "summary" : "Create a Certificate",
         "tags" : [
            "Certificates"
         ]
      }
   },
   "/certificates/{id}" : {
      "delete" : {
         "description" : "Deletes a Certificate.",
         "parameters" : [
            {
               "description" : "ID of the resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "204" : {
               "description" : "Certificate deleted"
            }
         },
         "summary" : "Delete a Certificate",
         "tags" : [
            "Certificates"
         ]
      },
      "get" : {
         "description" : "Gets a specific Certificate object.",
         "parameters" : [
            {
               "description" : "ID of the resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "certificate" : {
                           "certificate" : "-----BEGIN CERTIFICATE-----\n...",
                           "created" : "2019-01-08T12:10:00+00:00",
                           "domain_names" : [
                              "example.com",
                              "webmail.example.com",
                              "www.example.com"
                           ],
                           "fingerprint" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                           "id" : 897,
                           "labels" : {
                              "env" : "dev"
                           },
                           "name" : "my website cert",
                           "not_valid_after" : "2019-07-08T09:59:59+00:00",
                           "not_valid_before" : "2019-01-08T10:00:00+00:00",
                           "status" : null,
                           "type" : "uploaded",
                           "used_by" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ]
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "certificate" : {
                              "properties" : {
                                 "certificate" : {
                                    "description" : "Certificate and chain in PEM format, in order so that each record directly certifies the one preceding",
                                    "example" : "-----BEGIN CERTIFICATE-----\n...",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "domain_names" : {
                                    "description" : "Domains and subdomains covered by the Certificate",
                                    "example" : [
                                       "example.com",
                                       "webmail.example.com",
                                       "www.example.com"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 },
                                 "fingerprint" : {
                                    "description" : "SHA256 fingerprint of the Certificate",
                                    "example" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "not_valid_after" : {
                                    "description" : "Point in time when the Certificate stops being valid (in ISO-8601 format)",
                                    "example" : "2019-07-08T09:59:59+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "not_valid_before" : {
                                    "description" : "Point in time when the Certificate becomes valid (in ISO-8601 format)",
                                    "example" : "2019-01-08T10:00:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Current status of a type `managed` Certificate, always *null* for type `uploaded` Certificates",
                                    "nullable" : true,
                                    "properties" : {
                                       "error" : {
                                          "description" : "If issuance or renewal reports `failed`, this property contains information about what happened",
                                          "example" : null,
                                          "nullable" : true,
                                          "properties" : {
                                             "code" : {
                                                "type" : "string"
                                             },
                                             "message" : {
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "issuance" : {
                                          "description" : "Status of the issuance process of the Certificate",
                                          "enum" : [
                                             "pending",
                                             "completed",
                                             "failed"
                                          ],
                                          "example" : "valid",
                                          "type" : "string"
                                       },
                                       "renewal" : {
                                          "description" : "Status of the renewal process of the Certificate.",
                                          "enum" : [
                                             "scheduled",
                                             "pending",
                                             "failed",
                                             "unavailable"
                                          ],
                                          "example" : "scheduled",
                                          "type" : "string"
                                       }
                                    },
                                    "type" : "object"
                                 },
                                 "type" : {
                                    "description" : "Type of the Certificate",
                                    "enum" : [
                                       "uploaded",
                                       "managed"
                                    ],
                                    "example" : "uploaded",
                                    "type" : "string"
                                 },
                                 "used_by" : {
                                    "description" : "Resources currently using the Certificate",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of resource referenced",
                                             "example" : 4711,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "load_balancer",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "labels",
                                 "certificate",
                                 "created",
                                 "not_valid_before",
                                 "not_valid_after",
                                 "domain_names",
                                 "fingerprint",
                                 "used_by"
                              ],
                              "title" : "Certificate",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "certificate"
                        ],
                        "title" : "CertificateResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `certificate` key contains a Certificate object"
            }
         },
         "summary" : "Get a Certificate",
         "tags" : [
            "Certificates"
         ]
      },
      "put" : {
         "description" : "Updates the Certificate properties.\n\nNote that when updating labels, the Certificate’s current set of labels will be replaced with the labels provided in the request body. So, for example, if you want to add a new label, you have to provide all existing labels plus the new label in the request body.\n\nNote: if the Certificate object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "example" : {
                              "labelkey" : "value"
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "New Certificate name",
                           "example" : "my website cert",
                           "type" : "string"
                        }
                     },
                     "title" : "UpdateCertificateRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "certificate" : {
                           "certificate" : "-----BEGIN CERTIFICATE-----\n...",
                           "created" : "2019-01-08T12:10:00+00:00",
                           "domain_names" : [
                              "example.com",
                              "webmail.example.com",
                              "www.example.com"
                           ],
                           "fingerprint" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                           "id" : 897,
                           "labels" : {
                              "labelkey" : "value"
                           },
                           "name" : "my website cert",
                           "not_valid_after" : "2019-07-08T09:59:59+00:00",
                           "not_valid_before" : "2019-01-08T10:00:00+00:00",
                           "status" : null,
                           "type" : "uploaded",
                           "used_by" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ]
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "certificate" : {
                              "properties" : {
                                 "certificate" : {
                                    "description" : "Certificate and chain in PEM format, in order so that each record directly certifies the one preceding",
                                    "example" : "-----BEGIN CERTIFICATE-----\n...",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "domain_names" : {
                                    "description" : "Domains and subdomains covered by the Certificate",
                                    "example" : [
                                       "example.com",
                                       "webmail.example.com",
                                       "www.example.com"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 },
                                 "fingerprint" : {
                                    "description" : "SHA256 fingerprint of the Certificate",
                                    "example" : "03:c7:55:9b:2a:d1:04:17:09:f6:d0:7f:18:34:63:d4:3e:5f",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "not_valid_after" : {
                                    "description" : "Point in time when the Certificate stops being valid (in ISO-8601 format)",
                                    "example" : "2019-07-08T09:59:59+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "not_valid_before" : {
                                    "description" : "Point in time when the Certificate becomes valid (in ISO-8601 format)",
                                    "example" : "2019-01-08T10:00:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Current status of a type `managed` Certificate, always *null* for type `uploaded` Certificates",
                                    "nullable" : true,
                                    "properties" : {
                                       "error" : {
                                          "description" : "If issuance or renewal reports `failed`, this property contains information about what happened",
                                          "example" : null,
                                          "nullable" : true,
                                          "properties" : {
                                             "code" : {
                                                "type" : "string"
                                             },
                                             "message" : {
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "issuance" : {
                                          "description" : "Status of the issuance process of the Certificate",
                                          "enum" : [
                                             "pending",
                                             "completed",
                                             "failed"
                                          ],
                                          "example" : "valid",
                                          "type" : "string"
                                       },
                                       "renewal" : {
                                          "description" : "Status of the renewal process of the Certificate.",
                                          "enum" : [
                                             "scheduled",
                                             "pending",
                                             "failed",
                                             "unavailable"
                                          ],
                                          "example" : "scheduled",
                                          "type" : "string"
                                       }
                                    },
                                    "type" : "object"
                                 },
                                 "type" : {
                                    "description" : "Type of the Certificate",
                                    "enum" : [
                                       "uploaded",
                                       "managed"
                                    ],
                                    "example" : "uploaded",
                                    "type" : "string"
                                 },
                                 "used_by" : {
                                    "description" : "Resources currently using the Certificate",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of resource referenced",
                                             "example" : 4711,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "load_balancer",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "labels",
                                 "certificate",
                                 "created",
                                 "not_valid_before",
                                 "not_valid_after",
                                 "domain_names",
                                 "fingerprint",
                                 "used_by"
                              ],
                              "title" : "Certificate",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "certificate"
                        ],
                        "title" : "CertificateResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `certificate` key contains the Certificate that was just updated"
            }
         },
         "summary" : "Update a Certificate",
         "tags" : [
            "Certificates"
         ]
      }
   },
   "/certificates/{id}/actions" : {
      "get" : {
         "description" : "Returns all Action objects for a Certificate. You can sort the results by using the `sort` URI parameter, and filter them with the `status` parameter.\n\nOnly type `managed` Certificates can have Actions. For type `uploaded` Certificates the `actions` key will always contain an empty array.\n",
         "parameters" : [
            {
               "description" : "ID of the Resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "issue_certificate",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2021-01-30T23:57:00+00:00",
                              "id" : 14,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 896,
                                    "type" : "certificate"
                                 }
                              ],
                              "started" : "2021-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions for a Certificate",
         "tags" : [
            "Certificate Actions"
         ]
      }
   },
   "/certificates/{id}/actions/retry" : {
      "post" : {
         "description" : "Retry a failed Certificate issuance or renewal.\n\nOnly applicable if the type of the Certificate is `managed` and the issuance or renewal status is `failed`.\n\n#### Call specific error codes\n\n| Code                                                    | Description                                                               |\n|---------------------------------------------------------|---------------------------------------------------------------------------|\n| `caa_record_does_not_allow_ca`                          | CAA record does not allow certificate authority                           |\n| `ca_dns_validation_failed`                              | Certificate Authority: DNS validation failed                              |\n| `ca_too_many_authorizations_failed_recently`            | Certificate Authority: Too many authorizations failed recently            |\n| `ca_too_many_certificates_issued_for_registered_domain` | Certificate Authority: Too many certificates issued for registered domain |\n| `ca_too_many_duplicate_certificates`                    | Certificate Authority: Too many duplicate certificates                    |\n| `could_not_verify_domain_delegated_to_zone`             | Could not verify domain delegated to zone                                 |\n| `dns_zone_not_found`                                    | DNS zone not found                                                        |\n| `dns_zone_is_secondary_zone`                            | DNS zone is a secondary zone                                              |\n",
         "parameters" : [
            {
               "description" : "ID of the Certificate",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "issue_certificate",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2021-01-30T23:57:00+00:00",
                           "id" : 14,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 896,
                                 "type" : "certificate"
                              }
                           ],
                           "started" : "2021-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the resulting Action"
            }
         },
         "summary" : "Retry Issuance or Renewal",
         "tags" : [
            "Certificate Actions"
         ]
      }
   },
   "/certificates/{id}/actions/{action_id}" : {
      "get" : {
         "description" : "Returns a specific Action for a Certificate. Only type `managed` Certificates have Actions.",
         "parameters" : [
            {
               "description" : "ID of the Certificate",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "ID of the Action",
               "in" : "path",
               "name" : "action_id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "issue_certificate",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2021-01-30T23:57:00+00:00",
                           "id" : 14,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 896,
                                 "type" : "certificate"
                              }
                           ],
                           "started" : "2021-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the Certificate Action"
            }
         },
         "summary" : "Get an Action for a Certificate",
         "tags" : [
            "Certificate Actions"
         ]
      }
   },
   "/datacenters" : {
      "get" : {
         "description" : "Returns all Datacenter objects.",
         "parameters" : [
            {
               "description" : "Can be used to filter Datacenters by their name. The response will only contain the Datacenter matching the specified name. When the name does not match the Datacenter name format, an `invalid_input` error is returned.",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "datacenters" : {
                              "items" : {
                                 "properties" : {
                                    "description" : {
                                       "description" : "Description of the Datacenter",
                                       "example" : "Falkenstein DC Park 8",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "location" : {
                                       "properties" : {
                                          "city" : {
                                             "description" : "City the Location is closest to",
                                             "example" : "Falkenstein",
                                             "type" : "string"
                                          },
                                          "country" : {
                                             "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                             "example" : "DE",
                                             "type" : "string"
                                          },
                                          "description" : {
                                             "description" : "Description of the Location",
                                             "example" : "Falkenstein DC Park 1",
                                             "type" : "string"
                                          },
                                          "id" : {
                                             "description" : "ID of the Location",
                                             "example" : 1,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "latitude" : {
                                             "description" : "Latitude of the city closest to the Location",
                                             "example" : 50.47612,
                                             "type" : "number"
                                          },
                                          "longitude" : {
                                             "description" : "Longitude of the city closest to the Location",
                                             "example" : 12.370071,
                                             "type" : "number"
                                          },
                                          "name" : {
                                             "description" : "Unique identifier of the Location",
                                             "example" : "fsn1",
                                             "type" : "string"
                                          },
                                          "network_zone" : {
                                             "description" : "Name of network zone this Location resides in",
                                             "example" : "eu-central",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "name",
                                          "description",
                                          "country",
                                          "city",
                                          "latitude",
                                          "longitude",
                                          "network_zone"
                                       ],
                                       "type" : "object"
                                    },
                                    "name" : {
                                       "description" : "Unique identifier of the Datacenter",
                                       "example" : "fsn1-dc8",
                                       "type" : "string"
                                    },
                                    "server_types" : {
                                       "description" : "The Server types the Datacenter can handle",
                                       "properties" : {
                                          "available" : {
                                             "description" : "IDs of Server types that are supported and for which the Datacenter has enough resources left",
                                             "example" : [
                                                1,
                                                2,
                                                3
                                             ],
                                             "items" : {
                                                "type" : "number"
                                             },
                                             "type" : "array"
                                          },
                                          "available_for_migration" : {
                                             "description" : "IDs of Server types that are supported and for which the Datacenter has enough resources left",
                                             "example" : [
                                                1,
                                                2,
                                                3
                                             ],
                                             "items" : {
                                                "type" : "number"
                                             },
                                             "type" : "array"
                                          },
                                          "supported" : {
                                             "description" : "IDs of Server types that are supported in the Datacenter",
                                             "example" : [
                                                1,
                                                2,
                                                3
                                             ],
                                             "items" : {
                                                "type" : "number"
                                             },
                                             "type" : "array"
                                          }
                                       },
                                       "required" : [
                                          "supported",
                                          "available",
                                          "available_for_migration"
                                       ],
                                       "type" : "object"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "description",
                                    "location",
                                    "server_types"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "recommendation" : {
                              "description" : "The Datacenter which is recommended to be used to create new Servers.",
                              "example" : 1,
                              "type" : "number"
                           }
                        },
                        "required" : [
                           "datacenters",
                           "recommendation"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The reply contains the `datacenters` and `recommendation` keys"
            }
         },
         "summary" : "Get all Datacenters",
         "tags" : [
            "Datacenters"
         ]
      }
   },
   "/datacenters/{id}" : {
      "get" : {
         "description" : "Returns a specific Datacenter object.",
         "parameters" : [
            {
               "description" : "ID of Datacenter",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "datacenter" : {
                              "properties" : {
                                 "description" : {
                                    "description" : "Description of the Datacenter",
                                    "example" : "Falkenstein DC Park 8",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "location" : {
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Unique identifier of the Datacenter",
                                    "example" : "fsn1-dc8",
                                    "type" : "string"
                                 },
                                 "server_types" : {
                                    "description" : "The Server types the Datacenter can handle",
                                    "properties" : {
                                       "available" : {
                                          "description" : "IDs of Server types that are supported and for which the Datacenter has enough resources left",
                                          "example" : [
                                             1,
                                             2,
                                             3
                                          ],
                                          "items" : {
                                             "type" : "number"
                                          },
                                          "type" : "array"
                                       },
                                       "available_for_migration" : {
                                          "description" : "IDs of Server types that are supported and for which the Datacenter has enough resources left",
                                          "example" : [
                                             1,
                                             2,
                                             3
                                          ],
                                          "items" : {
                                             "type" : "number"
                                          },
                                          "type" : "array"
                                       },
                                       "supported" : {
                                          "description" : "IDs of Server types that are supported in the Datacenter",
                                          "example" : [
                                             1,
                                             2,
                                             3
                                          ],
                                          "items" : {
                                             "type" : "number"
                                          },
                                          "type" : "array"
                                       }
                                    },
                                    "required" : [
                                       "supported",
                                       "available",
                                       "available_for_migration"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "location",
                                 "server_types"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "datacenter"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `datacenter` key in the reply contains a Datacenter object with this structure"
            }
         },
         "summary" : "Get a Datacenter",
         "tags" : [
            "Datacenters"
         ]
      }
   },
   "/firewalls" : {
      "get" : {
         "description" : "Returns all Firewall objects.",
         "parameters" : [
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "name",
                     "name:asc",
                     "name:desc",
                     "created",
                     "created:asc",
                     "created:desc"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by their name. The response will only contain the resources matching the specified name",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by labels. The response will only contain resources matching the label selector.",
               "in" : "query",
               "name" : "label_selector",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "firewalls" : {
                              "items" : {
                                 "properties" : {
                                    "applied_to" : {
                                       "items" : {
                                          "properties" : {
                                             "applied_to_resources" : {
                                                "items" : {
                                                   "properties" : {
                                                      "server" : {
                                                         "properties" : {
                                                            "id" : {
                                                               "description" : "ID of the Resource",
                                                               "example" : 42,
                                                               "format" : "int64",
                                                               "type" : "integer"
                                                            }
                                                         },
                                                         "required" : [
                                                            "id"
                                                         ],
                                                         "type" : "object"
                                                      },
                                                      "type" : {
                                                         "description" : "Type of resource referenced",
                                                         "enum" : [
                                                            "server"
                                                         ],
                                                         "example" : "server",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "type" : "object"
                                                },
                                                "type" : "array"
                                             },
                                             "label_selector" : {
                                                "properties" : {
                                                   "selector" : {
                                                      "description" : "Label selector",
                                                      "example" : "env=prod",
                                                      "type" : "string"
                                                   }
                                                },
                                                "required" : [
                                                   "selector"
                                                ],
                                                "type" : "object"
                                             },
                                             "server" : {
                                                "properties" : {
                                                   "id" : {
                                                      "description" : "ID of the Resource",
                                                      "example" : 42,
                                                      "format" : "int64",
                                                      "type" : "integer"
                                                   }
                                                },
                                                "required" : [
                                                   "id"
                                                ],
                                                "type" : "object"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "enum" : [
                                                   "server",
                                                   "label_selector"
                                                ],
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "created" : {
                                       "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "labels" : {
                                       "additionalProperties" : {
                                          "type" : "string"
                                       },
                                       "description" : "User-defined labels (key-value pairs)",
                                       "type" : "object"
                                    },
                                    "name" : {
                                       "description" : "Name of the Resource. Must be unique per Project.",
                                       "example" : "my-resource",
                                       "type" : "string"
                                    },
                                    "rules" : {
                                       "items" : {
                                          "properties" : {
                                             "description" : {
                                                "description" : "Description of the Rule",
                                                "maxLength" : 255,
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "destination_ips" : {
                                                "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                                "example" : [
                                                   "28.239.13.1/32",
                                                   "28.239.14.0/24",
                                                   "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                                ],
                                                "items" : {
                                                   "type" : "string"
                                                },
                                                "type" : "array"
                                             },
                                             "direction" : {
                                                "description" : "Select traffic direction on which rule should be applied. Use `source_ips` for direction `in` and `destination_ips` for direction `out`.",
                                                "enum" : [
                                                   "in",
                                                   "out"
                                                ],
                                                "type" : "string"
                                             },
                                             "port" : {
                                                "description" : "Port or port range to which traffic will be allowed, only applicable for protocols TCP and UDP. A port range can be specified by separating two ports with a dash, e.g `1024-5000`.",
                                                "example" : "80",
                                                "type" : "string"
                                             },
                                             "protocol" : {
                                                "description" : "Type of traffic to allow",
                                                "enum" : [
                                                   "tcp",
                                                   "udp",
                                                   "icmp",
                                                   "esp",
                                                   "gre"
                                                ],
                                                "type" : "string"
                                             },
                                             "source_ips" : {
                                                "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                                "example" : [
                                                   "28.239.13.1/32",
                                                   "28.239.14.0/24",
                                                   "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                                ],
                                                "items" : {
                                                   "type" : "string"
                                                },
                                                "type" : "array"
                                             }
                                          },
                                          "required" : [
                                             "direction",
                                             "protocol"
                                          ],
                                          "title" : "Rule",
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "created",
                                    "rules",
                                    "applied_to"
                                 ],
                                 "title" : "Firewall",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "firewalls"
                        ],
                        "title" : "FirewallsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `firewalls` key contains an array of Firewall objects"
            }
         },
         "summary" : "Get all Firewalls",
         "tags" : [
            "Firewalls"
         ]
      },
      "post" : {
         "description" : "Creates a new Firewall.\n\n#### Call specific error codes\n\n| Code                          | Description                                                   |\n|------------------------------ |-------------------------------------------------------------- |\n| `server_already_added`        | Server added more than one time to resource                   |\n| `incompatible_network_type`   | The Network type is incompatible for the given resource       |\n| `firewall_resource_not_found` | The resource the Firewall should be attached to was not found |\n",
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "examples" : {
                     "request" : {
                        "value" : {
                           "apply_to" : [
                              {
                                 "server" : {
                                    "id" : 42
                                 },
                                 "type" : "server"
                              }
                           ],
                           "labels" : {
                              "env" : "dev"
                           },
                           "name" : "Corporate Intranet Protection",
                           "rules" : [
                              {
                                 "description" : "Allow port 80",
                                 "direction" : "in",
                                 "port" : "80",
                                 "protocol" : "tcp",
                                 "source_ips" : [
                                    "28.239.13.1/32",
                                    "28.239.14.0/24",
                                    "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                 ]
                              }
                           ]
                        }
                     }
                  },
                  "schema" : {
                     "properties" : {
                        "apply_to" : {
                           "description" : "Resources the Firewall should be applied to after creation",
                           "items" : {
                              "properties" : {
                                 "label_selector" : {
                                    "description" : "Configuration for type LabelSelector, required if type is `label_selector`",
                                    "properties" : {
                                       "selector" : {
                                          "description" : "Label selector",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "selector"
                                    ],
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "Configuration for type Server, required if type is `server`",
                                    "properties" : {
                                       "id" : {
                                          "description" : "ID of the Server",
                                          "format" : "int64",
                                          "type" : "integer"
                                       }
                                    },
                                    "required" : [
                                       "id"
                                    ],
                                    "type" : "object"
                                 },
                                 "type" : {
                                    "description" : "Type of the resource",
                                    "enum" : [
                                       "server",
                                       "label_selector"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "type"
                              ],
                              "type" : "object"
                           },
                           "type" : "array"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "Name of the Firewall",
                           "example" : "Corporate Intranet Protection",
                           "type" : "string"
                        },
                        "rules" : {
                           "description" : "Array of rules",
                           "example" : [
                              {
                                 "direction" : "in",
                                 "port" : "80",
                                 "protocol" : "tcp",
                                 "source_ips" : [
                                    "28.239.13.1/32",
                                    "28.239.14.0/24",
                                    "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                 ]
                              }
                           ],
                           "items" : {
                              "properties" : {
                                 "description" : {
                                    "description" : "Description of the Rule",
                                    "maxLength" : 255,
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "destination_ips" : {
                                    "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                    "example" : [
                                       "28.239.13.1/32",
                                       "28.239.14.0/24",
                                       "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 },
                                 "direction" : {
                                    "description" : "Select traffic direction on which rule should be applied. Use `source_ips` for direction `in` and `destination_ips` for direction `out`.",
                                    "enum" : [
                                       "in",
                                       "out"
                                    ],
                                    "type" : "string"
                                 },
                                 "port" : {
                                    "description" : "Port or port range to which traffic will be allowed, only applicable for protocols TCP and UDP. A port range can be specified by separating two ports with a dash, e.g `1024-5000`.",
                                    "example" : "80",
                                    "type" : "string"
                                 },
                                 "protocol" : {
                                    "description" : "Type of traffic to allow",
                                    "enum" : [
                                       "tcp",
                                       "udp",
                                       "icmp",
                                       "esp",
                                       "gre"
                                    ],
                                    "type" : "string"
                                 },
                                 "source_ips" : {
                                    "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                    "example" : [
                                       "28.239.13.1/32",
                                       "28.239.14.0/24",
                                       "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "direction",
                                 "protocol"
                              ],
                              "title" : "Rule",
                              "type" : "object"
                           },
                           "type" : "array"
                        }
                     },
                     "required" : [
                        "name"
                     ],
                     "title" : "CreateFirewallRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "example" : [
                                 {
                                    "command" : "set_firewall_rules",
                                    "error" : {
                                       "code" : "action_failed",
                                       "message" : "Action failed"
                                    },
                                    "finished" : "2016-01-30T23:56:00+00:00",
                                    "id" : 13,
                                    "progress" : 100,
                                    "resources" : [
                                       {
                                          "id" : 38,
                                          "type" : "firewall"
                                       }
                                    ],
                                    "started" : "2016-01-30T23:55:00+00:00",
                                    "status" : "success"
                                 },
                                 {
                                    "command" : "apply_firewall",
                                    "error" : {
                                       "code" : "action_failed",
                                       "message" : "Action failed"
                                    },
                                    "finished" : "2016-01-30T23:56:00+00:00",
                                    "id" : 14,
                                    "progress" : 100,
                                    "resources" : [
                                       {
                                          "id" : 42,
                                          "type" : "server"
                                       },
                                       {
                                          "id" : 38,
                                          "type" : "firewall"
                                       }
                                    ],
                                    "started" : "2016-01-30T23:55:00+00:00",
                                    "status" : "success"
                                 }
                              ],
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "firewall" : {
                              "properties" : {
                                 "applied_to" : {
                                    "items" : {
                                       "properties" : {
                                          "applied_to_resources" : {
                                             "items" : {
                                                "properties" : {
                                                   "server" : {
                                                      "properties" : {
                                                         "id" : {
                                                            "description" : "ID of the Resource",
                                                            "example" : 42,
                                                            "format" : "int64",
                                                            "type" : "integer"
                                                         }
                                                      },
                                                      "required" : [
                                                         "id"
                                                      ],
                                                      "type" : "object"
                                                   },
                                                   "type" : {
                                                      "description" : "Type of resource referenced",
                                                      "enum" : [
                                                         "server"
                                                      ],
                                                      "example" : "server",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          },
                                          "label_selector" : {
                                             "properties" : {
                                                "selector" : {
                                                   "description" : "Label selector",
                                                   "example" : "env=prod",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "selector"
                                             ],
                                             "type" : "object"
                                          },
                                          "server" : {
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Resource",
                                                   "example" : 42,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "type" : "object"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "enum" : [
                                                "server",
                                                "label_selector"
                                             ],
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "rules" : {
                                    "items" : {
                                       "properties" : {
                                          "description" : {
                                             "description" : "Description of the Rule",
                                             "maxLength" : 255,
                                             "nullable" : true,
                                             "type" : "string"
                                          },
                                          "destination_ips" : {
                                             "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                             "example" : [
                                                "28.239.13.1/32",
                                                "28.239.14.0/24",
                                                "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                             ],
                                             "items" : {
                                                "type" : "string"
                                             },
                                             "type" : "array"
                                          },
                                          "direction" : {
                                             "description" : "Select traffic direction on which rule should be applied. Use `source_ips` for direction `in` and `destination_ips` for direction `out`.",
                                             "enum" : [
                                                "in",
                                                "out"
                                             ],
                                             "type" : "string"
                                          },
                                          "port" : {
                                             "description" : "Port or port range to which traffic will be allowed, only applicable for protocols TCP and UDP. A port range can be specified by separating two ports with a dash, e.g `1024-5000`.",
                                             "example" : "80",
                                             "type" : "string"
                                          },
                                          "protocol" : {
                                             "description" : "Type of traffic to allow",
                                             "enum" : [
                                                "tcp",
                                                "udp",
                                                "icmp",
                                                "esp",
                                                "gre"
                                             ],
                                             "type" : "string"
                                          },
                                          "source_ips" : {
                                             "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                             "example" : [
                                                "28.239.13.1/32",
                                                "28.239.14.0/24",
                                                "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                             ],
                                             "items" : {
                                                "type" : "string"
                                             },
                                             "type" : "array"
                                          }
                                       },
                                       "required" : [
                                          "direction",
                                          "protocol"
                                       ],
                                       "title" : "Rule",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "created",
                                 "rules",
                                 "applied_to"
                              ],
                              "title" : "Firewall",
                              "type" : "object"
                           }
                        },
                        "title" : "CreateFirewallResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `firewall` key contains the Firewall that was just created"
            }
         },
         "summary" : "Create a Firewall",
         "tags" : [
            "Firewalls"
         ]
      }
   },
   "/firewalls/{id}" : {
      "delete" : {
         "description" : "Deletes a Firewall.\n\n#### Call specific error codes\n\n| Code                 | Description                               |\n|--------------------- |-------------------------------------------|\n| `resource_in_use`    | Firewall must not be in use to be deleted |\n",
         "parameters" : [
            {
               "description" : "ID of the resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "204" : {
               "description" : "Firewall deleted"
            }
         },
         "summary" : "Delete a Firewall",
         "tags" : [
            "Firewalls"
         ]
      },
      "get" : {
         "description" : "Gets a specific Firewall object.",
         "parameters" : [
            {
               "description" : "ID of the resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "firewall" : {
                              "properties" : {
                                 "applied_to" : {
                                    "items" : {
                                       "properties" : {
                                          "applied_to_resources" : {
                                             "items" : {
                                                "properties" : {
                                                   "server" : {
                                                      "properties" : {
                                                         "id" : {
                                                            "description" : "ID of the Resource",
                                                            "example" : 42,
                                                            "format" : "int64",
                                                            "type" : "integer"
                                                         }
                                                      },
                                                      "required" : [
                                                         "id"
                                                      ],
                                                      "type" : "object"
                                                   },
                                                   "type" : {
                                                      "description" : "Type of resource referenced",
                                                      "enum" : [
                                                         "server"
                                                      ],
                                                      "example" : "server",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          },
                                          "label_selector" : {
                                             "properties" : {
                                                "selector" : {
                                                   "description" : "Label selector",
                                                   "example" : "env=prod",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "selector"
                                             ],
                                             "type" : "object"
                                          },
                                          "server" : {
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Resource",
                                                   "example" : 42,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "type" : "object"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "enum" : [
                                                "server",
                                                "label_selector"
                                             ],
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "rules" : {
                                    "items" : {
                                       "properties" : {
                                          "description" : {
                                             "description" : "Description of the Rule",
                                             "maxLength" : 255,
                                             "nullable" : true,
                                             "type" : "string"
                                          },
                                          "destination_ips" : {
                                             "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                             "example" : [
                                                "28.239.13.1/32",
                                                "28.239.14.0/24",
                                                "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                             ],
                                             "items" : {
                                                "type" : "string"
                                             },
                                             "type" : "array"
                                          },
                                          "direction" : {
                                             "description" : "Select traffic direction on which rule should be applied. Use `source_ips` for direction `in` and `destination_ips` for direction `out`.",
                                             "enum" : [
                                                "in",
                                                "out"
                                             ],
                                             "type" : "string"
                                          },
                                          "port" : {
                                             "description" : "Port or port range to which traffic will be allowed, only applicable for protocols TCP and UDP. A port range can be specified by separating two ports with a dash, e.g `1024-5000`.",
                                             "example" : "80",
                                             "type" : "string"
                                          },
                                          "protocol" : {
                                             "description" : "Type of traffic to allow",
                                             "enum" : [
                                                "tcp",
                                                "udp",
                                                "icmp",
                                                "esp",
                                                "gre"
                                             ],
                                             "type" : "string"
                                          },
                                          "source_ips" : {
                                             "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                             "example" : [
                                                "28.239.13.1/32",
                                                "28.239.14.0/24",
                                                "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                             ],
                                             "items" : {
                                                "type" : "string"
                                             },
                                             "type" : "array"
                                          }
                                       },
                                       "required" : [
                                          "direction",
                                          "protocol"
                                       ],
                                       "title" : "Rule",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "created",
                                 "rules",
                                 "applied_to"
                              ],
                              "title" : "Firewall",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "firewall"
                        ],
                        "title" : "FirewallResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `firewall` key contains a Firewall object"
            }
         },
         "summary" : "Get a Firewall",
         "tags" : [
            "Firewalls"
         ]
      },
      "put" : {
         "description" : "Updates the Firewall.\n\nNote that when updating labels, the Firewall's current set of labels will be replaced with the labels provided in the request body. So, for example, if you want to add a new label, you have to provide all existing labels plus the new label in the request body.\n\nNote: if the Firewall object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "example" : {
                              "labelkey" : "value"
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "New Firewall name",
                           "example" : "new-name",
                           "type" : "string"
                        }
                     },
                     "title" : "UpdateFirewallRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "firewall" : {
                              "properties" : {
                                 "applied_to" : {
                                    "items" : {
                                       "properties" : {
                                          "applied_to_resources" : {
                                             "items" : {
                                                "properties" : {
                                                   "server" : {
                                                      "properties" : {
                                                         "id" : {
                                                            "description" : "ID of the Resource",
                                                            "example" : 42,
                                                            "format" : "int64",
                                                            "type" : "integer"
                                                         }
                                                      },
                                                      "required" : [
                                                         "id"
                                                      ],
                                                      "type" : "object"
                                                   },
                                                   "type" : {
                                                      "description" : "Type of resource referenced",
                                                      "enum" : [
                                                         "server"
                                                      ],
                                                      "example" : "server",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          },
                                          "label_selector" : {
                                             "properties" : {
                                                "selector" : {
                                                   "description" : "Label selector",
                                                   "example" : "env=prod",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "selector"
                                             ],
                                             "type" : "object"
                                          },
                                          "server" : {
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Resource",
                                                   "example" : 42,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "type" : "object"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "enum" : [
                                                "server",
                                                "label_selector"
                                             ],
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "rules" : {
                                    "items" : {
                                       "properties" : {
                                          "description" : {
                                             "description" : "Description of the Rule",
                                             "maxLength" : 255,
                                             "nullable" : true,
                                             "type" : "string"
                                          },
                                          "destination_ips" : {
                                             "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                             "example" : [
                                                "28.239.13.1/32",
                                                "28.239.14.0/24",
                                                "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                             ],
                                             "items" : {
                                                "type" : "string"
                                             },
                                             "type" : "array"
                                          },
                                          "direction" : {
                                             "description" : "Select traffic direction on which rule should be applied. Use `source_ips` for direction `in` and `destination_ips` for direction `out`.",
                                             "enum" : [
                                                "in",
                                                "out"
                                             ],
                                             "type" : "string"
                                          },
                                          "port" : {
                                             "description" : "Port or port range to which traffic will be allowed, only applicable for protocols TCP and UDP. A port range can be specified by separating two ports with a dash, e.g `1024-5000`.",
                                             "example" : "80",
                                             "type" : "string"
                                          },
                                          "protocol" : {
                                             "description" : "Type of traffic to allow",
                                             "enum" : [
                                                "tcp",
                                                "udp",
                                                "icmp",
                                                "esp",
                                                "gre"
                                             ],
                                             "type" : "string"
                                          },
                                          "source_ips" : {
                                             "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                             "example" : [
                                                "28.239.13.1/32",
                                                "28.239.14.0/24",
                                                "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                             ],
                                             "items" : {
                                                "type" : "string"
                                             },
                                             "type" : "array"
                                          }
                                       },
                                       "required" : [
                                          "direction",
                                          "protocol"
                                       ],
                                       "title" : "Rule",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "created",
                                 "rules",
                                 "applied_to"
                              ],
                              "title" : "Firewall",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "firewall"
                        ],
                        "title" : "FirewallResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `firewall` key contains the Firewall that was just updated"
            }
         },
         "summary" : "Update a Firewall",
         "tags" : [
            "Firewalls"
         ]
      }
   },
   "/firewalls/{id}/actions" : {
      "get" : {
         "description" : "Returns all Action objects for a Firewall. You can sort the results by using the `sort` URI parameter, and filter them with the `status` parameter.",
         "parameters" : [
            {
               "description" : "ID of the Resource",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "set_firewall_rules",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 13,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 38,
                                    "type" : "firewall"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions for a Firewall",
         "tags" : [
            "Firewall Actions"
         ]
      }
   },
   "/firewalls/{id}/actions/apply_to_resources" : {
      "post" : {
         "description" : "Applies one Firewall to multiple resources.\n\nCurrently servers (public network interface) and label selectors are supported.\n\n#### Call specific error codes\n\n| Code                          | Description                                                   |\n|-------------------------------|---------------------------------------------------------------|\n| `firewall_already_applied`    | Firewall was already applied on resource                      |\n| `incompatible_network_type`   | The Network type is incompatible for the given resource       |\n| `firewall_resource_not_found` | The resource the Firewall should be attached to was not found |\n",
         "parameters" : [
            {
               "description" : "ID of the Firewall",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "example" : {
                     "apply_to" : [
                        {
                           "server" : {
                              "id" : 42
                           },
                           "type" : "server"
                        }
                     ]
                  },
                  "schema" : {
                     "properties" : {
                        "apply_to" : {
                           "description" : "Resources the Firewall should be applied to",
                           "items" : {
                              "properties" : {
                                 "label_selector" : {
                                    "description" : "Configuration for type label_selector, required if type is `label_selector`",
                                    "properties" : {
                                       "selector" : {
                                          "description" : "Label selector",
                                          "example" : "env=prod",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "selector"
                                    ],
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "Configuration for type server, required if type is `server`",
                                    "properties" : {
                                       "id" : {
                                          "description" : "ID of the Server",
                                          "format" : "int64",
                                          "type" : "integer"
                                       }
                                    },
                                    "required" : [
                                       "id"
                                    ],
                                    "type" : "object"
                                 },
                                 "type" : {
                                    "description" : "Type of the resource",
                                    "enum" : [
                                       "server",
                                       "label_selector"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "title" : "FirewallApplyToResources",
                              "type" : "object"
                           },
                           "type" : "array"
                        }
                     },
                     "required" : [
                        "apply_to"
                     ],
                     "title" : "ApplyToResourcesRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "apply_firewall",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 14,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 42,
                                    "type" : "server"
                                 },
                                 {
                                    "id" : 38,
                                    "type" : "firewall"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains multiple `apply_firewall` Actions"
            }
         },
         "summary" : "Apply to Resources",
         "tags" : [
            "Firewall Actions"
         ]
      }
   },
   "/firewalls/{id}/actions/remove_from_resources" : {
      "post" : {
         "description" : "Removes one Firewall from multiple resources.\n\nCurrently only Servers (and their public network interfaces) are supported.\n\n#### Call specific error codes\n\n| Code                                  | Description                                                            |\n|---------------------------------------|------------------------------------------------------------------------|\n| `firewall_already_removed`            | Firewall was already removed from the resource                         |\n| `firewall_resource_not_found`         | The resource the Firewall should be attached to was not found          |\n| `firewall_managed_by_label_selector`  | Firewall was applied via label selector and cannot be removed manually |\n",
         "parameters" : [
            {
               "description" : "ID of the Firewall",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "example" : {
                     "remove_from" : [
                        {
                           "server" : {
                              "id" : 42
                           },
                           "type" : "server"
                        }
                     ]
                  },
                  "schema" : {
                     "properties" : {
                        "remove_from" : {
                           "description" : "Resources the Firewall should be removed from",
                           "items" : {
                              "properties" : {
                                 "label_selector" : {
                                    "description" : "Configuration for type label_selector, required if type is `label_selector`",
                                    "properties" : {
                                       "selector" : {
                                          "description" : "Label selector",
                                          "example" : "env=prod",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "selector"
                                    ],
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "Configuration for type server, required if type is `server`",
                                    "properties" : {
                                       "id" : {
                                          "description" : "ID of the Server",
                                          "format" : "int64",
                                          "type" : "integer"
                                       }
                                    },
                                    "required" : [
                                       "id"
                                    ],
                                    "type" : "object"
                                 },
                                 "type" : {
                                    "description" : "Type of the resource",
                                    "enum" : [
                                       "server",
                                       "label_selector"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "title" : "FirewallRemoveFromResources",
                              "type" : "object"
                           },
                           "type" : "array"
                        }
                     },
                     "required" : [
                        "remove_from"
                     ],
                     "title" : "RemoveFromResourcesRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "remove_firewall",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 14,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 42,
                                    "type" : "server"
                                 },
                                 {
                                    "id" : 38,
                                    "type" : "firewall"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains multiple `remove_firewall` Actions"
            }
         },
         "summary" : "Remove from Resources",
         "tags" : [
            "Firewall Actions"
         ]
      }
   },
   "/firewalls/{id}/actions/set_rules" : {
      "post" : {
         "description" : "Sets the rules of a Firewall.\n\nAll existing rules will be overwritten. Pass an empty `rules` array to remove all rules.\nThe maximum amount of rules that can be defined is 50.\n\n#### Call specific error codes\n\n| Code                          | Description                                                   |\n|-------------------------------|---------------------------------------------------------------|\n| `firewall_resource_not_found` | The resource the Firewall should be attached to was not found |\n",
         "parameters" : [
            {
               "description" : "ID of the Firewall",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "examples" : {
                     "rules" : {
                        "value" : {
                           "rules" : [
                              {
                                 "description" : "Allow port 80",
                                 "direction" : "in",
                                 "port" : "80",
                                 "protocol" : "tcp",
                                 "source_ips" : [
                                    "28.239.13.1/32",
                                    "28.239.14.0/24",
                                    "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                 ]
                              }
                           ]
                        }
                     }
                  },
                  "schema" : {
                     "properties" : {
                        "rules" : {
                           "description" : "Array of rules",
                           "items" : {
                              "properties" : {
                                 "description" : {
                                    "description" : "Description of the Rule",
                                    "maxLength" : 255,
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "destination_ips" : {
                                    "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                    "example" : [
                                       "28.239.13.1/32",
                                       "28.239.14.0/24",
                                       "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 },
                                 "direction" : {
                                    "description" : "Select traffic direction on which rule should be applied. Use `source_ips` for direction `in` and `destination_ips` for direction `out`.",
                                    "enum" : [
                                       "in",
                                       "out"
                                    ],
                                    "type" : "string"
                                 },
                                 "port" : {
                                    "description" : "Port or port range to which traffic will be allowed, only applicable for protocols TCP and UDP. A port range can be specified by separating two ports with a dash, e.g `1024-5000`.",
                                    "example" : "80",
                                    "type" : "string"
                                 },
                                 "protocol" : {
                                    "description" : "Type of traffic to allow",
                                    "enum" : [
                                       "tcp",
                                       "udp",
                                       "icmp",
                                       "esp",
                                       "gre"
                                    ],
                                    "type" : "string"
                                 },
                                 "source_ips" : {
                                    "description" : "List of permitted IPv4/IPv6 addresses in CIDR notation. Use `0.0.0.0/0` to allow all IPv4 addresses and `::/0` to allow all IPv6 addresses. You can specify 100 CIDRs at most.",
                                    "example" : [
                                       "28.239.13.1/32",
                                       "28.239.14.0/24",
                                       "ff21:1eac:9a3b:ee58:5ca:990c:8bc9:c03b/128"
                                    ],
                                    "items" : {
                                       "type" : "string"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "direction",
                                 "protocol"
                              ],
                              "title" : "Rule",
                              "type" : "object"
                           },
                           "maxItems" : 50,
                           "type" : "array"
                        }
                     },
                     "required" : [
                        "rules"
                     ],
                     "title" : "SetRulesRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "set_firewall_rules",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 13,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 38,
                                    "type" : "firewall"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           },
                           {
                              "command" : "apply_firewall",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 14,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 38,
                                    "type" : "firewall"
                                 },
                                 {
                                    "id" : 42,
                                    "type" : "server"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains one `set_firewall_rules` Action plus one `apply_firewall` Action per resource where the Firewall is active"
            }
         },
         "summary" : "Set Rules",
         "tags" : [
            "Firewall Actions"
         ]
      }
   },
   "/firewalls/{id}/actions/{action_id}" : {
      "get" : {
         "description" : "Returns a specific Action for a Firewall.",
         "parameters" : [
            {
               "description" : "ID of the Firewall",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "ID of the Action",
               "in" : "path",
               "name" : "action_id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "set_firewall_rules",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 38,
                                 "type" : "firewall"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the Firewall Action"
            }
         },
         "summary" : "Get an Action for a Firewall",
         "tags" : [
            "Firewall Actions"
         ]
      }
   },
   "/floating_ips" : {
      "get" : {
         "description" : "Returns all Floating IP objects.",
         "parameters" : [
            {
               "description" : "Can be used to filter Floating IPs by their name. The response will only contain the Floating IP matching the specified name.",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter Floating IPs by labels. The response will only contain Floating IPs matching the label selector.",
               "in" : "query",
               "name" : "label_selector",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times. Choices id id:asc id:desc created created:asc created:desc",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "created",
                     "created:asc",
                     "created:desc"
                  ],
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "floating_ips" : {
                              "items" : {
                                 "properties" : {
                                    "blocked" : {
                                       "description" : "Whether the IP is blocked",
                                       "example" : false,
                                       "type" : "boolean"
                                    },
                                    "created" : {
                                       "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "description" : {
                                       "description" : "Description of the Resource",
                                       "example" : "this describes my resource",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "dns_ptr" : {
                                       "description" : "Array of reverse DNS entries",
                                       "items" : {
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "DNS pointer for the specific IP address",
                                                "example" : "server.example.com",
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "Single IPv4 or IPv6 address",
                                                "example" : "2001:db8::1",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "ip",
                                             "dns_ptr"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "home_location" : {
                                       "description" : "Location the Floating IP was created in. Routing is optimized for this Location.",
                                       "properties" : {
                                          "city" : {
                                             "description" : "City the Location is closest to",
                                             "example" : "Falkenstein",
                                             "type" : "string"
                                          },
                                          "country" : {
                                             "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                             "example" : "DE",
                                             "type" : "string"
                                          },
                                          "description" : {
                                             "description" : "Description of the Location",
                                             "example" : "Falkenstein DC Park 1",
                                             "type" : "string"
                                          },
                                          "id" : {
                                             "description" : "ID of the Location",
                                             "example" : 1,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "latitude" : {
                                             "description" : "Latitude of the city closest to the Location",
                                             "example" : 50.47612,
                                             "type" : "number"
                                          },
                                          "longitude" : {
                                             "description" : "Longitude of the city closest to the Location",
                                             "example" : 12.370071,
                                             "type" : "number"
                                          },
                                          "name" : {
                                             "description" : "Unique identifier of the Location",
                                             "example" : "fsn1",
                                             "type" : "string"
                                          },
                                          "network_zone" : {
                                             "description" : "Name of network zone this Location resides in",
                                             "example" : "eu-central",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "name",
                                          "description",
                                          "country",
                                          "city",
                                          "latitude",
                                          "longitude",
                                          "network_zone"
                                       ],
                                       "type" : "object"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "ip" : {
                                       "description" : "IP address",
                                       "example" : "131.232.99.1",
                                       "type" : "string"
                                    },
                                    "labels" : {
                                       "additionalProperties" : {
                                          "type" : "string"
                                       },
                                       "description" : "User-defined labels (key-value pairs)",
                                       "type" : "object"
                                    },
                                    "name" : {
                                       "description" : "Name of the Resource. Must be unique per Project.",
                                       "example" : "my-resource",
                                       "type" : "string"
                                    },
                                    "protection" : {
                                       "description" : "Protection configuration for the Resource",
                                       "properties" : {
                                          "delete" : {
                                             "description" : "If true, prevents the Resource from being deleted",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "delete"
                                       ],
                                       "type" : "object"
                                    },
                                    "server" : {
                                       "description" : "ID of the Server the Floating IP is assigned to, null if it is not assigned at all",
                                       "example" : 42,
                                       "format" : "int64",
                                       "nullable" : true,
                                       "type" : "integer"
                                    },
                                    "type" : {
                                       "description" : "Type of the Floating IP",
                                       "enum" : [
                                          "ipv4",
                                          "ipv6"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "description",
                                    "ip",
                                    "type",
                                    "server",
                                    "dns_ptr",
                                    "home_location",
                                    "blocked",
                                    "protection",
                                    "labels",
                                    "created"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "floating_ips"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `floating_ips` key in the reply contains an array of Floating IP objects with this structure"
            }
         },
         "summary" : "Get all Floating IPs",
         "tags" : [
            "Floating IPs"
         ]
      },
      "post" : {
         "description" : "Creates a new Floating IP assigned to a Server. If you want to create a Floating IP that is not bound to a Server, you need to provide the `home_location` key instead of `server`. This can be either the ID or the name of the Location this IP shall be created in. Note that a Floating IP can be assigned to a Server in any Location later on. For optimal routing it is advised to use the Floating IP in the same Location it was created in.",
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "description" : {
                           "example" : "Web Frontend",
                           "type" : "string"
                        },
                        "home_location" : {
                           "description" : "Home Location (routing is optimized for that Location). Only optional if Server argument is passed.",
                           "example" : "fsn1",
                           "type" : "string"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "example" : {
                              "labelkey" : "value"
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "example" : "Web Frontend",
                           "type" : "string"
                        },
                        "server" : {
                           "description" : "ID of the Server to assign the Floating IP to",
                           "example" : 42,
                           "format" : "int64",
                           "type" : "integer"
                        },
                        "type" : {
                           "description" : "Floating IP type",
                           "enum" : [
                              "ipv4",
                              "ipv6"
                           ],
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "type"
                     ],
                     "title" : "CreateFloatingIPRequest",
                     "type" : "object"
                  }
               }
            },
            "description" : "The `type` argument is required while `home_location` and `server` are mutually exclusive."
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "create_floating_ip",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        },
                        "floating_ip" : {
                           "blocked" : false,
                           "created" : "2016-01-30T23:50:00+00:00",
                           "description" : "Web Frontend",
                           "dns_ptr" : [
                              {
                                 "dns_ptr" : "server.example.com",
                                 "ip" : "2001:db8::1"
                              }
                           ],
                           "home_location" : {
                              "city" : "Falkenstein",
                              "country" : "DE",
                              "description" : "Falkenstein DC Park 1",
                              "id" : 1,
                              "latitude" : 50.47612,
                              "longitude" : 12.370071,
                              "name" : "fsn1",
                              "network_zone" : "eu-central"
                           },
                           "id" : 4711,
                           "ip" : "131.232.99.1",
                           "labels" : {
                              "env" : "dev"
                           },
                           "name" : "Web Frontend",
                           "protection" : {
                              "delete" : false
                           },
                           "server" : 42,
                           "type" : "ipv4"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           },
                           "floating_ip" : {
                              "properties" : {
                                 "blocked" : {
                                    "description" : "Whether the IP is blocked",
                                    "example" : false,
                                    "type" : "boolean"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Resource",
                                    "example" : "this describes my resource",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "dns_ptr" : {
                                    "description" : "Array of reverse DNS entries",
                                    "items" : {
                                       "properties" : {
                                          "dns_ptr" : {
                                             "description" : "DNS pointer for the specific IP address",
                                             "example" : "server.example.com",
                                             "type" : "string"
                                          },
                                          "ip" : {
                                             "description" : "Single IPv4 or IPv6 address",
                                             "example" : "2001:db8::1",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "ip",
                                          "dns_ptr"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "home_location" : {
                                    "description" : "Location the Floating IP was created in. Routing is optimized for this Location.",
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "ip" : {
                                    "description" : "IP address",
                                    "example" : "131.232.99.1",
                                    "type" : "string"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "ID of the Server the Floating IP is assigned to, null if it is not assigned at all",
                                    "example" : 42,
                                    "format" : "int64",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "type" : {
                                    "description" : "Type of the Floating IP",
                                    "enum" : [
                                       "ipv4",
                                       "ipv6"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "ip",
                                 "type",
                                 "server",
                                 "dns_ptr",
                                 "home_location",
                                 "blocked",
                                 "protection",
                                 "labels",
                                 "created"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "floating_ip"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `floating_ip` key in the reply contains the object that was just created"
            }
         },
         "summary" : "Create a Floating IP",
         "tags" : [
            "Floating IPs"
         ]
      }
   },
   "/floating_ips/{id}" : {
      "delete" : {
         "description" : "Deletes a Floating IP. If it is currently assigned to a Server it will automatically get unassigned.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "204" : {
               "description" : "Floating IP deleted"
            }
         },
         "summary" : "Delete a Floating IP",
         "tags" : [
            "Floating IPs"
         ]
      },
      "get" : {
         "description" : "Returns a specific Floating IP object.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "floating_ip" : {
                              "properties" : {
                                 "blocked" : {
                                    "description" : "Whether the IP is blocked",
                                    "example" : false,
                                    "type" : "boolean"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Resource",
                                    "example" : "this describes my resource",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "dns_ptr" : {
                                    "description" : "Array of reverse DNS entries",
                                    "items" : {
                                       "properties" : {
                                          "dns_ptr" : {
                                             "description" : "DNS pointer for the specific IP address",
                                             "example" : "server.example.com",
                                             "type" : "string"
                                          },
                                          "ip" : {
                                             "description" : "Single IPv4 or IPv6 address",
                                             "example" : "2001:db8::1",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "ip",
                                          "dns_ptr"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "home_location" : {
                                    "description" : "Location the Floating IP was created in. Routing is optimized for this Location.",
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "ip" : {
                                    "description" : "IP address",
                                    "example" : "131.232.99.1",
                                    "type" : "string"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "ID of the Server the Floating IP is assigned to, null if it is not assigned at all",
                                    "example" : 42,
                                    "format" : "int64",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "type" : {
                                    "description" : "Type of the Floating IP",
                                    "enum" : [
                                       "ipv4",
                                       "ipv6"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "ip",
                                 "type",
                                 "server",
                                 "dns_ptr",
                                 "home_location",
                                 "blocked",
                                 "protection",
                                 "labels",
                                 "created"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "floating_ip"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `floating_ip` key in the reply contains a Floating IP object with this structure"
            }
         },
         "summary" : "Get a Floating IP",
         "tags" : [
            "Floating IPs"
         ]
      },
      "put" : {
         "description" : "Updates the description or labels of a Floating IP.\nAlso note that when updating labels, the Floating IP’s current set of labels will be replaced with the labels provided in the request body. So, for example, if you want to add a new label, you have to provide all existing labels plus the new label in the request body.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "description" : {
                           "description" : "New Description to set",
                           "example" : "Web Frontend",
                           "type" : "string"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "example" : {
                              "labelkey" : "value"
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "New unique name to set",
                           "example" : "Web Frontend",
                           "type" : "string"
                        }
                     },
                     "title" : "UpdateFloatingIPRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "floating_ip" : {
                           "blocked" : false,
                           "created" : "2016-01-30T23:50:00+00:00",
                           "description" : "Web Frontend",
                           "dns_ptr" : [
                              {
                                 "dns_ptr" : "server.example.com",
                                 "ip" : "2001:db8::1"
                              }
                           ],
                           "home_location" : {
                              "city" : "Falkenstein",
                              "country" : "DE",
                              "description" : "Falkenstein DC Park 1",
                              "id" : 1,
                              "latitude" : 50.47612,
                              "longitude" : 12.370071,
                              "name" : "fsn1",
                              "network_zone" : "eu-central"
                           },
                           "id" : 4711,
                           "ip" : "131.232.99.1",
                           "labels" : {
                              "labelkey" : "value"
                           },
                           "name" : "Web Frontend",
                           "protection" : {
                              "delete" : false
                           },
                           "server" : 42,
                           "type" : "ipv4"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "floating_ip" : {
                              "properties" : {
                                 "blocked" : {
                                    "description" : "Whether the IP is blocked",
                                    "example" : false,
                                    "type" : "boolean"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Resource",
                                    "example" : "this describes my resource",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "dns_ptr" : {
                                    "description" : "Array of reverse DNS entries",
                                    "items" : {
                                       "properties" : {
                                          "dns_ptr" : {
                                             "description" : "DNS pointer for the specific IP address",
                                             "example" : "server.example.com",
                                             "type" : "string"
                                          },
                                          "ip" : {
                                             "description" : "Single IPv4 or IPv6 address",
                                             "example" : "2001:db8::1",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "ip",
                                          "dns_ptr"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "home_location" : {
                                    "description" : "Location the Floating IP was created in. Routing is optimized for this Location.",
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "ip" : {
                                    "description" : "IP address",
                                    "example" : "131.232.99.1",
                                    "type" : "string"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "ID of the Server the Floating IP is assigned to, null if it is not assigned at all",
                                    "example" : 42,
                                    "format" : "int64",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "type" : {
                                    "description" : "Type of the Floating IP",
                                    "enum" : [
                                       "ipv4",
                                       "ipv6"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "ip",
                                 "type",
                                 "server",
                                 "dns_ptr",
                                 "home_location",
                                 "blocked",
                                 "protection",
                                 "labels",
                                 "created"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "floating_ip"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `floating_ip` key in the reply contains the modified Floating IP object with the new description"
            }
         },
         "summary" : "Update a Floating IP",
         "tags" : [
            "Floating IPs"
         ]
      }
   },
   "/floating_ips/{id}/actions" : {
      "get" : {
         "description" : "Returns all Action objects for a Floating IP. You can sort the results by using the `sort` URI parameter, and filter them with the `status` parameter.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "assign_floating_ip",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 13,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 4711,
                                    "type" : "server"
                                 },
                                 {
                                    "id" : 4712,
                                    "type" : "floating_ip"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions for a Floating IP",
         "tags" : [
            "Floating IP Actions"
         ]
      }
   },
   "/floating_ips/{id}/actions/assign" : {
      "post" : {
         "description" : "Assigns a Floating IP to a Server.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "description" : "#### Call specific error codes\n\n| Code                          | Description                                                   |\n|------------------------------ |-------------------------------------------------------------- |\n| `floating_ip_assigned`        | The floating IP is already assigned                           |\n",
                     "properties" : {
                        "server" : {
                           "description" : "ID of the Server the Floating IP shall be assigned to",
                           "example" : 42,
                           "format" : "int64",
                           "type" : "integer"
                        }
                     },
                     "required" : [
                        "server"
                     ],
                     "title" : "AssignFloatingIPRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "assign_floating_ip",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              },
                              {
                                 "id" : 4711,
                                 "type" : "floating_ip"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `assign` Action"
            }
         },
         "summary" : "Assign a Floating IP to a Server",
         "tags" : [
            "Floating IP Actions"
         ]
      }
   },
   "/floating_ips/{id}/actions/change_dns_ptr" : {
      "post" : {
         "description" : "Changes the hostname that will appear when getting the hostname belonging to this Floating IP.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "dns_ptr" : {
                           "description" : "Hostname to set as a reverse DNS PTR entry, will reset to original default value if `null`",
                           "example" : "server02.example.com",
                           "nullable" : true,
                           "type" : "string"
                        },
                        "ip" : {
                           "description" : "IP address for which to set the reverse DNS entry",
                           "example" : "1.2.3.4",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "ip",
                        "dns_ptr"
                     ],
                     "title" : "ChangeDNSPTRRequest",
                     "type" : "object"
                  }
               }
            },
            "description" : "Select the IP address for which to change the DNS entry by passing `ip`. For a Floating IP of type `ipv4` this must exactly match the IP address of the Floating IP. For a Floating IP of type `ipv6` this must be a single IP within the IPv6 /64 range that belongs to this Floating IP. You can add up to 100 IPv6 reverse DNS entries.\n\nThe target hostname is set by passing `dns_ptr`.\n"
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_dns_ptr",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "floating_ip"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_dns_ptr` Action"
            }
         },
         "summary" : "Change reverse DNS entry for a Floating IP",
         "tags" : [
            "Floating IP Actions"
         ]
      }
   },
   "/floating_ips/{id}/actions/change_protection" : {
      "post" : {
         "description" : "Changes the protection configuration of the Floating IP.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "delete" : {
                           "description" : "If true, prevents the Floating IP from being deleted",
                           "example" : true,
                           "type" : "boolean"
                        }
                     },
                     "title" : "ChangeProtectionRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_protection",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "floating_ip"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_protection` Action"
            }
         },
         "summary" : "Change Floating IP Protection",
         "tags" : [
            "Floating IP Actions"
         ]
      }
   },
   "/floating_ips/{id}/actions/unassign" : {
      "post" : {
         "description" : "Unassigns a Floating IP, resulting in it being unreachable. You may assign it to a Server again at a later time.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "unassign_floating_ip",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              },
                              {
                                 "id" : 4711,
                                 "type" : "floating_ip"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `unassign` Action"
            }
         },
         "summary" : "Unassign a Floating IP",
         "tags" : [
            "Floating IP Actions"
         ]
      }
   },
   "/floating_ips/{id}/actions/{action_id}" : {
      "get" : {
         "description" : "Returns a specific Action object for a Floating IP.",
         "parameters" : [
            {
               "description" : "ID of the Floating IP",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "ID of the Action",
               "in" : "path",
               "name" : "action_id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "assign_floating_ip",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              },
                              {
                                 "id" : 4711,
                                 "type" : "floating_ip"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key in the reply has this structure"
            }
         },
         "summary" : "Get an Action for a Floating IP",
         "tags" : [
            "Floating IP Actions"
         ]
      }
   },
   "/images" : {
      "get" : {
         "description" : "Returns all Image objects. You can select specific Image types only and sort the results by using URI parameters.",
         "parameters" : [
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "name",
                     "name:asc",
                     "name:desc",
                     "created",
                     "created:asc",
                     "created:desc"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "type",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "system",
                     "snapshot",
                     "backup",
                     "app"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times. The response will only contain Images matching the status.",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "available",
                     "creating"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times. Server ID linked to the Image. Only available for Images of type `backup`",
               "in" : "query",
               "name" : "bound_to",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "include_deprecated",
               "required" : false,
               "schema" : {
                  "type" : "boolean"
               }
            },
            {
               "description" : "Can be used to filter resources by their name. The response will only contain the resources matching the specified name",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by labels. The response will only contain resources matching the label selector.",
               "in" : "query",
               "name" : "label_selector",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Return only Images with the given architecture.",
               "in" : "query",
               "name" : "architecture",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "images" : {
                              "items" : {
                                 "properties" : {
                                    "architecture" : {
                                       "description" : "Type of cpu architecture this image is compatible with.",
                                       "enum" : [
                                          "x86",
                                          "arm"
                                       ],
                                       "example" : "x86",
                                       "type" : "string"
                                    },
                                    "bound_to" : {
                                       "description" : "ID of Server the Image is bound to. Only set for Images of type `backup`.",
                                       "example" : null,
                                       "format" : "int64",
                                       "nullable" : true,
                                       "type" : "integer"
                                    },
                                    "created" : {
                                       "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "created_from" : {
                                       "description" : "Information about the Server the Image was created from",
                                       "nullable" : true,
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Server the Image was created from",
                                             "example" : 1,
                                             "type" : "integer"
                                          },
                                          "name" : {
                                             "description" : "Server name at the time the Image was created",
                                             "example" : "Server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "name"
                                       ],
                                       "type" : "object"
                                    },
                                    "deleted" : {
                                       "description" : "Point in time where the Image was deleted (in ISO-8601 format)",
                                       "example" : null,
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "deprecated" : {
                                       "description" : "Point in time when the Image is considered to be deprecated (in ISO-8601 format)",
                                       "example" : "2018-02-28T00:00:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "description" : {
                                       "description" : "Description of the Image",
                                       "example" : "Ubuntu 20.04 Standard 64 bit",
                                       "type" : "string"
                                    },
                                    "disk_size" : {
                                       "description" : "Size of the disk contained in the Image in GB",
                                       "example" : 10,
                                       "type" : "number"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "image_size" : {
                                       "description" : "Size of the Image file in our storage in GB. For snapshot Images this is the value relevant for calculating costs for the Image.",
                                       "example" : 2.3,
                                       "nullable" : true,
                                       "type" : "number"
                                    },
                                    "labels" : {
                                       "additionalProperties" : {
                                          "type" : "string"
                                       },
                                       "description" : "User-defined labels (key-value pairs)",
                                       "type" : "object"
                                    },
                                    "name" : {
                                       "description" : "Unique identifier of the Image. This value is only set for system Images.",
                                       "example" : "ubuntu-20.04",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "os_flavor" : {
                                       "description" : "Flavor of operating system contained in the Image",
                                       "enum" : [
                                          "ubuntu",
                                          "centos",
                                          "debian",
                                          "fedora",
                                          "unknown"
                                       ],
                                       "example" : "ubuntu",
                                       "type" : "string"
                                    },
                                    "os_version" : {
                                       "description" : "Operating system version",
                                       "example" : "20.04",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "protection" : {
                                       "description" : "Protection configuration for the Resource",
                                       "properties" : {
                                          "delete" : {
                                             "description" : "If true, prevents the Resource from being deleted",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "delete"
                                       ],
                                       "type" : "object"
                                    },
                                    "rapid_deploy" : {
                                       "description" : "Indicates that rapid deploy of the Image is available",
                                       "example" : false,
                                       "type" : "boolean"
                                    },
                                    "status" : {
                                       "description" : "Whether the Image can be used or if it's still being created or unavailable",
                                       "enum" : [
                                          "available",
                                          "creating",
                                          "unavailable"
                                       ],
                                       "type" : "string"
                                    },
                                    "type" : {
                                       "description" : "Type of the Image",
                                       "enum" : [
                                          "system",
                                          "app",
                                          "snapshot",
                                          "backup",
                                          "temporary"
                                       ],
                                       "example" : "snapshot",
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "type",
                                    "status",
                                    "name",
                                    "description",
                                    "image_size",
                                    "disk_size",
                                    "created",
                                    "created_from",
                                    "bound_to",
                                    "os_flavor",
                                    "os_version",
                                    "protection",
                                    "deprecated",
                                    "deleted",
                                    "labels",
                                    "architecture"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "images"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `images` key in the reply contains an array of Image objects with this structure"
            }
         },
         "summary" : "Get all Images",
         "tags" : [
            "Images"
         ]
      }
   },
   "/images/{id}" : {
      "delete" : {
         "description" : "Deletes an Image. Only Images of type `snapshot` and `backup` can be deleted.",
         "parameters" : [
            {
               "description" : "ID of the Image",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "204" : {
               "description" : "Image deleted"
            }
         },
         "summary" : "Delete an Image",
         "tags" : [
            "Images"
         ]
      },
      "get" : {
         "description" : "Returns a specific Image object.",
         "parameters" : [
            {
               "description" : "ID of the Image",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "image" : {
                              "properties" : {
                                 "architecture" : {
                                    "description" : "Type of cpu architecture this image is compatible with.",
                                    "enum" : [
                                       "x86",
                                       "arm"
                                    ],
                                    "example" : "x86",
                                    "type" : "string"
                                 },
                                 "bound_to" : {
                                    "description" : "ID of Server the Image is bound to. Only set for Images of type `backup`.",
                                    "example" : null,
                                    "format" : "int64",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "created_from" : {
                                    "description" : "Information about the Server the Image was created from",
                                    "nullable" : true,
                                    "properties" : {
                                       "id" : {
                                          "description" : "ID of the Server the Image was created from",
                                          "example" : 1,
                                          "type" : "integer"
                                       },
                                       "name" : {
                                          "description" : "Server name at the time the Image was created",
                                          "example" : "Server",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name"
                                    ],
                                    "type" : "object"
                                 },
                                 "deleted" : {
                                    "description" : "Point in time where the Image was deleted (in ISO-8601 format)",
                                    "example" : null,
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "deprecated" : {
                                    "description" : "Point in time when the Image is considered to be deprecated (in ISO-8601 format)",
                                    "example" : "2018-02-28T00:00:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Image",
                                    "example" : "Ubuntu 20.04 Standard 64 bit",
                                    "type" : "string"
                                 },
                                 "disk_size" : {
                                    "description" : "Size of the disk contained in the Image in GB",
                                    "example" : 10,
                                    "type" : "number"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "image_size" : {
                                    "description" : "Size of the Image file in our storage in GB. For snapshot Images this is the value relevant for calculating costs for the Image.",
                                    "example" : 2.3,
                                    "nullable" : true,
                                    "type" : "number"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Unique identifier of the Image. This value is only set for system Images.",
                                    "example" : "ubuntu-20.04",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "os_flavor" : {
                                    "description" : "Flavor of operating system contained in the Image",
                                    "enum" : [
                                       "ubuntu",
                                       "centos",
                                       "debian",
                                       "fedora",
                                       "unknown"
                                    ],
                                    "example" : "ubuntu",
                                    "type" : "string"
                                 },
                                 "os_version" : {
                                    "description" : "Operating system version",
                                    "example" : "20.04",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "rapid_deploy" : {
                                    "description" : "Indicates that rapid deploy of the Image is available",
                                    "example" : false,
                                    "type" : "boolean"
                                 },
                                 "status" : {
                                    "description" : "Whether the Image can be used or if it's still being created or unavailable",
                                    "enum" : [
                                       "available",
                                       "creating",
                                       "unavailable"
                                    ],
                                    "type" : "string"
                                 },
                                 "type" : {
                                    "description" : "Type of the Image",
                                    "enum" : [
                                       "system",
                                       "app",
                                       "snapshot",
                                       "backup",
                                       "temporary"
                                    ],
                                    "example" : "snapshot",
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "type",
                                 "status",
                                 "name",
                                 "description",
                                 "image_size",
                                 "disk_size",
                                 "created",
                                 "created_from",
                                 "bound_to",
                                 "os_flavor",
                                 "os_version",
                                 "protection",
                                 "deprecated",
                                 "deleted",
                                 "labels",
                                 "architecture"
                              ],
                              "type" : "object"
                           }
                        },
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `image` key in the reply contains an Image object with this structure"
            }
         },
         "summary" : "Get an Image",
         "tags" : [
            "Images"
         ]
      },
      "put" : {
         "description" : "Updates the Image. You may change the description, convert a Backup Image to a Snapshot Image or change the Image labels. Only Images of type `snapshot` and `backup` can be updated.\n\nNote that when updating labels, the current set of labels will be replaced with the labels provided in the request body. So, for example, if you want to add a new label, you have to provide all existing labels plus the new label in the request body.\n",
         "parameters" : [
            {
               "description" : "ID of the Image",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "description" : {
                           "description" : "New description of Image",
                           "example" : "My new Image description",
                           "type" : "string"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "example" : {
                              "labelkey" : "value"
                           },
                           "type" : "object"
                        },
                        "type" : {
                           "description" : "Destination Image type to convert to",
                           "enum" : [
                              "snapshot"
                           ],
                           "type" : "string"
                        }
                     },
                     "title" : "UpdateImageRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "image" : {
                           "architecture" : "x86",
                           "bound_to" : null,
                           "created" : "2016-01-30T23:50:00+00:00",
                           "created_from" : {
                              "id" : 1,
                              "name" : "Server"
                           },
                           "deleted" : null,
                           "deprecated" : "2018-02-28T00:00:00+00:00",
                           "description" : "My new Image description",
                           "disk_size" : 10,
                           "id" : 4711,
                           "image_size" : 2.3,
                           "labels" : {
                              "labelkey" : "value"
                           },
                           "name" : null,
                           "os_flavor" : "ubuntu",
                           "os_version" : "20.04",
                           "protection" : {
                              "delete" : false
                           },
                           "rapid_deploy" : false,
                           "status" : "available",
                           "type" : "snapshot"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "image" : {
                              "properties" : {
                                 "architecture" : {
                                    "description" : "Type of cpu architecture this image is compatible with.",
                                    "enum" : [
                                       "x86",
                                       "arm"
                                    ],
                                    "example" : "x86",
                                    "type" : "string"
                                 },
                                 "bound_to" : {
                                    "description" : "ID of Server the Image is bound to. Only set for Images of type `backup`.",
                                    "example" : null,
                                    "format" : "int64",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "created_from" : {
                                    "description" : "Information about the Server the Image was created from",
                                    "nullable" : true,
                                    "properties" : {
                                       "id" : {
                                          "description" : "ID of the Server the Image was created from",
                                          "example" : 1,
                                          "type" : "integer"
                                       },
                                       "name" : {
                                          "description" : "Server name at the time the Image was created",
                                          "example" : "Server",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name"
                                    ],
                                    "type" : "object"
                                 },
                                 "deleted" : {
                                    "description" : "Point in time where the Image was deleted (in ISO-8601 format)",
                                    "example" : null,
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "deprecated" : {
                                    "description" : "Point in time when the Image is considered to be deprecated (in ISO-8601 format)",
                                    "example" : "2018-02-28T00:00:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Image",
                                    "example" : "Ubuntu 20.04 Standard 64 bit",
                                    "type" : "string"
                                 },
                                 "disk_size" : {
                                    "description" : "Size of the disk contained in the Image in GB",
                                    "example" : 10,
                                    "type" : "number"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "image_size" : {
                                    "description" : "Size of the Image file in our storage in GB. For snapshot Images this is the value relevant for calculating costs for the Image.",
                                    "example" : 2.3,
                                    "nullable" : true,
                                    "type" : "number"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Unique identifier of the Image. This value is only set for system Images.",
                                    "example" : "ubuntu-20.04",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "os_flavor" : {
                                    "description" : "Flavor of operating system contained in the Image",
                                    "enum" : [
                                       "ubuntu",
                                       "centos",
                                       "debian",
                                       "fedora",
                                       "unknown"
                                    ],
                                    "example" : "ubuntu",
                                    "type" : "string"
                                 },
                                 "os_version" : {
                                    "description" : "Operating system version",
                                    "example" : "20.04",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "rapid_deploy" : {
                                    "description" : "Indicates that rapid deploy of the Image is available",
                                    "example" : false,
                                    "type" : "boolean"
                                 },
                                 "status" : {
                                    "description" : "Whether the Image can be used or if it's still being created or unavailable",
                                    "enum" : [
                                       "available",
                                       "creating",
                                       "unavailable"
                                    ],
                                    "type" : "string"
                                 },
                                 "type" : {
                                    "description" : "Type of the Image",
                                    "enum" : [
                                       "system",
                                       "app",
                                       "snapshot",
                                       "backup",
                                       "temporary"
                                    ],
                                    "example" : "snapshot",
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "type",
                                 "status",
                                 "name",
                                 "description",
                                 "image_size",
                                 "disk_size",
                                 "created",
                                 "created_from",
                                 "bound_to",
                                 "os_flavor",
                                 "os_version",
                                 "protection",
                                 "deprecated",
                                 "deleted",
                                 "labels",
                                 "architecture"
                              ],
                              "type" : "object"
                           }
                        },
                        "type" : "object"
                     }
                  }
               },
               "description" : "The image key in the reply contains the modified Image object"
            }
         },
         "summary" : "Update an Image",
         "tags" : [
            "Images"
         ]
      }
   },
   "/images/{id}/actions" : {
      "get" : {
         "description" : "Returns all Action objects for an Image. You can sort the results by using the `sort` URI parameter, and filter them with the `status` parameter.",
         "parameters" : [
            {
               "description" : "ID of the Image",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "change_protection",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 13,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 4711,
                                    "type" : "image"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions for an Image",
         "tags" : [
            "Image Actions"
         ]
      }
   },
   "/images/{id}/actions/change_protection" : {
      "post" : {
         "description" : "Changes the protection configuration of the Image. Can only be used on snapshots.",
         "parameters" : [
            {
               "description" : "ID of the Image",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "delete" : {
                           "description" : "If true, prevents the snapshot from being deleted",
                           "example" : true,
                           "type" : "boolean"
                        }
                     },
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_protection",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "image"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_protection` Action"
            }
         },
         "summary" : "Change Image Protection",
         "tags" : [
            "Image Actions"
         ]
      }
   },
   "/images/{id}/actions/{action_id}" : {
      "get" : {
         "description" : "Returns a specific Action for an Image.",
         "parameters" : [
            {
               "description" : "ID of the Image",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "ID of the Action",
               "in" : "path",
               "name" : "action_id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_protection",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "image"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the Image Action"
            }
         },
         "summary" : "Get an Action for an Image",
         "tags" : [
            "Image Actions"
         ]
      }
   },
   "/isos" : {
      "get" : {
         "description" : "Returns all available ISO objects.",
         "parameters" : [
            {
               "description" : "Can be used to filter ISOs by their name. The response will only contain the ISO matching the specified name.",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Return only ISOs with the given architecture.",
               "in" : "query",
               "name" : "architecture",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Include Images with wildcard architecture (architecture is null). Works only if architecture filter is specified.",
               "in" : "query",
               "name" : "include_architecture_wildcard",
               "required" : false,
               "schema" : {
                  "type" : "boolean"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "isos" : {
                              "items" : {
                                 "properties" : {
                                    "architecture" : {
                                       "description" : "Type of cpu architecture this iso is compatible with. Null indicates no restriction on the architecture (wildcard).",
                                       "enum" : [
                                          "x86",
                                          "arm"
                                       ],
                                       "example" : "x86",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "deprecated" : {
                                       "description" : "ISO 8601 timestamp of deprecation, null if ISO is still available. After the deprecation time it will no longer be possible to attach the ISO to Servers.",
                                       "example" : "2018-02-28T00:00:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "description" : {
                                       "description" : "Description of the ISO",
                                       "example" : "FreeBSD 11.0 x64",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "name" : {
                                       "description" : "Unique identifier of the ISO. Only set for public ISOs",
                                       "example" : "FreeBSD-11.0-RELEASE-amd64-dvd1",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "type" : {
                                       "description" : "Type of the ISO",
                                       "enum" : [
                                          "public",
                                          "private"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "description",
                                    "type",
                                    "deprecated",
                                    "architecture"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "isos"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `isos` key in the reply contains an array of iso objects with this structure"
            }
         },
         "summary" : "Get all ISOs",
         "tags" : [
            "ISOs"
         ]
      }
   },
   "/isos/{id}" : {
      "get" : {
         "description" : "Returns a specific ISO object.",
         "parameters" : [
            {
               "description" : "ID of the ISO",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "iso" : {
                              "properties" : {
                                 "architecture" : {
                                    "description" : "Type of cpu architecture this iso is compatible with. Null indicates no restriction on the architecture (wildcard).",
                                    "enum" : [
                                       "x86",
                                       "arm"
                                    ],
                                    "example" : "x86",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "deprecated" : {
                                    "description" : "ISO 8601 timestamp of deprecation, null if ISO is still available. After the deprecation time it will no longer be possible to attach the ISO to Servers.",
                                    "example" : "2018-02-28T00:00:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the ISO",
                                    "example" : "FreeBSD 11.0 x64",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "name" : {
                                    "description" : "Unique identifier of the ISO. Only set for public ISOs",
                                    "example" : "FreeBSD-11.0-RELEASE-amd64-dvd1",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "type" : {
                                    "description" : "Type of the ISO",
                                    "enum" : [
                                       "public",
                                       "private"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "type",
                                 "deprecated",
                                 "architecture"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "iso"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `iso` key in the reply contains an array of ISO objects with this structure"
            }
         },
         "summary" : "Get an ISO",
         "tags" : [
            "ISOs"
         ]
      }
   },
   "/load_balancer_types" : {
      "get" : {
         "description" : "Gets all Load Balancer type objects.",
         "parameters" : [
            {
               "description" : "Can be used to filter Load Balancer types by their name. The response will only contain the Load Balancer type matching the specified name.",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "load_balancer_types" : {
                              "items" : {
                                 "properties" : {
                                    "deprecated" : {
                                       "description" : "Point in time when the Load Balancer type is deprecated (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:50:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "description" : {
                                       "description" : "Description of the Load Balancer type",
                                       "example" : "LB11",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Load Balancer type",
                                       "example" : 1,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "max_assigned_certificates" : {
                                       "description" : "Number of SSL Certificates that can be assigned to a single Load Balancer",
                                       "example" : 10,
                                       "type" : "number"
                                    },
                                    "max_connections" : {
                                       "description" : "Number of maximum simultaneous open connections",
                                       "example" : 20000,
                                       "type" : "number"
                                    },
                                    "max_services" : {
                                       "description" : "Number of services a Load Balancer of this type can have",
                                       "example" : 5,
                                       "type" : "number"
                                    },
                                    "max_targets" : {
                                       "description" : "Number of targets a single Load Balancer can have",
                                       "example" : 25,
                                       "type" : "number"
                                    },
                                    "name" : {
                                       "description" : "Unique identifier of the Load Balancer type",
                                       "example" : "lb11",
                                       "type" : "string"
                                    },
                                    "prices" : {
                                       "description" : "Prices in different network zones",
                                       "items" : {
                                          "properties" : {
                                             "location" : {
                                                "description" : "Name of the Location the price is for",
                                                "example" : "fsn1",
                                                "type" : "string"
                                             },
                                             "price_hourly" : {
                                                "description" : "Hourly costs for a Resource in this Location",
                                                "properties" : {
                                                   "gross" : {
                                                      "description" : "Price with VAT added",
                                                      "example" : "1.1900000000000000",
                                                      "type" : "string"
                                                   },
                                                   "net" : {
                                                      "description" : "Price without VAT",
                                                      "example" : "1.0000000000",
                                                      "type" : "string"
                                                   }
                                                },
                                                "required" : [
                                                   "net",
                                                   "gross"
                                                ],
                                                "type" : "object"
                                             },
                                             "price_monthly" : {
                                                "description" : "Monthly costs for a Resource in this Location",
                                                "properties" : {
                                                   "gross" : {
                                                      "description" : "Price with VAT added",
                                                      "example" : "1.1900000000000000",
                                                      "type" : "string"
                                                   },
                                                   "net" : {
                                                      "description" : "Price without VAT",
                                                      "example" : "1.0000000000",
                                                      "type" : "string"
                                                   }
                                                },
                                                "required" : [
                                                   "net",
                                                   "gross"
                                                ],
                                                "type" : "object"
                                             }
                                          },
                                          "required" : [
                                             "location",
                                             "price_hourly",
                                             "price_monthly"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "description",
                                    "max_connections",
                                    "max_services",
                                    "max_targets",
                                    "max_assigned_certificates",
                                    "deprecated",
                                    "prices"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           }
                        },
                        "required" : [
                           "load_balancer_types"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `load_balancer_types` key in the reply contains an array of Load Balancer type objects with this structure"
            }
         },
         "summary" : "Get all Load Balancer Types",
         "tags" : [
            "Load Balancer Types"
         ]
      }
   },
   "/load_balancer_types/{id}" : {
      "get" : {
         "description" : "Gets a specific Load Balancer type object.",
         "parameters" : [
            {
               "description" : "ID of Load Balancer type",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "load_balancer_type" : {
                              "properties" : {
                                 "deprecated" : {
                                    "description" : "Point in time when the Load Balancer type is deprecated (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:50:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Load Balancer type",
                                    "example" : "LB11",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Load Balancer type",
                                    "example" : 1,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "max_assigned_certificates" : {
                                    "description" : "Number of SSL Certificates that can be assigned to a single Load Balancer",
                                    "example" : 10,
                                    "type" : "number"
                                 },
                                 "max_connections" : {
                                    "description" : "Number of maximum simultaneous open connections",
                                    "example" : 20000,
                                    "type" : "number"
                                 },
                                 "max_services" : {
                                    "description" : "Number of services a Load Balancer of this type can have",
                                    "example" : 5,
                                    "type" : "number"
                                 },
                                 "max_targets" : {
                                    "description" : "Number of targets a single Load Balancer can have",
                                    "example" : 25,
                                    "type" : "number"
                                 },
                                 "name" : {
                                    "description" : "Unique identifier of the Load Balancer type",
                                    "example" : "lb11",
                                    "type" : "string"
                                 },
                                 "prices" : {
                                    "description" : "Prices in different network zones",
                                    "items" : {
                                       "properties" : {
                                          "location" : {
                                             "description" : "Name of the Location the price is for",
                                             "example" : "fsn1",
                                             "type" : "string"
                                          },
                                          "price_hourly" : {
                                             "description" : "Hourly costs for a Resource in this Location",
                                             "properties" : {
                                                "gross" : {
                                                   "description" : "Price with VAT added",
                                                   "example" : "1.1900000000000000",
                                                   "type" : "string"
                                                },
                                                "net" : {
                                                   "description" : "Price without VAT",
                                                   "example" : "1.0000000000",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "net",
                                                "gross"
                                             ],
                                             "type" : "object"
                                          },
                                          "price_monthly" : {
                                             "description" : "Monthly costs for a Resource in this Location",
                                             "properties" : {
                                                "gross" : {
                                                   "description" : "Price with VAT added",
                                                   "example" : "1.1900000000000000",
                                                   "type" : "string"
                                                },
                                                "net" : {
                                                   "description" : "Price without VAT",
                                                   "example" : "1.0000000000",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "net",
                                                "gross"
                                             ],
                                             "type" : "object"
                                          }
                                       },
                                       "required" : [
                                          "location",
                                          "price_hourly",
                                          "price_monthly"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "max_connections",
                                 "max_services",
                                 "max_targets",
                                 "max_assigned_certificates",
                                 "deprecated",
                                 "prices"
                              ],
                              "type" : "object"
                           }
                        },
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `load_balancer_type` key in the reply contains a Load Balancer type object with this structure"
            }
         },
         "summary" : "Get a Load Balancer Type",
         "tags" : [
            "Load Balancer Types"
         ]
      }
   },
   "/load_balancers" : {
      "get" : {
         "description" : "Gets all existing Load Balancers that you have available.",
         "parameters" : [
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "name",
                     "name:asc",
                     "name:desc",
                     "created",
                     "created:asc",
                     "created:desc"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by their name. The response will only contain the resources matching the specified name",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter resources by labels. The response will only contain resources matching the label selector.",
               "in" : "query",
               "name" : "label_selector",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "load_balancers" : {
                              "items" : {
                                 "properties" : {
                                    "algorithm" : {
                                       "description" : "Algorithm of the Load Balancer",
                                       "properties" : {
                                          "type" : {
                                             "description" : "Type of the algorithm",
                                             "enum" : [
                                                "round_robin",
                                                "least_connections"
                                             ],
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "created" : {
                                       "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "included_traffic" : {
                                       "description" : "Free Traffic for the current billing period in bytes",
                                       "example" : 10000,
                                       "type" : "integer"
                                    },
                                    "ingoing_traffic" : {
                                       "description" : "Inbound Traffic for the current billing period in bytes",
                                       "nullable" : true,
                                       "type" : "integer"
                                    },
                                    "labels" : {
                                       "additionalProperties" : {
                                          "type" : "string"
                                       },
                                       "description" : "User-defined labels (key-value pairs)",
                                       "type" : "object"
                                    },
                                    "load_balancer_type" : {
                                       "properties" : {
                                          "deprecated" : {
                                             "description" : "Point in time when the Load Balancer type is deprecated (in ISO-8601 format)",
                                             "example" : "2016-01-30T23:50:00+00:00",
                                             "nullable" : true,
                                             "type" : "string"
                                          },
                                          "description" : {
                                             "description" : "Description of the Load Balancer type",
                                             "example" : "LB11",
                                             "type" : "string"
                                          },
                                          "id" : {
                                             "description" : "ID of the Load Balancer type",
                                             "example" : 1,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "max_assigned_certificates" : {
                                             "description" : "Number of SSL Certificates that can be assigned to a single Load Balancer",
                                             "example" : 10,
                                             "type" : "number"
                                          },
                                          "max_connections" : {
                                             "description" : "Number of maximum simultaneous open connections",
                                             "example" : 20000,
                                             "type" : "number"
                                          },
                                          "max_services" : {
                                             "description" : "Number of services a Load Balancer of this type can have",
                                             "example" : 5,
                                             "type" : "number"
                                          },
                                          "max_targets" : {
                                             "description" : "Number of targets a single Load Balancer can have",
                                             "example" : 25,
                                             "type" : "number"
                                          },
                                          "name" : {
                                             "description" : "Unique identifier of the Load Balancer type",
                                             "example" : "lb11",
                                             "type" : "string"
                                          },
                                          "prices" : {
                                             "description" : "Prices in different network zones",
                                             "items" : {
                                                "properties" : {
                                                   "location" : {
                                                      "description" : "Name of the Location the price is for",
                                                      "example" : "fsn1",
                                                      "type" : "string"
                                                   },
                                                   "price_hourly" : {
                                                      "description" : "Hourly costs for a Resource in this Location",
                                                      "properties" : {
                                                         "gross" : {
                                                            "description" : "Price with VAT added",
                                                            "example" : "1.1900000000000000",
                                                            "type" : "string"
                                                         },
                                                         "net" : {
                                                            "description" : "Price without VAT",
                                                            "example" : "1.0000000000",
                                                            "type" : "string"
                                                         }
                                                      },
                                                      "required" : [
                                                         "net",
                                                         "gross"
                                                      ],
                                                      "type" : "object"
                                                   },
                                                   "price_monthly" : {
                                                      "description" : "Monthly costs for a Resource in this Location",
                                                      "properties" : {
                                                         "gross" : {
                                                            "description" : "Price with VAT added",
                                                            "example" : "1.1900000000000000",
                                                            "type" : "string"
                                                         },
                                                         "net" : {
                                                            "description" : "Price without VAT",
                                                            "example" : "1.0000000000",
                                                            "type" : "string"
                                                         }
                                                      },
                                                      "required" : [
                                                         "net",
                                                         "gross"
                                                      ],
                                                      "type" : "object"
                                                   }
                                                },
                                                "required" : [
                                                   "location",
                                                   "price_hourly",
                                                   "price_monthly"
                                                ],
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "name",
                                          "description",
                                          "max_connections",
                                          "max_services",
                                          "max_targets",
                                          "max_assigned_certificates",
                                          "deprecated",
                                          "prices"
                                       ],
                                       "type" : "object"
                                    },
                                    "location" : {
                                       "properties" : {
                                          "city" : {
                                             "description" : "City the Location is closest to",
                                             "example" : "Falkenstein",
                                             "type" : "string"
                                          },
                                          "country" : {
                                             "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                             "example" : "DE",
                                             "type" : "string"
                                          },
                                          "description" : {
                                             "description" : "Description of the Location",
                                             "example" : "Falkenstein DC Park 1",
                                             "type" : "string"
                                          },
                                          "id" : {
                                             "description" : "ID of the Location",
                                             "example" : 1,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "latitude" : {
                                             "description" : "Latitude of the city closest to the Location",
                                             "example" : 50.47612,
                                             "type" : "number"
                                          },
                                          "longitude" : {
                                             "description" : "Longitude of the city closest to the Location",
                                             "example" : 12.370071,
                                             "type" : "number"
                                          },
                                          "name" : {
                                             "description" : "Unique identifier of the Location",
                                             "example" : "fsn1",
                                             "type" : "string"
                                          },
                                          "network_zone" : {
                                             "description" : "Name of network zone this Location resides in",
                                             "example" : "eu-central",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "name",
                                          "description",
                                          "country",
                                          "city",
                                          "latitude",
                                          "longitude",
                                          "network_zone"
                                       ],
                                       "type" : "object"
                                    },
                                    "name" : {
                                       "description" : "Name of the Resource. Must be unique per Project.",
                                       "example" : "my-resource",
                                       "type" : "string"
                                    },
                                    "outgoing_traffic" : {
                                       "description" : "Outbound Traffic for the current billing period in bytes",
                                       "nullable" : true,
                                       "type" : "integer"
                                    },
                                    "private_net" : {
                                       "description" : "Private networks information",
                                       "items" : {
                                          "properties" : {
                                             "ip" : {
                                                "description" : "IP address (v4) of this Load Balancer in this Network",
                                                "example" : "10.0.0.2",
                                                "type" : "string"
                                             },
                                             "network" : {
                                                "description" : "ID of the Network",
                                                "example" : 4711,
                                                "format" : "int64",
                                                "type" : "integer"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "protection" : {
                                       "description" : "Protection configuration for the Resource",
                                       "properties" : {
                                          "delete" : {
                                             "description" : "If true, prevents the Resource from being deleted",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "delete"
                                       ],
                                       "type" : "object"
                                    },
                                    "public_net" : {
                                       "description" : "Public network information",
                                       "properties" : {
                                          "enabled" : {
                                             "description" : "Public Interface enabled or not",
                                             "type" : "boolean"
                                          },
                                          "ipv4" : {
                                             "description" : "IP address (v4)",
                                             "properties" : {
                                                "dns_ptr" : {
                                                   "description" : "Reverse DNS PTR entry for the IPv4 address of this Load Balancer",
                                                   "example" : "lb1.example.com",
                                                   "nullable" : true,
                                                   "type" : "string"
                                                },
                                                "ip" : {
                                                   "description" : "IP address (v4) of this Load Balancer",
                                                   "example" : "1.2.3.4",
                                                   "nullable" : true,
                                                   "type" : "string"
                                                }
                                             },
                                             "type" : "object"
                                          },
                                          "ipv6" : {
                                             "description" : "IP address (v6)",
                                             "properties" : {
                                                "dns_ptr" : {
                                                   "description" : "Reverse DNS PTR entry for the IPv6 address of this Load Balancer",
                                                   "example" : "lb1.example.com",
                                                   "nullable" : true,
                                                   "type" : "string"
                                                },
                                                "ip" : {
                                                   "description" : "IP address (v6) of this Load Balancer",
                                                   "example" : "2001:db8::1",
                                                   "nullable" : true,
                                                   "type" : "string"
                                                }
                                             },
                                             "type" : "object"
                                          }
                                       },
                                       "required" : [
                                          "enabled",
                                          "ipv4",
                                          "ipv6"
                                       ],
                                       "type" : "object"
                                    },
                                    "services" : {
                                       "description" : "List of services that belong to this Load Balancer",
                                       "items" : {
                                          "properties" : {
                                             "destination_port" : {
                                                "description" : "Port the Load Balancer will balance to",
                                                "example" : 80,
                                                "type" : "integer"
                                             },
                                             "health_check" : {
                                                "additionalProperties" : false,
                                                "description" : "Service health check",
                                                "properties" : {
                                                   "http" : {
                                                      "additionalProperties" : false,
                                                      "description" : "Additional configuration for protocol http",
                                                      "properties" : {
                                                         "domain" : {
                                                            "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                                            "example" : "example.com",
                                                            "nullable" : true,
                                                            "type" : "string"
                                                         },
                                                         "path" : {
                                                            "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                                            "example" : "/",
                                                            "type" : "string"
                                                         },
                                                         "response" : {
                                                            "description" : "String that must be contained in HTTP response in order to pass the health check",
                                                            "example" : "{\"status\": \"ok\"}",
                                                            "type" : "string"
                                                         },
                                                         "status_codes" : {
                                                            "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                                            "example" : [
                                                               "2??",
                                                               "3??"
                                                            ],
                                                            "items" : {
                                                               "type" : "string"
                                                            },
                                                            "type" : "array"
                                                         },
                                                         "tls" : {
                                                            "description" : "Use HTTPS for health check",
                                                            "example" : false,
                                                            "type" : "boolean"
                                                         }
                                                      },
                                                      "required" : [
                                                         "domain",
                                                         "path"
                                                      ],
                                                      "type" : "object"
                                                   },
                                                   "interval" : {
                                                      "description" : "Time interval in seconds health checks are performed",
                                                      "example" : 15,
                                                      "type" : "integer"
                                                   },
                                                   "port" : {
                                                      "description" : "Port the health check will be performed on",
                                                      "example" : 4711,
                                                      "type" : "integer"
                                                   },
                                                   "protocol" : {
                                                      "description" : "Type of the health check",
                                                      "enum" : [
                                                         "tcp",
                                                         "http"
                                                      ],
                                                      "example" : "http",
                                                      "type" : "string"
                                                   },
                                                   "retries" : {
                                                      "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                                      "example" : 3,
                                                      "type" : "integer"
                                                   },
                                                   "timeout" : {
                                                      "description" : "Time in seconds after an attempt is considered a timeout",
                                                      "example" : 10,
                                                      "type" : "integer"
                                                   }
                                                },
                                                "required" : [
                                                   "protocol",
                                                   "port",
                                                   "interval",
                                                   "timeout",
                                                   "retries"
                                                ],
                                                "title" : "LoadBalancerServiceHealthCheck",
                                                "type" : "object"
                                             },
                                             "http" : {
                                                "description" : "Configuration option for protocols http and https",
                                                "properties" : {
                                                   "certificates" : {
                                                      "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                                      "example" : [
                                                         897
                                                      ],
                                                      "items" : {
                                                         "type" : "integer"
                                                      },
                                                      "type" : "array"
                                                   },
                                                   "cookie_lifetime" : {
                                                      "description" : "Lifetime of the cookie used for sticky sessions",
                                                      "example" : 300,
                                                      "type" : "integer"
                                                   },
                                                   "cookie_name" : {
                                                      "description" : "Name of the cookie used for sticky sessions",
                                                      "example" : "HCLBSTICKY",
                                                      "type" : "string"
                                                   },
                                                   "redirect_http" : {
                                                      "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                                      "example" : true,
                                                      "type" : "boolean"
                                                   },
                                                   "sticky_sessions" : {
                                                      "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                                      "example" : true,
                                                      "type" : "boolean"
                                                   }
                                                },
                                                "title" : "LoadBalancerServiceHTTP",
                                                "type" : "object"
                                             },
                                             "listen_port" : {
                                                "description" : "Port the Load Balancer listens on",
                                                "example" : 443,
                                                "type" : "integer"
                                             },
                                             "protocol" : {
                                                "description" : "Protocol of the Load Balancer",
                                                "enum" : [
                                                   "tcp",
                                                   "http",
                                                   "https"
                                                ],
                                                "example" : "https",
                                                "type" : "string"
                                             },
                                             "proxyprotocol" : {
                                                "description" : "Is Proxyprotocol enabled or not",
                                                "example" : false,
                                                "type" : "boolean"
                                             }
                                          },
                                          "required" : [
                                             "protocol",
                                             "listen_port",
                                             "destination_port",
                                             "proxyprotocol",
                                             "health_check"
                                          ],
                                          "title" : "LoadBalancerService",
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "targets" : {
                                       "description" : "List of targets that belong to this Load Balancer",
                                       "items" : {
                                          "properties" : {
                                             "health_status" : {
                                                "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                                "items" : {
                                                   "properties" : {
                                                      "listen_port" : {
                                                         "example" : 443,
                                                         "type" : "integer"
                                                      },
                                                      "status" : {
                                                         "enum" : [
                                                            "healthy",
                                                            "unhealthy",
                                                            "unknown"
                                                         ],
                                                         "example" : "healthy",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "type" : "object"
                                                },
                                                "title" : "LoadBalancerTargetHealthStatus",
                                                "type" : "array"
                                             },
                                             "ip" : {
                                                "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                                                "properties" : {
                                                   "ip" : {
                                                      "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                                      "example" : "203.0.113.1",
                                                      "type" : "string"
                                                   }
                                                },
                                                "required" : [
                                                   "ip"
                                                ],
                                                "title" : "LoadBalancerTargetIP",
                                                "type" : "object"
                                             },
                                             "label_selector" : {
                                                "description" : "Label selector used to determine targets. Only present for target type \"label_selector\".",
                                                "properties" : {
                                                   "selector" : {
                                                      "description" : "Label selector",
                                                      "example" : "env=prod",
                                                      "type" : "string"
                                                   }
                                                },
                                                "required" : [
                                                   "selector"
                                                ],
                                                "title" : "LoadBalancerTargetLabelSelector",
                                                "type" : "object"
                                             },
                                             "server" : {
                                                "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                                "properties" : {
                                                   "id" : {
                                                      "description" : "ID of the Server",
                                                      "example" : 80,
                                                      "format" : "int64",
                                                      "type" : "integer"
                                                   }
                                                },
                                                "required" : [
                                                   "id"
                                                ],
                                                "title" : "LoadBalancerTargetServer",
                                                "type" : "object"
                                             },
                                             "targets" : {
                                                "description" : "List of resolved label selector target Servers. Only present for type \"label_selector\".",
                                                "items" : {
                                                   "properties" : {
                                                      "health_status" : {
                                                         "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                                         "items" : {
                                                            "properties" : {
                                                               "listen_port" : {
                                                                  "example" : 443,
                                                                  "type" : "integer"
                                                               },
                                                               "status" : {
                                                                  "enum" : [
                                                                     "healthy",
                                                                     "unhealthy",
                                                                     "unknown"
                                                                  ],
                                                                  "example" : "healthy",
                                                                  "type" : "string"
                                                               }
                                                            },
                                                            "type" : "object"
                                                         },
                                                         "title" : "LoadBalancerTargetHealthStatus",
                                                         "type" : "array"
                                                      },
                                                      "server" : {
                                                         "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                                         "properties" : {
                                                            "id" : {
                                                               "description" : "ID of the Server",
                                                               "example" : 80,
                                                               "format" : "int64",
                                                               "type" : "integer"
                                                            }
                                                         },
                                                         "required" : [
                                                            "id"
                                                         ],
                                                         "title" : "LoadBalancerTargetServer",
                                                         "type" : "object"
                                                      },
                                                      "type" : {
                                                         "description" : "Type of the resource. Here always \"server\".",
                                                         "example" : "server",
                                                         "type" : "string"
                                                      },
                                                      "use_private_ip" : {
                                                         "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                                         "title" : "LoadBalancerTargetUsePrivateIP",
                                                         "type" : "boolean"
                                                      }
                                                   },
                                                   "title" : "LoadBalancerTargetTarget",
                                                   "type" : "object"
                                                },
                                                "type" : "array"
                                             },
                                             "type" : {
                                                "description" : "Type of the resource",
                                                "enum" : [
                                                   "server",
                                                   "label_selector",
                                                   "ip"
                                                ],
                                                "type" : "string"
                                             },
                                             "use_private_ip" : {
                                                "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                                "title" : "LoadBalancerTargetUsePrivateIP",
                                                "type" : "boolean"
                                             }
                                          },
                                          "required" : [
                                             "type"
                                          ],
                                          "title" : "LoadBalancerTarget",
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "public_net",
                                    "private_net",
                                    "location",
                                    "load_balancer_type",
                                    "protection",
                                    "labels",
                                    "created",
                                    "services",
                                    "targets",
                                    "algorithm",
                                    "outgoing_traffic",
                                    "ingoing_traffic",
                                    "included_traffic"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "load_balancers"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `load_balancers` key contains a list of Load Balancers"
            }
         },
         "summary" : "Get all Load Balancers",
         "tags" : [
            "Load Balancers"
         ]
      },
      "post" : {
         "description" : "Creates a Load Balancer.\n\n#### Call specific error codes\n\n| Code                                    | Description                                                                                           |\n|-----------------------------------------|-------------------------------------------------------------------------------------------------------|\n| `cloud_resource_ip_not_allowed`         | The IP you are trying to add as a target belongs to a Hetzner Cloud resource                          |\n| `ip_not_owned`                          | The IP is not owned by the owner of the project of the Load Balancer                                  |\n| `load_balancer_not_attached_to_network` | The Load Balancer is not attached to a network                                                        |\n| `robot_unavailable`                     | Robot was not available. The caller may retry the operation after a short delay.                      |\n| `server_not_attached_to_network`        | The server you are trying to add as a target is not attached to the same network as the Load Balancer |\n| `source_port_already_used`              | The source port you are trying to add is already in use                                               |\n| `target_already_defined`                | The Load Balancer target you are trying to define is already defined                                  |\n",
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "algorithm" : {
                           "description" : "Algorithm of the Load Balancer",
                           "properties" : {
                              "type" : {
                                 "description" : "Type of the algorithm",
                                 "enum" : [
                                    "round_robin",
                                    "least_connections"
                                 ],
                                 "type" : "string"
                              }
                           },
                           "required" : [
                              "type"
                           ],
                           "title" : "LoadBalancerAlgorithm",
                           "type" : "object"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "properties" : {
                              "labelkey" : {
                                 "description" : "New label",
                                 "example" : "value",
                                 "type" : "string"
                              }
                           },
                           "type" : "object"
                        },
                        "load_balancer_type" : {
                           "description" : "ID or name of the Load Balancer type this Load Balancer should be created with",
                           "example" : "lb11",
                           "type" : "string"
                        },
                        "location" : {
                           "description" : "ID or name of Location to create Load Balancer in",
                           "type" : "string"
                        },
                        "name" : {
                           "description" : "Name of the Load Balancer",
                           "example" : "Web Frontend",
                           "type" : "string"
                        },
                        "network" : {
                           "description" : "ID of the network the Load Balancer should be attached to on creation",
                           "example" : 123,
                           "format" : "int64",
                           "type" : "integer"
                        },
                        "network_zone" : {
                           "description" : "Name of network zone",
                           "example" : "eu-central",
                           "type" : "string"
                        },
                        "public_interface" : {
                           "description" : "Enable or disable the public interface of the Load Balancer",
                           "example" : true,
                           "type" : "boolean"
                        },
                        "services" : {
                           "description" : "Array of services",
                           "items" : {
                              "properties" : {
                                 "destination_port" : {
                                    "description" : "Port the Load Balancer will balance to",
                                    "example" : 80,
                                    "type" : "integer"
                                 },
                                 "health_check" : {
                                    "additionalProperties" : false,
                                    "description" : "Service health check",
                                    "properties" : {
                                       "http" : {
                                          "additionalProperties" : false,
                                          "description" : "Additional configuration for protocol http",
                                          "properties" : {
                                             "domain" : {
                                                "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                                "example" : "example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "path" : {
                                                "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                                "example" : "/",
                                                "type" : "string"
                                             },
                                             "response" : {
                                                "description" : "String that must be contained in HTTP response in order to pass the health check",
                                                "example" : "{\"status\": \"ok\"}",
                                                "type" : "string"
                                             },
                                             "status_codes" : {
                                                "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                                "example" : [
                                                   "2??",
                                                   "3??"
                                                ],
                                                "items" : {
                                                   "type" : "string"
                                                },
                                                "type" : "array"
                                             },
                                             "tls" : {
                                                "description" : "Use HTTPS for health check",
                                                "example" : false,
                                                "type" : "boolean"
                                             }
                                          },
                                          "required" : [
                                             "domain",
                                             "path"
                                          ],
                                          "type" : "object"
                                       },
                                       "interval" : {
                                          "description" : "Time interval in seconds health checks are performed",
                                          "example" : 15,
                                          "type" : "integer"
                                       },
                                       "port" : {
                                          "description" : "Port the health check will be performed on",
                                          "example" : 4711,
                                          "type" : "integer"
                                       },
                                       "protocol" : {
                                          "description" : "Type of the health check",
                                          "enum" : [
                                             "tcp",
                                             "http"
                                          ],
                                          "example" : "http",
                                          "type" : "string"
                                       },
                                       "retries" : {
                                          "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                          "example" : 3,
                                          "type" : "integer"
                                       },
                                       "timeout" : {
                                          "description" : "Time in seconds after an attempt is considered a timeout",
                                          "example" : 10,
                                          "type" : "integer"
                                       }
                                    },
                                    "required" : [
                                       "protocol",
                                       "port",
                                       "interval",
                                       "timeout",
                                       "retries"
                                    ],
                                    "title" : "LoadBalancerServiceHealthCheck",
                                    "type" : "object"
                                 },
                                 "http" : {
                                    "description" : "Configuration option for protocols http and https",
                                    "properties" : {
                                       "certificates" : {
                                          "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                          "example" : [
                                             897
                                          ],
                                          "items" : {
                                             "type" : "integer"
                                          },
                                          "type" : "array"
                                       },
                                       "cookie_lifetime" : {
                                          "description" : "Lifetime of the cookie used for sticky sessions",
                                          "example" : 300,
                                          "type" : "integer"
                                       },
                                       "cookie_name" : {
                                          "description" : "Name of the cookie used for sticky sessions",
                                          "example" : "HCLBSTICKY",
                                          "type" : "string"
                                       },
                                       "redirect_http" : {
                                          "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                          "example" : true,
                                          "type" : "boolean"
                                       },
                                       "sticky_sessions" : {
                                          "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                          "example" : true,
                                          "type" : "boolean"
                                       }
                                    },
                                    "title" : "LoadBalancerServiceHTTP",
                                    "type" : "object"
                                 },
                                 "listen_port" : {
                                    "description" : "Port the Load Balancer listens on",
                                    "example" : 443,
                                    "type" : "integer"
                                 },
                                 "protocol" : {
                                    "description" : "Protocol of the Load Balancer",
                                    "enum" : [
                                       "tcp",
                                       "http",
                                       "https"
                                    ],
                                    "example" : "https",
                                    "type" : "string"
                                 },
                                 "proxyprotocol" : {
                                    "description" : "Is Proxyprotocol enabled or not",
                                    "example" : false,
                                    "type" : "boolean"
                                 }
                              },
                              "required" : [
                                 "protocol",
                                 "listen_port",
                                 "destination_port",
                                 "proxyprotocol",
                                 "health_check"
                              ],
                              "title" : "LoadBalancerService",
                              "type" : "object"
                           },
                           "type" : "array"
                        },
                        "targets" : {
                           "description" : "Array of targets",
                           "items" : {
                              "properties" : {
                                 "health_status" : {
                                    "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                    "items" : {
                                       "properties" : {
                                          "listen_port" : {
                                             "example" : 443,
                                             "type" : "integer"
                                          },
                                          "status" : {
                                             "enum" : [
                                                "healthy",
                                                "unhealthy",
                                                "unknown"
                                             ],
                                             "example" : "healthy",
                                             "type" : "string"
                                          }
                                       },
                                       "type" : "object"
                                    },
                                    "title" : "LoadBalancerTargetHealthStatus",
                                    "type" : "array"
                                 },
                                 "ip" : {
                                    "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                                    "properties" : {
                                       "ip" : {
                                          "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                          "example" : "203.0.113.1",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "ip"
                                    ],
                                    "title" : "LoadBalancerTargetIP",
                                    "type" : "object"
                                 },
                                 "label_selector" : {
                                    "description" : "Label selector used to determine targets. Only present for target type \"label_selector\".",
                                    "properties" : {
                                       "selector" : {
                                          "description" : "Label selector",
                                          "example" : "env=prod",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "selector"
                                    ],
                                    "title" : "LoadBalancerTargetLabelSelector",
                                    "type" : "object"
                                 },
                                 "server" : {
                                    "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                    "properties" : {
                                       "id" : {
                                          "description" : "ID of the Server",
                                          "example" : 80,
                                          "format" : "int64",
                                          "type" : "integer"
                                       }
                                    },
                                    "required" : [
                                       "id"
                                    ],
                                    "title" : "LoadBalancerTargetServer",
                                    "type" : "object"
                                 },
                                 "targets" : {
                                    "description" : "List of resolved label selector target Servers. Only present for type \"label_selector\".",
                                    "items" : {
                                       "properties" : {
                                          "health_status" : {
                                             "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                             "items" : {
                                                "properties" : {
                                                   "listen_port" : {
                                                      "example" : 443,
                                                      "type" : "integer"
                                                   },
                                                   "status" : {
                                                      "enum" : [
                                                         "healthy",
                                                         "unhealthy",
                                                         "unknown"
                                                      ],
                                                      "example" : "healthy",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "title" : "LoadBalancerTargetHealthStatus",
                                             "type" : "array"
                                          },
                                          "server" : {
                                             "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Server",
                                                   "example" : 80,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "title" : "LoadBalancerTargetServer",
                                             "type" : "object"
                                          },
                                          "type" : {
                                             "description" : "Type of the resource. Here always \"server\".",
                                             "example" : "server",
                                             "type" : "string"
                                          },
                                          "use_private_ip" : {
                                             "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                             "title" : "LoadBalancerTargetUsePrivateIP",
                                             "type" : "boolean"
                                          }
                                       },
                                       "title" : "LoadBalancerTargetTarget",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "type" : {
                                    "description" : "Type of the resource",
                                    "enum" : [
                                       "server",
                                       "label_selector",
                                       "ip"
                                    ],
                                    "type" : "string"
                                 },
                                 "use_private_ip" : {
                                    "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                    "title" : "LoadBalancerTargetUsePrivateIP",
                                    "type" : "boolean"
                                 }
                              },
                              "required" : [
                                 "type"
                              ],
                              "title" : "LoadBalancerTarget",
                              "type" : "object"
                           },
                           "type" : "array"
                        }
                     },
                     "required" : [
                        "name",
                        "load_balancer_type",
                        "algorithm"
                     ],
                     "title" : "CreateLoadBalancerRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "create_load_balancer",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        },
                        "load_balancer" : {
                           "algorithm" : {
                              "type" : "round_robin"
                           },
                           "created" : "2016-01-30T23:50:00+00:00",
                           "id" : 4711,
                           "included_traffic" : 654321,
                           "ingoing_traffic" : 123456,
                           "labels" : {
                              "env" : "dev"
                           },
                           "load_balancer_type" : {
                              "deprecated" : "2016-01-30T23:50:00+00:00",
                              "description" : "LB11",
                              "id" : 1,
                              "max_assigned_certificates" : 10,
                              "max_connections" : 20000,
                              "max_services" : 5,
                              "max_targets" : 25,
                              "name" : "lb11",
                              "prices" : [
                                 {
                                    "location" : "fsn1",
                                    "price_hourly" : {
                                       "gross" : "1.1900000000000000",
                                       "net" : "1.0000000000"
                                    },
                                    "price_monthly" : {
                                       "gross" : "1.1900000000000000",
                                       "net" : "1.0000000000"
                                    }
                                 }
                              ]
                           },
                           "location" : {
                              "city" : "Falkenstein",
                              "country" : "DE",
                              "description" : "Falkenstein DC Park 1",
                              "id" : 1,
                              "latitude" : 50.47612,
                              "longitude" : 12.370071,
                              "name" : "fsn1",
                              "network_zone" : "eu-central"
                           },
                           "name" : "Web Frontend",
                           "outgoing_traffic" : 123456,
                           "private_net" : [
                              {
                                 "ip" : "10.0.0.2",
                                 "network" : 4711
                              }
                           ],
                           "protection" : {
                              "delete" : false
                           },
                           "public_net" : {
                              "enabled" : false,
                              "ipv4" : {
                                 "ip" : "1.2.3.4"
                              },
                              "ipv6" : {
                                 "ip" : "2001:db8::1"
                              }
                           },
                           "services" : [
                              {
                                 "destination_port" : 80,
                                 "health_check" : {
                                    "http" : {
                                       "domain" : "example.com",
                                       "path" : "/",
                                       "response" : "{\"status\": \"ok\"}",
                                       "status_codes" : [
                                          "2??,3??"
                                       ],
                                       "tls" : false
                                    },
                                    "interval" : 15,
                                    "port" : 4711,
                                    "protocol" : "http",
                                    "retries" : 3,
                                    "timeout" : 10
                                 },
                                 "http" : {
                                    "certificates" : [
                                       897
                                    ],
                                    "cookie_lifetime" : 300,
                                    "cookie_name" : "HCLBSTICKY",
                                    "redirect_http" : true,
                                    "sticky_sessions" : true
                                 },
                                 "listen_port" : 443,
                                 "protocol" : "http",
                                 "proxyprotocol" : false
                              }
                           ],
                           "targets" : [
                              {
                                 "health_status" : [
                                    {
                                       "listen_port" : 443,
                                       "status" : "healthy"
                                    }
                                 ],
                                 "server" : {
                                    "id" : 80
                                 },
                                 "targets" : [
                                    {
                                       "health_status" : [
                                          {
                                             "listen_port" : 443,
                                             "status" : "healthy"
                                          }
                                       ],
                                       "label_selector" : null,
                                       "server" : {
                                          "id" : 80
                                       },
                                       "type" : "server",
                                       "use_private_ip" : true
                                    }
                                 ],
                                 "type" : "server",
                                 "use_private_ip" : true
                              }
                           ]
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           },
                           "load_balancer" : {
                              "properties" : {
                                 "algorithm" : {
                                    "description" : "Algorithm of the Load Balancer",
                                    "properties" : {
                                       "type" : {
                                          "description" : "Type of the algorithm",
                                          "enum" : [
                                             "round_robin",
                                             "least_connections"
                                          ],
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "type"
                                    ],
                                    "type" : "object"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "included_traffic" : {
                                    "description" : "Free Traffic for the current billing period in bytes",
                                    "example" : 10000,
                                    "type" : "integer"
                                 },
                                 "ingoing_traffic" : {
                                    "description" : "Inbound Traffic for the current billing period in bytes",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "load_balancer_type" : {
                                    "properties" : {
                                       "deprecated" : {
                                          "description" : "Point in time when the Load Balancer type is deprecated (in ISO-8601 format)",
                                          "example" : "2016-01-30T23:50:00+00:00",
                                          "nullable" : true,
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Load Balancer type",
                                          "example" : "LB11",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Load Balancer type",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "max_assigned_certificates" : {
                                          "description" : "Number of SSL Certificates that can be assigned to a single Load Balancer",
                                          "example" : 10,
                                          "type" : "number"
                                       },
                                       "max_connections" : {
                                          "description" : "Number of maximum simultaneous open connections",
                                          "example" : 20000,
                                          "type" : "number"
                                       },
                                       "max_services" : {
                                          "description" : "Number of services a Load Balancer of this type can have",
                                          "example" : 5,
                                          "type" : "number"
                                       },
                                       "max_targets" : {
                                          "description" : "Number of targets a single Load Balancer can have",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Load Balancer type",
                                          "example" : "lb11",
                                          "type" : "string"
                                       },
                                       "prices" : {
                                          "description" : "Prices in different network zones",
                                          "items" : {
                                             "properties" : {
                                                "location" : {
                                                   "description" : "Name of the Location the price is for",
                                                   "example" : "fsn1",
                                                   "type" : "string"
                                                },
                                                "price_hourly" : {
                                                   "description" : "Hourly costs for a Resource in this Location",
                                                   "properties" : {
                                                      "gross" : {
                                                         "description" : "Price with VAT added",
                                                         "example" : "1.1900000000000000",
                                                         "type" : "string"
                                                      },
                                                      "net" : {
                                                         "description" : "Price without VAT",
                                                         "example" : "1.0000000000",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "required" : [
                                                      "net",
                                                      "gross"
                                                   ],
                                                   "type" : "object"
                                                },
                                                "price_monthly" : {
                                                   "description" : "Monthly costs for a Resource in this Location",
                                                   "properties" : {
                                                      "gross" : {
                                                         "description" : "Price with VAT added",
                                                         "example" : "1.1900000000000000",
                                                         "type" : "string"
                                                      },
                                                      "net" : {
                                                         "description" : "Price without VAT",
                                                         "example" : "1.0000000000",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "required" : [
                                                      "net",
                                                      "gross"
                                                   ],
                                                   "type" : "object"
                                                }
                                             },
                                             "required" : [
                                                "location",
                                                "price_hourly",
                                                "price_monthly"
                                             ],
                                             "type" : "object"
                                          },
                                          "type" : "array"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "max_connections",
                                       "max_services",
                                       "max_targets",
                                       "max_assigned_certificates",
                                       "deprecated",
                                       "prices"
                                    ],
                                    "type" : "object"
                                 },
                                 "location" : {
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "outgoing_traffic" : {
                                    "description" : "Outbound Traffic for the current billing period in bytes",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "private_net" : {
                                    "description" : "Private networks information",
                                    "items" : {
                                       "properties" : {
                                          "ip" : {
                                             "description" : "IP address (v4) of this Load Balancer in this Network",
                                             "example" : "10.0.0.2",
                                             "type" : "string"
                                          },
                                          "network" : {
                                             "description" : "ID of the Network",
                                             "example" : 4711,
                                             "format" : "int64",
                                             "type" : "integer"
                                          }
                                       },
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "public_net" : {
                                    "description" : "Public network information",
                                    "properties" : {
                                       "enabled" : {
                                          "description" : "Public Interface enabled or not",
                                          "type" : "boolean"
                                       },
                                       "ipv4" : {
                                          "description" : "IP address (v4)",
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "Reverse DNS PTR entry for the IPv4 address of this Load Balancer",
                                                "example" : "lb1.example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "IP address (v4) of this Load Balancer",
                                                "example" : "1.2.3.4",
                                                "nullable" : true,
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "ipv6" : {
                                          "description" : "IP address (v6)",
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "Reverse DNS PTR entry for the IPv6 address of this Load Balancer",
                                                "example" : "lb1.example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "IP address (v6) of this Load Balancer",
                                                "example" : "2001:db8::1",
                                                "nullable" : true,
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       }
                                    },
                                    "required" : [
                                       "enabled",
                                       "ipv4",
                                       "ipv6"
                                    ],
                                    "type" : "object"
                                 },
                                 "services" : {
                                    "description" : "List of services that belong to this Load Balancer",
                                    "items" : {
                                       "properties" : {
                                          "destination_port" : {
                                             "description" : "Port the Load Balancer will balance to",
                                             "example" : 80,
                                             "type" : "integer"
                                          },
                                          "health_check" : {
                                             "additionalProperties" : false,
                                             "description" : "Service health check",
                                             "properties" : {
                                                "http" : {
                                                   "additionalProperties" : false,
                                                   "description" : "Additional configuration for protocol http",
                                                   "properties" : {
                                                      "domain" : {
                                                         "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                                         "example" : "example.com",
                                                         "nullable" : true,
                                                         "type" : "string"
                                                      },
                                                      "path" : {
                                                         "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                                         "example" : "/",
                                                         "type" : "string"
                                                      },
                                                      "response" : {
                                                         "description" : "String that must be contained in HTTP response in order to pass the health check",
                                                         "example" : "{\"status\": \"ok\"}",
                                                         "type" : "string"
                                                      },
                                                      "status_codes" : {
                                                         "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                                         "example" : [
                                                            "2??",
                                                            "3??"
                                                         ],
                                                         "items" : {
                                                            "type" : "string"
                                                         },
                                                         "type" : "array"
                                                      },
                                                      "tls" : {
                                                         "description" : "Use HTTPS for health check",
                                                         "example" : false,
                                                         "type" : "boolean"
                                                      }
                                                   },
                                                   "required" : [
                                                      "domain",
                                                      "path"
                                                   ],
                                                   "type" : "object"
                                                },
                                                "interval" : {
                                                   "description" : "Time interval in seconds health checks are performed",
                                                   "example" : 15,
                                                   "type" : "integer"
                                                },
                                                "port" : {
                                                   "description" : "Port the health check will be performed on",
                                                   "example" : 4711,
                                                   "type" : "integer"
                                                },
                                                "protocol" : {
                                                   "description" : "Type of the health check",
                                                   "enum" : [
                                                      "tcp",
                                                      "http"
                                                   ],
                                                   "example" : "http",
                                                   "type" : "string"
                                                },
                                                "retries" : {
                                                   "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                                   "example" : 3,
                                                   "type" : "integer"
                                                },
                                                "timeout" : {
                                                   "description" : "Time in seconds after an attempt is considered a timeout",
                                                   "example" : 10,
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "protocol",
                                                "port",
                                                "interval",
                                                "timeout",
                                                "retries"
                                             ],
                                             "title" : "LoadBalancerServiceHealthCheck",
                                             "type" : "object"
                                          },
                                          "http" : {
                                             "description" : "Configuration option for protocols http and https",
                                             "properties" : {
                                                "certificates" : {
                                                   "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                                   "example" : [
                                                      897
                                                   ],
                                                   "items" : {
                                                      "type" : "integer"
                                                   },
                                                   "type" : "array"
                                                },
                                                "cookie_lifetime" : {
                                                   "description" : "Lifetime of the cookie used for sticky sessions",
                                                   "example" : 300,
                                                   "type" : "integer"
                                                },
                                                "cookie_name" : {
                                                   "description" : "Name of the cookie used for sticky sessions",
                                                   "example" : "HCLBSTICKY",
                                                   "type" : "string"
                                                },
                                                "redirect_http" : {
                                                   "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                                   "example" : true,
                                                   "type" : "boolean"
                                                },
                                                "sticky_sessions" : {
                                                   "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                                   "example" : true,
                                                   "type" : "boolean"
                                                }
                                             },
                                             "title" : "LoadBalancerServiceHTTP",
                                             "type" : "object"
                                          },
                                          "listen_port" : {
                                             "description" : "Port the Load Balancer listens on",
                                             "example" : 443,
                                             "type" : "integer"
                                          },
                                          "protocol" : {
                                             "description" : "Protocol of the Load Balancer",
                                             "enum" : [
                                                "tcp",
                                                "http",
                                                "https"
                                             ],
                                             "example" : "https",
                                             "type" : "string"
                                          },
                                          "proxyprotocol" : {
                                             "description" : "Is Proxyprotocol enabled or not",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "protocol",
                                          "listen_port",
                                          "destination_port",
                                          "proxyprotocol",
                                          "health_check"
                                       ],
                                       "title" : "LoadBalancerService",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "targets" : {
                                    "description" : "List of targets that belong to this Load Balancer",
                                    "items" : {
                                       "properties" : {
                                          "health_status" : {
                                             "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                             "items" : {
                                                "properties" : {
                                                   "listen_port" : {
                                                      "example" : 443,
                                                      "type" : "integer"
                                                   },
                                                   "status" : {
                                                      "enum" : [
                                                         "healthy",
                                                         "unhealthy",
                                                         "unknown"
                                                      ],
                                                      "example" : "healthy",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "title" : "LoadBalancerTargetHealthStatus",
                                             "type" : "array"
                                          },
                                          "ip" : {
                                             "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                                             "properties" : {
                                                "ip" : {
                                                   "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                                   "example" : "203.0.113.1",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "ip"
                                             ],
                                             "title" : "LoadBalancerTargetIP",
                                             "type" : "object"
                                          },
                                          "label_selector" : {
                                             "description" : "Label selector used to determine targets. Only present for target type \"label_selector\".",
                                             "properties" : {
                                                "selector" : {
                                                   "description" : "Label selector",
                                                   "example" : "env=prod",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "selector"
                                             ],
                                             "title" : "LoadBalancerTargetLabelSelector",
                                             "type" : "object"
                                          },
                                          "server" : {
                                             "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Server",
                                                   "example" : 80,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "title" : "LoadBalancerTargetServer",
                                             "type" : "object"
                                          },
                                          "targets" : {
                                             "description" : "List of resolved label selector target Servers. Only present for type \"label_selector\".",
                                             "items" : {
                                                "properties" : {
                                                   "health_status" : {
                                                      "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                                      "items" : {
                                                         "properties" : {
                                                            "listen_port" : {
                                                               "example" : 443,
                                                               "type" : "integer"
                                                            },
                                                            "status" : {
                                                               "enum" : [
                                                                  "healthy",
                                                                  "unhealthy",
                                                                  "unknown"
                                                               ],
                                                               "example" : "healthy",
                                                               "type" : "string"
                                                            }
                                                         },
                                                         "type" : "object"
                                                      },
                                                      "title" : "LoadBalancerTargetHealthStatus",
                                                      "type" : "array"
                                                   },
                                                   "server" : {
                                                      "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                                      "properties" : {
                                                         "id" : {
                                                            "description" : "ID of the Server",
                                                            "example" : 80,
                                                            "format" : "int64",
                                                            "type" : "integer"
                                                         }
                                                      },
                                                      "required" : [
                                                         "id"
                                                      ],
                                                      "title" : "LoadBalancerTargetServer",
                                                      "type" : "object"
                                                   },
                                                   "type" : {
                                                      "description" : "Type of the resource. Here always \"server\".",
                                                      "example" : "server",
                                                      "type" : "string"
                                                   },
                                                   "use_private_ip" : {
                                                      "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                                      "title" : "LoadBalancerTargetUsePrivateIP",
                                                      "type" : "boolean"
                                                   }
                                                },
                                                "title" : "LoadBalancerTargetTarget",
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          },
                                          "type" : {
                                             "description" : "Type of the resource",
                                             "enum" : [
                                                "server",
                                                "label_selector",
                                                "ip"
                                             ],
                                             "type" : "string"
                                          },
                                          "use_private_ip" : {
                                             "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                             "title" : "LoadBalancerTargetUsePrivateIP",
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "title" : "LoadBalancerTarget",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "public_net",
                                 "private_net",
                                 "location",
                                 "load_balancer_type",
                                 "protection",
                                 "labels",
                                 "created",
                                 "services",
                                 "targets",
                                 "algorithm",
                                 "outgoing_traffic",
                                 "ingoing_traffic",
                                 "included_traffic"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "load_balancer",
                           "action"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `load_balancer` key contains the Load Balancer that was just created"
            }
         },
         "summary" : "Create a Load Balancer",
         "tags" : [
            "Load Balancers"
         ]
      }
   },
   "/load_balancers/{id}" : {
      "delete" : {
         "description" : "Deletes a Load Balancer.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "204" : {
               "description" : "Load Balancer deleted"
            }
         },
         "summary" : "Delete a Load Balancer",
         "tags" : [
            "Load Balancers"
         ]
      },
      "get" : {
         "description" : "Gets a specific Load Balancer object.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "load_balancer" : {
                              "properties" : {
                                 "algorithm" : {
                                    "description" : "Algorithm of the Load Balancer",
                                    "properties" : {
                                       "type" : {
                                          "description" : "Type of the algorithm",
                                          "enum" : [
                                             "round_robin",
                                             "least_connections"
                                          ],
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "type"
                                    ],
                                    "type" : "object"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "included_traffic" : {
                                    "description" : "Free Traffic for the current billing period in bytes",
                                    "example" : 10000,
                                    "type" : "integer"
                                 },
                                 "ingoing_traffic" : {
                                    "description" : "Inbound Traffic for the current billing period in bytes",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "load_balancer_type" : {
                                    "properties" : {
                                       "deprecated" : {
                                          "description" : "Point in time when the Load Balancer type is deprecated (in ISO-8601 format)",
                                          "example" : "2016-01-30T23:50:00+00:00",
                                          "nullable" : true,
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Load Balancer type",
                                          "example" : "LB11",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Load Balancer type",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "max_assigned_certificates" : {
                                          "description" : "Number of SSL Certificates that can be assigned to a single Load Balancer",
                                          "example" : 10,
                                          "type" : "number"
                                       },
                                       "max_connections" : {
                                          "description" : "Number of maximum simultaneous open connections",
                                          "example" : 20000,
                                          "type" : "number"
                                       },
                                       "max_services" : {
                                          "description" : "Number of services a Load Balancer of this type can have",
                                          "example" : 5,
                                          "type" : "number"
                                       },
                                       "max_targets" : {
                                          "description" : "Number of targets a single Load Balancer can have",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Load Balancer type",
                                          "example" : "lb11",
                                          "type" : "string"
                                       },
                                       "prices" : {
                                          "description" : "Prices in different network zones",
                                          "items" : {
                                             "properties" : {
                                                "location" : {
                                                   "description" : "Name of the Location the price is for",
                                                   "example" : "fsn1",
                                                   "type" : "string"
                                                },
                                                "price_hourly" : {
                                                   "description" : "Hourly costs for a Resource in this Location",
                                                   "properties" : {
                                                      "gross" : {
                                                         "description" : "Price with VAT added",
                                                         "example" : "1.1900000000000000",
                                                         "type" : "string"
                                                      },
                                                      "net" : {
                                                         "description" : "Price without VAT",
                                                         "example" : "1.0000000000",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "required" : [
                                                      "net",
                                                      "gross"
                                                   ],
                                                   "type" : "object"
                                                },
                                                "price_monthly" : {
                                                   "description" : "Monthly costs for a Resource in this Location",
                                                   "properties" : {
                                                      "gross" : {
                                                         "description" : "Price with VAT added",
                                                         "example" : "1.1900000000000000",
                                                         "type" : "string"
                                                      },
                                                      "net" : {
                                                         "description" : "Price without VAT",
                                                         "example" : "1.0000000000",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "required" : [
                                                      "net",
                                                      "gross"
                                                   ],
                                                   "type" : "object"
                                                }
                                             },
                                             "required" : [
                                                "location",
                                                "price_hourly",
                                                "price_monthly"
                                             ],
                                             "type" : "object"
                                          },
                                          "type" : "array"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "max_connections",
                                       "max_services",
                                       "max_targets",
                                       "max_assigned_certificates",
                                       "deprecated",
                                       "prices"
                                    ],
                                    "type" : "object"
                                 },
                                 "location" : {
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "outgoing_traffic" : {
                                    "description" : "Outbound Traffic for the current billing period in bytes",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "private_net" : {
                                    "description" : "Private networks information",
                                    "items" : {
                                       "properties" : {
                                          "ip" : {
                                             "description" : "IP address (v4) of this Load Balancer in this Network",
                                             "example" : "10.0.0.2",
                                             "type" : "string"
                                          },
                                          "network" : {
                                             "description" : "ID of the Network",
                                             "example" : 4711,
                                             "format" : "int64",
                                             "type" : "integer"
                                          }
                                       },
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "public_net" : {
                                    "description" : "Public network information",
                                    "properties" : {
                                       "enabled" : {
                                          "description" : "Public Interface enabled or not",
                                          "type" : "boolean"
                                       },
                                       "ipv4" : {
                                          "description" : "IP address (v4)",
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "Reverse DNS PTR entry for the IPv4 address of this Load Balancer",
                                                "example" : "lb1.example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "IP address (v4) of this Load Balancer",
                                                "example" : "1.2.3.4",
                                                "nullable" : true,
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "ipv6" : {
                                          "description" : "IP address (v6)",
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "Reverse DNS PTR entry for the IPv6 address of this Load Balancer",
                                                "example" : "lb1.example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "IP address (v6) of this Load Balancer",
                                                "example" : "2001:db8::1",
                                                "nullable" : true,
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       }
                                    },
                                    "required" : [
                                       "enabled",
                                       "ipv4",
                                       "ipv6"
                                    ],
                                    "type" : "object"
                                 },
                                 "services" : {
                                    "description" : "List of services that belong to this Load Balancer",
                                    "items" : {
                                       "properties" : {
                                          "destination_port" : {
                                             "description" : "Port the Load Balancer will balance to",
                                             "example" : 80,
                                             "type" : "integer"
                                          },
                                          "health_check" : {
                                             "additionalProperties" : false,
                                             "description" : "Service health check",
                                             "properties" : {
                                                "http" : {
                                                   "additionalProperties" : false,
                                                   "description" : "Additional configuration for protocol http",
                                                   "properties" : {
                                                      "domain" : {
                                                         "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                                         "example" : "example.com",
                                                         "nullable" : true,
                                                         "type" : "string"
                                                      },
                                                      "path" : {
                                                         "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                                         "example" : "/",
                                                         "type" : "string"
                                                      },
                                                      "response" : {
                                                         "description" : "String that must be contained in HTTP response in order to pass the health check",
                                                         "example" : "{\"status\": \"ok\"}",
                                                         "type" : "string"
                                                      },
                                                      "status_codes" : {
                                                         "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                                         "example" : [
                                                            "2??",
                                                            "3??"
                                                         ],
                                                         "items" : {
                                                            "type" : "string"
                                                         },
                                                         "type" : "array"
                                                      },
                                                      "tls" : {
                                                         "description" : "Use HTTPS for health check",
                                                         "example" : false,
                                                         "type" : "boolean"
                                                      }
                                                   },
                                                   "required" : [
                                                      "domain",
                                                      "path"
                                                   ],
                                                   "type" : "object"
                                                },
                                                "interval" : {
                                                   "description" : "Time interval in seconds health checks are performed",
                                                   "example" : 15,
                                                   "type" : "integer"
                                                },
                                                "port" : {
                                                   "description" : "Port the health check will be performed on",
                                                   "example" : 4711,
                                                   "type" : "integer"
                                                },
                                                "protocol" : {
                                                   "description" : "Type of the health check",
                                                   "enum" : [
                                                      "tcp",
                                                      "http"
                                                   ],
                                                   "example" : "http",
                                                   "type" : "string"
                                                },
                                                "retries" : {
                                                   "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                                   "example" : 3,
                                                   "type" : "integer"
                                                },
                                                "timeout" : {
                                                   "description" : "Time in seconds after an attempt is considered a timeout",
                                                   "example" : 10,
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "protocol",
                                                "port",
                                                "interval",
                                                "timeout",
                                                "retries"
                                             ],
                                             "title" : "LoadBalancerServiceHealthCheck",
                                             "type" : "object"
                                          },
                                          "http" : {
                                             "description" : "Configuration option for protocols http and https",
                                             "properties" : {
                                                "certificates" : {
                                                   "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                                   "example" : [
                                                      897
                                                   ],
                                                   "items" : {
                                                      "type" : "integer"
                                                   },
                                                   "type" : "array"
                                                },
                                                "cookie_lifetime" : {
                                                   "description" : "Lifetime of the cookie used for sticky sessions",
                                                   "example" : 300,
                                                   "type" : "integer"
                                                },
                                                "cookie_name" : {
                                                   "description" : "Name of the cookie used for sticky sessions",
                                                   "example" : "HCLBSTICKY",
                                                   "type" : "string"
                                                },
                                                "redirect_http" : {
                                                   "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                                   "example" : true,
                                                   "type" : "boolean"
                                                },
                                                "sticky_sessions" : {
                                                   "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                                   "example" : true,
                                                   "type" : "boolean"
                                                }
                                             },
                                             "title" : "LoadBalancerServiceHTTP",
                                             "type" : "object"
                                          },
                                          "listen_port" : {
                                             "description" : "Port the Load Balancer listens on",
                                             "example" : 443,
                                             "type" : "integer"
                                          },
                                          "protocol" : {
                                             "description" : "Protocol of the Load Balancer",
                                             "enum" : [
                                                "tcp",
                                                "http",
                                                "https"
                                             ],
                                             "example" : "https",
                                             "type" : "string"
                                          },
                                          "proxyprotocol" : {
                                             "description" : "Is Proxyprotocol enabled or not",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "protocol",
                                          "listen_port",
                                          "destination_port",
                                          "proxyprotocol",
                                          "health_check"
                                       ],
                                       "title" : "LoadBalancerService",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "targets" : {
                                    "description" : "List of targets that belong to this Load Balancer",
                                    "items" : {
                                       "properties" : {
                                          "health_status" : {
                                             "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                             "items" : {
                                                "properties" : {
                                                   "listen_port" : {
                                                      "example" : 443,
                                                      "type" : "integer"
                                                   },
                                                   "status" : {
                                                      "enum" : [
                                                         "healthy",
                                                         "unhealthy",
                                                         "unknown"
                                                      ],
                                                      "example" : "healthy",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "title" : "LoadBalancerTargetHealthStatus",
                                             "type" : "array"
                                          },
                                          "ip" : {
                                             "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                                             "properties" : {
                                                "ip" : {
                                                   "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                                   "example" : "203.0.113.1",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "ip"
                                             ],
                                             "title" : "LoadBalancerTargetIP",
                                             "type" : "object"
                                          },
                                          "label_selector" : {
                                             "description" : "Label selector used to determine targets. Only present for target type \"label_selector\".",
                                             "properties" : {
                                                "selector" : {
                                                   "description" : "Label selector",
                                                   "example" : "env=prod",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "selector"
                                             ],
                                             "title" : "LoadBalancerTargetLabelSelector",
                                             "type" : "object"
                                          },
                                          "server" : {
                                             "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Server",
                                                   "example" : 80,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "title" : "LoadBalancerTargetServer",
                                             "type" : "object"
                                          },
                                          "targets" : {
                                             "description" : "List of resolved label selector target Servers. Only present for type \"label_selector\".",
                                             "items" : {
                                                "properties" : {
                                                   "health_status" : {
                                                      "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                                      "items" : {
                                                         "properties" : {
                                                            "listen_port" : {
                                                               "example" : 443,
                                                               "type" : "integer"
                                                            },
                                                            "status" : {
                                                               "enum" : [
                                                                  "healthy",
                                                                  "unhealthy",
                                                                  "unknown"
                                                               ],
                                                               "example" : "healthy",
                                                               "type" : "string"
                                                            }
                                                         },
                                                         "type" : "object"
                                                      },
                                                      "title" : "LoadBalancerTargetHealthStatus",
                                                      "type" : "array"
                                                   },
                                                   "server" : {
                                                      "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                                      "properties" : {
                                                         "id" : {
                                                            "description" : "ID of the Server",
                                                            "example" : 80,
                                                            "format" : "int64",
                                                            "type" : "integer"
                                                         }
                                                      },
                                                      "required" : [
                                                         "id"
                                                      ],
                                                      "title" : "LoadBalancerTargetServer",
                                                      "type" : "object"
                                                   },
                                                   "type" : {
                                                      "description" : "Type of the resource. Here always \"server\".",
                                                      "example" : "server",
                                                      "type" : "string"
                                                   },
                                                   "use_private_ip" : {
                                                      "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                                      "title" : "LoadBalancerTargetUsePrivateIP",
                                                      "type" : "boolean"
                                                   }
                                                },
                                                "title" : "LoadBalancerTargetTarget",
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          },
                                          "type" : {
                                             "description" : "Type of the resource",
                                             "enum" : [
                                                "server",
                                                "label_selector",
                                                "ip"
                                             ],
                                             "type" : "string"
                                          },
                                          "use_private_ip" : {
                                             "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                             "title" : "LoadBalancerTargetUsePrivateIP",
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "title" : "LoadBalancerTarget",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "public_net",
                                 "private_net",
                                 "location",
                                 "load_balancer_type",
                                 "protection",
                                 "labels",
                                 "created",
                                 "services",
                                 "targets",
                                 "algorithm",
                                 "outgoing_traffic",
                                 "ingoing_traffic",
                                 "included_traffic"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "load_balancer"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `load_balancer` key contains the Load Balancer"
            }
         },
         "summary" : "Get a Load Balancer",
         "tags" : [
            "Load Balancers"
         ]
      },
      "put" : {
         "description" : "Updates a Load Balancer. You can update a Load Balancer’s name and a Load Balancer’s labels.\n\nNote that when updating labels, the Load Balancer’s current set of labels will be replaced with the labels provided in the request body. So, for example, if you want to add a new label, you have to provide all existing labels plus the new label in the request body.\n\nNote: if the Load Balancer object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "example" : {
                              "labelkey" : "value"
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "New Load Balancer name",
                           "example" : "new-name",
                           "type" : "string"
                        }
                     },
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "load_balancer" : {
                           "algorithm" : {
                              "type" : "round_robin"
                           },
                           "created" : "2016-01-30T23:50:00+00:00",
                           "id" : 4711,
                           "included_traffic" : 654321,
                           "ingoing_traffic" : 123456,
                           "labels" : {
                              "labelkey" : "value"
                           },
                           "load_balancer_type" : {
                              "deprecated" : "2016-01-30T23:50:00+00:00",
                              "description" : "LB11",
                              "id" : 1,
                              "max_assigned_certificates" : 10,
                              "max_connections" : 20000,
                              "max_services" : 5,
                              "max_targets" : 25,
                              "name" : "lb11",
                              "prices" : [
                                 {
                                    "location" : "fsn1",
                                    "price_hourly" : {
                                       "gross" : "1.1900000000000000",
                                       "net" : "1.0000000000"
                                    },
                                    "price_monthly" : {
                                       "gross" : "1.1900000000000000",
                                       "net" : "1.0000000000"
                                    }
                                 }
                              ]
                           },
                           "location" : {
                              "city" : "Falkenstein",
                              "country" : "DE",
                              "description" : "Falkenstein DC Park 1",
                              "id" : 1,
                              "latitude" : 50.47612,
                              "longitude" : 12.370071,
                              "name" : "fsn1",
                              "network_zone" : "eu-central"
                           },
                           "name" : "new-name",
                           "outgoing_traffic" : 123456,
                           "private_net" : [
                              {
                                 "ip" : "10.0.0.2",
                                 "network" : 4711
                              }
                           ],
                           "protection" : {
                              "delete" : false
                           },
                           "public_net" : {
                              "enabled" : false,
                              "ipv4" : {
                                 "ip" : "1.2.3.4"
                              },
                              "ipv6" : {
                                 "ip" : "2001:db8::1"
                              }
                           },
                           "services" : [
                              {
                                 "destination_port" : 80,
                                 "health_check" : {
                                    "http" : {
                                       "domain" : "example.com",
                                       "path" : "/",
                                       "response" : "{\"status\": \"ok\"}",
                                       "status_codes" : [
                                          "2??,3??"
                                       ],
                                       "tls" : false
                                    },
                                    "interval" : 15,
                                    "port" : 4711,
                                    "protocol" : "http",
                                    "retries" : 3,
                                    "timeout" : 10
                                 },
                                 "http" : {
                                    "certificates" : [
                                       897
                                    ],
                                    "cookie_lifetime" : 300,
                                    "cookie_name" : "HCLBSTICKY",
                                    "redirect_http" : true,
                                    "sticky_sessions" : true
                                 },
                                 "listen_port" : 443,
                                 "protocol" : "http",
                                 "proxyprotocol" : false
                              }
                           ],
                           "targets" : [
                              {
                                 "health_status" : [
                                    {
                                       "listen_port" : 443,
                                       "status" : "healthy"
                                    }
                                 ],
                                 "ip" : {
                                    "ip" : "203.0.113.1"
                                 },
                                 "label_selector" : {
                                    "selector" : "env=prod"
                                 },
                                 "server" : {
                                    "id" : 80
                                 },
                                 "targets" : [
                                    {
                                       "health_status" : [
                                          {
                                             "listen_port" : 443,
                                             "status" : "healthy"
                                          }
                                       ],
                                       "label_selector" : null,
                                       "server" : {
                                          "id" : 80
                                       },
                                       "type" : "server",
                                       "use_private_ip" : true
                                    }
                                 ],
                                 "type" : "server",
                                 "use_private_ip" : true
                              }
                           ]
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "load_balancer" : {
                              "properties" : {
                                 "algorithm" : {
                                    "description" : "Algorithm of the Load Balancer",
                                    "properties" : {
                                       "type" : {
                                          "description" : "Type of the algorithm",
                                          "enum" : [
                                             "round_robin",
                                             "least_connections"
                                          ],
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "type"
                                    ],
                                    "type" : "object"
                                 },
                                 "created" : {
                                    "description" : "Point in time when the Resource was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "included_traffic" : {
                                    "description" : "Free Traffic for the current billing period in bytes",
                                    "example" : 10000,
                                    "type" : "integer"
                                 },
                                 "ingoing_traffic" : {
                                    "description" : "Inbound Traffic for the current billing period in bytes",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "labels" : {
                                    "additionalProperties" : {
                                       "type" : "string"
                                    },
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "load_balancer_type" : {
                                    "properties" : {
                                       "deprecated" : {
                                          "description" : "Point in time when the Load Balancer type is deprecated (in ISO-8601 format)",
                                          "example" : "2016-01-30T23:50:00+00:00",
                                          "nullable" : true,
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Load Balancer type",
                                          "example" : "LB11",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Load Balancer type",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "max_assigned_certificates" : {
                                          "description" : "Number of SSL Certificates that can be assigned to a single Load Balancer",
                                          "example" : 10,
                                          "type" : "number"
                                       },
                                       "max_connections" : {
                                          "description" : "Number of maximum simultaneous open connections",
                                          "example" : 20000,
                                          "type" : "number"
                                       },
                                       "max_services" : {
                                          "description" : "Number of services a Load Balancer of this type can have",
                                          "example" : 5,
                                          "type" : "number"
                                       },
                                       "max_targets" : {
                                          "description" : "Number of targets a single Load Balancer can have",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Load Balancer type",
                                          "example" : "lb11",
                                          "type" : "string"
                                       },
                                       "prices" : {
                                          "description" : "Prices in different network zones",
                                          "items" : {
                                             "properties" : {
                                                "location" : {
                                                   "description" : "Name of the Location the price is for",
                                                   "example" : "fsn1",
                                                   "type" : "string"
                                                },
                                                "price_hourly" : {
                                                   "description" : "Hourly costs for a Resource in this Location",
                                                   "properties" : {
                                                      "gross" : {
                                                         "description" : "Price with VAT added",
                                                         "example" : "1.1900000000000000",
                                                         "type" : "string"
                                                      },
                                                      "net" : {
                                                         "description" : "Price without VAT",
                                                         "example" : "1.0000000000",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "required" : [
                                                      "net",
                                                      "gross"
                                                   ],
                                                   "type" : "object"
                                                },
                                                "price_monthly" : {
                                                   "description" : "Monthly costs for a Resource in this Location",
                                                   "properties" : {
                                                      "gross" : {
                                                         "description" : "Price with VAT added",
                                                         "example" : "1.1900000000000000",
                                                         "type" : "string"
                                                      },
                                                      "net" : {
                                                         "description" : "Price without VAT",
                                                         "example" : "1.0000000000",
                                                         "type" : "string"
                                                      }
                                                   },
                                                   "required" : [
                                                      "net",
                                                      "gross"
                                                   ],
                                                   "type" : "object"
                                                }
                                             },
                                             "required" : [
                                                "location",
                                                "price_hourly",
                                                "price_monthly"
                                             ],
                                             "type" : "object"
                                          },
                                          "type" : "array"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "max_connections",
                                       "max_services",
                                       "max_targets",
                                       "max_assigned_certificates",
                                       "deprecated",
                                       "prices"
                                    ],
                                    "type" : "object"
                                 },
                                 "location" : {
                                    "properties" : {
                                       "city" : {
                                          "description" : "City the Location is closest to",
                                          "example" : "Falkenstein",
                                          "type" : "string"
                                       },
                                       "country" : {
                                          "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                          "example" : "DE",
                                          "type" : "string"
                                       },
                                       "description" : {
                                          "description" : "Description of the Location",
                                          "example" : "Falkenstein DC Park 1",
                                          "type" : "string"
                                       },
                                       "id" : {
                                          "description" : "ID of the Location",
                                          "example" : 1,
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "latitude" : {
                                          "description" : "Latitude of the city closest to the Location",
                                          "example" : 50.47612,
                                          "type" : "number"
                                       },
                                       "longitude" : {
                                          "description" : "Longitude of the city closest to the Location",
                                          "example" : 12.370071,
                                          "type" : "number"
                                       },
                                       "name" : {
                                          "description" : "Unique identifier of the Location",
                                          "example" : "fsn1",
                                          "type" : "string"
                                       },
                                       "network_zone" : {
                                          "description" : "Name of network zone this Location resides in",
                                          "example" : "eu-central",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "id",
                                       "name",
                                       "description",
                                       "country",
                                       "city",
                                       "latitude",
                                       "longitude",
                                       "network_zone"
                                    ],
                                    "type" : "object"
                                 },
                                 "name" : {
                                    "description" : "Name of the Resource. Must be unique per Project.",
                                    "example" : "my-resource",
                                    "type" : "string"
                                 },
                                 "outgoing_traffic" : {
                                    "description" : "Outbound Traffic for the current billing period in bytes",
                                    "nullable" : true,
                                    "type" : "integer"
                                 },
                                 "private_net" : {
                                    "description" : "Private networks information",
                                    "items" : {
                                       "properties" : {
                                          "ip" : {
                                             "description" : "IP address (v4) of this Load Balancer in this Network",
                                             "example" : "10.0.0.2",
                                             "type" : "string"
                                          },
                                          "network" : {
                                             "description" : "ID of the Network",
                                             "example" : 4711,
                                             "format" : "int64",
                                             "type" : "integer"
                                          }
                                       },
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Resource",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Resource from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "public_net" : {
                                    "description" : "Public network information",
                                    "properties" : {
                                       "enabled" : {
                                          "description" : "Public Interface enabled or not",
                                          "type" : "boolean"
                                       },
                                       "ipv4" : {
                                          "description" : "IP address (v4)",
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "Reverse DNS PTR entry for the IPv4 address of this Load Balancer",
                                                "example" : "lb1.example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "IP address (v4) of this Load Balancer",
                                                "example" : "1.2.3.4",
                                                "nullable" : true,
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       },
                                       "ipv6" : {
                                          "description" : "IP address (v6)",
                                          "properties" : {
                                             "dns_ptr" : {
                                                "description" : "Reverse DNS PTR entry for the IPv6 address of this Load Balancer",
                                                "example" : "lb1.example.com",
                                                "nullable" : true,
                                                "type" : "string"
                                             },
                                             "ip" : {
                                                "description" : "IP address (v6) of this Load Balancer",
                                                "example" : "2001:db8::1",
                                                "nullable" : true,
                                                "type" : "string"
                                             }
                                          },
                                          "type" : "object"
                                       }
                                    },
                                    "required" : [
                                       "enabled",
                                       "ipv4",
                                       "ipv6"
                                    ],
                                    "type" : "object"
                                 },
                                 "services" : {
                                    "description" : "List of services that belong to this Load Balancer",
                                    "items" : {
                                       "properties" : {
                                          "destination_port" : {
                                             "description" : "Port the Load Balancer will balance to",
                                             "example" : 80,
                                             "type" : "integer"
                                          },
                                          "health_check" : {
                                             "additionalProperties" : false,
                                             "description" : "Service health check",
                                             "properties" : {
                                                "http" : {
                                                   "additionalProperties" : false,
                                                   "description" : "Additional configuration for protocol http",
                                                   "properties" : {
                                                      "domain" : {
                                                         "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                                         "example" : "example.com",
                                                         "nullable" : true,
                                                         "type" : "string"
                                                      },
                                                      "path" : {
                                                         "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                                         "example" : "/",
                                                         "type" : "string"
                                                      },
                                                      "response" : {
                                                         "description" : "String that must be contained in HTTP response in order to pass the health check",
                                                         "example" : "{\"status\": \"ok\"}",
                                                         "type" : "string"
                                                      },
                                                      "status_codes" : {
                                                         "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                                         "example" : [
                                                            "2??",
                                                            "3??"
                                                         ],
                                                         "items" : {
                                                            "type" : "string"
                                                         },
                                                         "type" : "array"
                                                      },
                                                      "tls" : {
                                                         "description" : "Use HTTPS for health check",
                                                         "example" : false,
                                                         "type" : "boolean"
                                                      }
                                                   },
                                                   "required" : [
                                                      "domain",
                                                      "path"
                                                   ],
                                                   "type" : "object"
                                                },
                                                "interval" : {
                                                   "description" : "Time interval in seconds health checks are performed",
                                                   "example" : 15,
                                                   "type" : "integer"
                                                },
                                                "port" : {
                                                   "description" : "Port the health check will be performed on",
                                                   "example" : 4711,
                                                   "type" : "integer"
                                                },
                                                "protocol" : {
                                                   "description" : "Type of the health check",
                                                   "enum" : [
                                                      "tcp",
                                                      "http"
                                                   ],
                                                   "example" : "http",
                                                   "type" : "string"
                                                },
                                                "retries" : {
                                                   "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                                   "example" : 3,
                                                   "type" : "integer"
                                                },
                                                "timeout" : {
                                                   "description" : "Time in seconds after an attempt is considered a timeout",
                                                   "example" : 10,
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "protocol",
                                                "port",
                                                "interval",
                                                "timeout",
                                                "retries"
                                             ],
                                             "title" : "LoadBalancerServiceHealthCheck",
                                             "type" : "object"
                                          },
                                          "http" : {
                                             "description" : "Configuration option for protocols http and https",
                                             "properties" : {
                                                "certificates" : {
                                                   "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                                   "example" : [
                                                      897
                                                   ],
                                                   "items" : {
                                                      "type" : "integer"
                                                   },
                                                   "type" : "array"
                                                },
                                                "cookie_lifetime" : {
                                                   "description" : "Lifetime of the cookie used for sticky sessions",
                                                   "example" : 300,
                                                   "type" : "integer"
                                                },
                                                "cookie_name" : {
                                                   "description" : "Name of the cookie used for sticky sessions",
                                                   "example" : "HCLBSTICKY",
                                                   "type" : "string"
                                                },
                                                "redirect_http" : {
                                                   "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                                   "example" : true,
                                                   "type" : "boolean"
                                                },
                                                "sticky_sessions" : {
                                                   "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                                   "example" : true,
                                                   "type" : "boolean"
                                                }
                                             },
                                             "title" : "LoadBalancerServiceHTTP",
                                             "type" : "object"
                                          },
                                          "listen_port" : {
                                             "description" : "Port the Load Balancer listens on",
                                             "example" : 443,
                                             "type" : "integer"
                                          },
                                          "protocol" : {
                                             "description" : "Protocol of the Load Balancer",
                                             "enum" : [
                                                "tcp",
                                                "http",
                                                "https"
                                             ],
                                             "example" : "https",
                                             "type" : "string"
                                          },
                                          "proxyprotocol" : {
                                             "description" : "Is Proxyprotocol enabled or not",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "protocol",
                                          "listen_port",
                                          "destination_port",
                                          "proxyprotocol",
                                          "health_check"
                                       ],
                                       "title" : "LoadBalancerService",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "targets" : {
                                    "description" : "List of targets that belong to this Load Balancer",
                                    "items" : {
                                       "properties" : {
                                          "health_status" : {
                                             "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                             "items" : {
                                                "properties" : {
                                                   "listen_port" : {
                                                      "example" : 443,
                                                      "type" : "integer"
                                                   },
                                                   "status" : {
                                                      "enum" : [
                                                         "healthy",
                                                         "unhealthy",
                                                         "unknown"
                                                      ],
                                                      "example" : "healthy",
                                                      "type" : "string"
                                                   }
                                                },
                                                "type" : "object"
                                             },
                                             "title" : "LoadBalancerTargetHealthStatus",
                                             "type" : "array"
                                          },
                                          "ip" : {
                                             "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                                             "properties" : {
                                                "ip" : {
                                                   "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                                   "example" : "203.0.113.1",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "ip"
                                             ],
                                             "title" : "LoadBalancerTargetIP",
                                             "type" : "object"
                                          },
                                          "label_selector" : {
                                             "description" : "Label selector used to determine targets. Only present for target type \"label_selector\".",
                                             "properties" : {
                                                "selector" : {
                                                   "description" : "Label selector",
                                                   "example" : "env=prod",
                                                   "type" : "string"
                                                }
                                             },
                                             "required" : [
                                                "selector"
                                             ],
                                             "title" : "LoadBalancerTargetLabelSelector",
                                             "type" : "object"
                                          },
                                          "server" : {
                                             "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                             "properties" : {
                                                "id" : {
                                                   "description" : "ID of the Server",
                                                   "example" : 80,
                                                   "format" : "int64",
                                                   "type" : "integer"
                                                }
                                             },
                                             "required" : [
                                                "id"
                                             ],
                                             "title" : "LoadBalancerTargetServer",
                                             "type" : "object"
                                          },
                                          "targets" : {
                                             "description" : "List of resolved label selector target Servers. Only present for type \"label_selector\".",
                                             "items" : {
                                                "properties" : {
                                                   "health_status" : {
                                                      "description" : "List of health statuses of the services on this target. Only present for target types \"server\" and \"ip\".",
                                                      "items" : {
                                                         "properties" : {
                                                            "listen_port" : {
                                                               "example" : 443,
                                                               "type" : "integer"
                                                            },
                                                            "status" : {
                                                               "enum" : [
                                                                  "healthy",
                                                                  "unhealthy",
                                                                  "unknown"
                                                               ],
                                                               "example" : "healthy",
                                                               "type" : "string"
                                                            }
                                                         },
                                                         "type" : "object"
                                                      },
                                                      "title" : "LoadBalancerTargetHealthStatus",
                                                      "type" : "array"
                                                   },
                                                   "server" : {
                                                      "description" : "Server where the traffic should be routed to. Only present for target type \"server\".",
                                                      "properties" : {
                                                         "id" : {
                                                            "description" : "ID of the Server",
                                                            "example" : 80,
                                                            "format" : "int64",
                                                            "type" : "integer"
                                                         }
                                                      },
                                                      "required" : [
                                                         "id"
                                                      ],
                                                      "title" : "LoadBalancerTargetServer",
                                                      "type" : "object"
                                                   },
                                                   "type" : {
                                                      "description" : "Type of the resource. Here always \"server\".",
                                                      "example" : "server",
                                                      "type" : "string"
                                                   },
                                                   "use_private_ip" : {
                                                      "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                                      "title" : "LoadBalancerTargetUsePrivateIP",
                                                      "type" : "boolean"
                                                   }
                                                },
                                                "title" : "LoadBalancerTargetTarget",
                                                "type" : "object"
                                             },
                                             "type" : "array"
                                          },
                                          "type" : {
                                             "description" : "Type of the resource",
                                             "enum" : [
                                                "server",
                                                "label_selector",
                                                "ip"
                                             ],
                                             "type" : "string"
                                          },
                                          "use_private_ip" : {
                                             "description" : "Use the private network IP instead of the public IP. Default value is false. Only present for target types \"server\" and \"label_selector\".",
                                             "title" : "LoadBalancerTargetUsePrivateIP",
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "type"
                                       ],
                                       "title" : "LoadBalancerTarget",
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "public_net",
                                 "private_net",
                                 "location",
                                 "load_balancer_type",
                                 "protection",
                                 "labels",
                                 "created",
                                 "services",
                                 "targets",
                                 "algorithm",
                                 "outgoing_traffic",
                                 "ingoing_traffic",
                                 "included_traffic"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "load_balancer"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `load_balancer` key contains the updated Load Balancer"
            }
         },
         "summary" : "Update a Load Balancer",
         "tags" : [
            "Load Balancers"
         ]
      }
   },
   "/load_balancers/{id}/actions" : {
      "get" : {
         "description" : "Returns all Action objects for a Load Balancer. You can sort the results by using the `sort` URI parameter, and filter them with the `status` parameter.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "add_service",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 13,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 4711,
                                    "type" : "load_balancer"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions for a Load Balancer",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/add_service" : {
      "post" : {
         "description" : "Adds a service to a Load Balancer.\n\n#### Call specific error codes\n\n| Code                       | Description                                             |\n|----------------------------|---------------------------------------------------------|\n| `source_port_already_used` | The source port you are trying to add is already in use |\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "destination_port" : {
                           "description" : "Port the Load Balancer will balance to",
                           "example" : 80,
                           "type" : "integer"
                        },
                        "health_check" : {
                           "additionalProperties" : false,
                           "description" : "Service health check",
                           "properties" : {
                              "http" : {
                                 "additionalProperties" : false,
                                 "description" : "Additional configuration for protocol http",
                                 "properties" : {
                                    "domain" : {
                                       "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                       "example" : "example.com",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "path" : {
                                       "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                       "example" : "/",
                                       "type" : "string"
                                    },
                                    "response" : {
                                       "description" : "String that must be contained in HTTP response in order to pass the health check",
                                       "example" : "{\"status\": \"ok\"}",
                                       "type" : "string"
                                    },
                                    "status_codes" : {
                                       "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                       "example" : [
                                          "2??",
                                          "3??"
                                       ],
                                       "items" : {
                                          "type" : "string"
                                       },
                                       "type" : "array"
                                    },
                                    "tls" : {
                                       "description" : "Use HTTPS for health check",
                                       "example" : false,
                                       "type" : "boolean"
                                    }
                                 },
                                 "required" : [
                                    "domain",
                                    "path"
                                 ],
                                 "type" : "object"
                              },
                              "interval" : {
                                 "description" : "Time interval in seconds health checks are performed",
                                 "example" : 15,
                                 "type" : "integer"
                              },
                              "port" : {
                                 "description" : "Port the health check will be performed on",
                                 "example" : 4711,
                                 "type" : "integer"
                              },
                              "protocol" : {
                                 "description" : "Type of the health check",
                                 "enum" : [
                                    "tcp",
                                    "http"
                                 ],
                                 "example" : "http",
                                 "type" : "string"
                              },
                              "retries" : {
                                 "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                 "example" : 3,
                                 "type" : "integer"
                              },
                              "timeout" : {
                                 "description" : "Time in seconds after an attempt is considered a timeout",
                                 "example" : 10,
                                 "type" : "integer"
                              }
                           },
                           "required" : [
                              "protocol",
                              "port",
                              "interval",
                              "timeout",
                              "retries"
                           ],
                           "title" : "LoadBalancerServiceHealthCheck",
                           "type" : "object"
                        },
                        "http" : {
                           "description" : "Configuration option for protocols http and https",
                           "properties" : {
                              "certificates" : {
                                 "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                 "example" : [
                                    897
                                 ],
                                 "items" : {
                                    "type" : "integer"
                                 },
                                 "type" : "array"
                              },
                              "cookie_lifetime" : {
                                 "description" : "Lifetime of the cookie used for sticky sessions",
                                 "example" : 300,
                                 "type" : "integer"
                              },
                              "cookie_name" : {
                                 "description" : "Name of the cookie used for sticky sessions",
                                 "example" : "HCLBSTICKY",
                                 "type" : "string"
                              },
                              "redirect_http" : {
                                 "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                 "example" : true,
                                 "type" : "boolean"
                              },
                              "sticky_sessions" : {
                                 "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                 "example" : true,
                                 "type" : "boolean"
                              }
                           },
                           "title" : "LoadBalancerServiceHTTP",
                           "type" : "object"
                        },
                        "listen_port" : {
                           "description" : "Port the Load Balancer listens on",
                           "example" : 443,
                           "type" : "integer"
                        },
                        "protocol" : {
                           "description" : "Protocol of the Load Balancer",
                           "enum" : [
                              "tcp",
                              "http",
                              "https"
                           ],
                           "example" : "https",
                           "type" : "string"
                        },
                        "proxyprotocol" : {
                           "description" : "Is Proxyprotocol enabled or not",
                           "example" : false,
                           "type" : "boolean"
                        }
                     },
                     "required" : [
                        "protocol",
                        "listen_port",
                        "destination_port",
                        "proxyprotocol",
                        "health_check"
                     ],
                     "title" : "LoadBalancerService",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "add_service",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `add_service` Action"
            }
         },
         "summary" : "Add Service",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/add_target" : {
      "post" : {
         "description" : "Adds a target to a Load Balancer.\n\n#### Call specific error codes\n\n| Code                                    | Description                                                                                           |\n|-----------------------------------------|-------------------------------------------------------------------------------------------------------|\n| `cloud_resource_ip_not_allowed`         | The IP you are trying to add as a target belongs to a Hetzner Cloud resource                          |\n| `ip_not_owned`                          | The IP you are trying to add as a target is not owned by the Project owner                            |\n| `load_balancer_not_attached_to_network` | The Load Balancer is not attached to a network                                                        |\n| `robot_unavailable`                     | Robot was not available. The caller may retry the operation after a short delay.                      |\n| `server_not_attached_to_network`        | The server you are trying to add as a target is not attached to the same network as the Load Balancer |\n| `target_already_defined`                | The Load Balancer target you are trying to define is already defined                                  |\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip" : {
                           "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                           "properties" : {
                              "ip" : {
                                 "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                 "example" : "203.0.113.1",
                                 "type" : "string"
                              }
                           },
                           "required" : [
                              "ip"
                           ],
                           "title" : "LoadBalancerTargetIP",
                           "type" : "object"
                        },
                        "label_selector" : {
                           "description" : "Configuration for label selector targets, required if type is `label_selector`",
                           "properties" : {
                              "selector" : {
                                 "description" : "Label selector",
                                 "example" : "env=prod",
                                 "type" : "string"
                              }
                           },
                           "required" : [
                              "selector"
                           ],
                           "type" : "object"
                        },
                        "server" : {
                           "additionalProperties" : false,
                           "description" : "Configuration for type Server, required if type is `server`",
                           "properties" : {
                              "id" : {
                                 "description" : "ID of the Server",
                                 "example" : 80,
                                 "format" : "int64",
                                 "type" : "integer"
                              }
                           },
                           "required" : [
                              "id"
                           ],
                           "type" : "object"
                        },
                        "type" : {
                           "description" : "Type of the resource",
                           "enum" : [
                              "server",
                              "label_selector",
                              "ip"
                           ],
                           "type" : "string"
                        },
                        "use_private_ip" : {
                           "description" : "Use the private network IP instead of the public IP of the Server, requires the Server and Load Balancer to be in the same network. Default value is false.",
                           "example" : true,
                           "type" : "boolean"
                        }
                     },
                     "required" : [
                        "type"
                     ],
                     "title" : "AddTargetRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "add_target",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `add_target` Action"
            }
         },
         "summary" : "Add Target",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/attach_to_network" : {
      "post" : {
         "description" : "Attach a Load Balancer to a Network.\n\n**Call specific error codes**\n\n| Code                             | Description                                                           |\n|----------------------------------|-----------------------------------------------------------------------|\n| `load_balancer_already_attached` | The Load Balancer is already attached to a network                    |\n| `ip_not_available`               | The provided Network IP is not available                              |\n| `no_subnet_available`            | No Subnet or IP is available for the Load Balancer within the network |\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip" : {
                           "description" : "IP to request to be assigned to this Load Balancer; if you do not provide this then you will be auto assigned an IP address",
                           "example" : "10.0.1.1",
                           "type" : "string"
                        },
                        "network" : {
                           "description" : "ID of an existing network to attach the Load Balancer to",
                           "example" : 4711,
                           "format" : "int64",
                           "type" : "integer"
                        }
                     },
                     "required" : [
                        "network"
                     ],
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "attach_to_network",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `attach_to_network` Action"
            }
         },
         "summary" : "Attach a Load Balancer to a Network",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/change_algorithm" : {
      "post" : {
         "description" : "Change the algorithm that determines to which target new requests are sent.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "type" : {
                           "description" : "Algorithm of the Load Balancer",
                           "enum" : [
                              "round_robin",
                              "least_connections"
                           ],
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "type"
                     ],
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_algorithm",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_algorithm` Action"
            }
         },
         "summary" : "Change Algorithm",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/change_dns_ptr" : {
      "post" : {
         "description" : "Changes the hostname that will appear when getting the hostname belonging to the public IPs (IPv4 and IPv6) of this Load Balancer.\n\nFloating IPs assigned to the Server are not affected by this.\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "dns_ptr" : {
                           "description" : "Hostname to set as a reverse DNS PTR entry",
                           "example" : "lb1.example.com",
                           "nullable" : true,
                           "type" : "string"
                        },
                        "ip" : {
                           "description" : "Public IP address for which the reverse DNS entry should be set",
                           "example" : "1.2.3.4",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "ip",
                        "dns_ptr"
                     ],
                     "title" : "ChangeLoadbalancerDnsPtrRequest",
                     "type" : "object"
                  }
               }
            },
            "description" : "Select the IP address for which to change the DNS entry by passing `ip`. It can be either IPv4 or IPv6. The target hostname is set by passing `dns_ptr`."
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_dns_ptr",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key in the reply contains an Action object with this structure"
            }
         },
         "summary" : "Change reverse DNS entry for this Load Balancer",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/change_protection" : {
      "post" : {
         "description" : "Changes the protection configuration of a Load Balancer.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "delete" : {
                           "description" : "If true, prevents the Load Balancer from being deleted",
                           "example" : true,
                           "type" : "boolean"
                        }
                     },
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_protection",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_protection` Action"
            }
         },
         "summary" : "Change Load Balancer Protection",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/change_type" : {
      "post" : {
         "description" : "Changes the type (Max Services, Max Targets and Max Connections) of a Load Balancer.\n\n**Call specific error codes**\n\n| Code                         | Description                                                     |\n|------------------------------|-----------------------------------------------------------------|\n| `invalid_load_balancer_type` | The Load Balancer type does not fit for the given Load Balancer |\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "load_balancer_type" : {
                           "description" : "ID or name of Load Balancer type the Load Balancer should migrate to",
                           "example" : "lb21",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "load_balancer_type"
                     ],
                     "title" : "ChangeTypeRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_load_balancer_type",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_load_balancer_type` Action"
            }
         },
         "summary" : "Change the Type of a Load Balancer",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/delete_service" : {
      "post" : {
         "description" : "Delete a service of a Load Balancer.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "listen_port" : {
                           "description" : "The listen port of the service you want to delete",
                           "example" : 4711,
                           "type" : "number"
                        }
                     },
                     "required" : [
                        "listen_port"
                     ],
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "delete_service",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `delete_service` Action"
            }
         },
         "summary" : "Delete Service",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/detach_from_network" : {
      "post" : {
         "description" : "Detaches a Load Balancer from a network.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "network" : {
                           "description" : "ID of an existing network to detach the Load Balancer from",
                           "example" : 4711,
                           "format" : "int64",
                           "type" : "integer"
                        }
                     },
                     "required" : [
                        "network"
                     ],
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "detach_from_network",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              },
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `detach_from_network` Action"
            }
         },
         "summary" : "Detach a Load Balancer from a Network",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/disable_public_interface" : {
      "post" : {
         "description" : "Disable the public interface of a Load Balancer. The Load Balancer will be not accessible from the internet via its public IPs.\n\n#### Call specific error codes\n\n| Code                                      | Description                                                                    |\n|-------------------------------------------|--------------------------------------------------------------------------------|\n| `load_balancer_not_attached_to_network`   |  The Load Balancer is not attached to a network                                |\n| `targets_without_use_private_ip`          | The Load Balancer has targets that use the public IP instead of the private IP |\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "disable_public_interface",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              },
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `disable_public_interface` Action"
            }
         },
         "summary" : "Disable the public interface of a Load Balancer",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/enable_public_interface" : {
      "post" : {
         "description" : "Enable the public interface of a Load Balancer. The Load Balancer will be accessible from the internet via its public IPs.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "enable_public_interface",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 42,
                                 "type" : "server"
                              },
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `enable_public_interface` Action"
            }
         },
         "summary" : "Enable the public interface of a Load Balancer",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/remove_target" : {
      "post" : {
         "description" : "Removes a target from a Load Balancer.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip" : {
                           "description" : "IP targets where the traffic should be routed to. It is only possible to use the (Public or vSwitch) IPs of Hetzner Online Root Servers belonging to the project owner. IPs belonging to other users are blocked. Additionally IPs belonging to services provided by Hetzner Cloud (Servers, Load Balancers, ...) are blocked as well. Only present for target type \"ip\".",
                           "properties" : {
                              "ip" : {
                                 "description" : "IP of a server that belongs to the same customer (public IPv4/IPv6) or private IP in a Subnetwork type vswitch.",
                                 "example" : "203.0.113.1",
                                 "type" : "string"
                              }
                           },
                           "required" : [
                              "ip"
                           ],
                           "title" : "LoadBalancerTargetIP",
                           "type" : "object"
                        },
                        "label_selector" : {
                           "description" : "Configuration for label selector targets, required if type is `label_selector`",
                           "properties" : {
                              "selector" : {
                                 "description" : "Label selector",
                                 "example" : "env=prod",
                                 "type" : "string"
                              }
                           },
                           "required" : [
                              "selector"
                           ],
                           "type" : "object"
                        },
                        "server" : {
                           "additionalProperties" : false,
                           "description" : "Configuration for type Server, required if type is `server`",
                           "properties" : {
                              "id" : {
                                 "description" : "ID of the Server",
                                 "example" : 80,
                                 "format" : "int64",
                                 "type" : "integer"
                              }
                           },
                           "required" : [
                              "id"
                           ],
                           "type" : "object"
                        },
                        "type" : {
                           "description" : "Type of the resource",
                           "enum" : [
                              "server",
                              "label_selector",
                              "ip"
                           ],
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "type"
                     ],
                     "title" : "RemoveTargetRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "remove_target",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `remove_target` Action"
            }
         },
         "summary" : "Remove Target",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/update_service" : {
      "post" : {
         "description" : "Updates a Load Balancer Service.\n\n#### Call specific error codes\n\n| Code                       | Description                                             |\n|----------------------------|---------------------------------------------------------|\n| `source_port_already_used` | The source port you are trying to add is already in use |\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "destination_port" : {
                           "description" : "Port the Load Balancer will balance to",
                           "example" : 80,
                           "type" : "integer"
                        },
                        "health_check" : {
                           "additionalProperties" : false,
                           "description" : "Service health check",
                           "properties" : {
                              "http" : {
                                 "additionalProperties" : false,
                                 "description" : "Additional configuration for protocol http",
                                 "properties" : {
                                    "domain" : {
                                       "description" : "Host header to send in the HTTP request. May not contain spaces, percent or backslash symbols. Can be null, in that case no host header is sent.",
                                       "example" : "example.com",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "path" : {
                                       "description" : "HTTP path to use for health checks. May not contain literal spaces, use percent-encoding instead.",
                                       "example" : "/",
                                       "type" : "string"
                                    },
                                    "response" : {
                                       "description" : "String that must be contained in HTTP response in order to pass the health check",
                                       "example" : "{\"status\": \"ok\"}",
                                       "type" : "string"
                                    },
                                    "status_codes" : {
                                       "description" : "List of returned HTTP status codes in order to pass the health check. Supports the wildcards `?` for exactly one character and `*` for multiple ones. The default is to pass the health check for any status code between 2?? and 3??.",
                                       "example" : [
                                          "2??",
                                          "3??"
                                       ],
                                       "items" : {
                                          "type" : "string"
                                       },
                                       "type" : "array"
                                    },
                                    "tls" : {
                                       "description" : "Use HTTPS for health check",
                                       "example" : false,
                                       "type" : "boolean"
                                    }
                                 },
                                 "required" : [
                                    "domain",
                                    "path"
                                 ],
                                 "type" : "object"
                              },
                              "interval" : {
                                 "description" : "Time interval in seconds health checks are performed",
                                 "example" : 15,
                                 "type" : "integer"
                              },
                              "port" : {
                                 "description" : "Port the health check will be performed on",
                                 "example" : 4711,
                                 "type" : "integer"
                              },
                              "protocol" : {
                                 "description" : "Type of the health check",
                                 "enum" : [
                                    "tcp",
                                    "http"
                                 ],
                                 "example" : "http",
                                 "type" : "string"
                              },
                              "retries" : {
                                 "description" : "Unsuccessful retries needed until a target is considered unhealthy; an unhealthy target needs the same number of successful retries to become healthy again",
                                 "example" : 3,
                                 "type" : "integer"
                              },
                              "timeout" : {
                                 "description" : "Time in seconds after an attempt is considered a timeout",
                                 "example" : 10,
                                 "type" : "integer"
                              }
                           },
                           "required" : [
                              "protocol",
                              "port",
                              "interval",
                              "timeout",
                              "retries"
                           ],
                           "title" : "LoadBalancerServiceHealthCheck",
                           "type" : "object"
                        },
                        "http" : {
                           "description" : "Configuration option for protocols http and https",
                           "properties" : {
                              "certificates" : {
                                 "description" : "IDs of the Certificates to use for TLS/SSL termination by the Load Balancer; empty for TLS/SSL passthrough or if `protocol` is \"http\"",
                                 "example" : [
                                    897
                                 ],
                                 "items" : {
                                    "type" : "integer"
                                 },
                                 "type" : "array"
                              },
                              "cookie_lifetime" : {
                                 "description" : "Lifetime of the cookie used for sticky sessions",
                                 "example" : 300,
                                 "type" : "integer"
                              },
                              "cookie_name" : {
                                 "description" : "Name of the cookie used for sticky sessions",
                                 "example" : "HCLBSTICKY",
                                 "type" : "string"
                              },
                              "redirect_http" : {
                                 "description" : "Redirect HTTP requests to HTTPS. Only available if protocol is \"https\". Default `false`",
                                 "example" : true,
                                 "type" : "boolean"
                              },
                              "sticky_sessions" : {
                                 "description" : "Use sticky sessions. Only available if protocol is \"http\" or \"https\". Default `false`",
                                 "example" : true,
                                 "type" : "boolean"
                              }
                           },
                           "title" : "LoadBalancerServiceHTTP",
                           "type" : "object"
                        },
                        "listen_port" : {
                           "description" : "Port the Load Balancer listens on",
                           "example" : 443,
                           "type" : "integer"
                        },
                        "protocol" : {
                           "description" : "Protocol of the Load Balancer",
                           "enum" : [
                              "tcp",
                              "http",
                              "https"
                           ],
                           "example" : "https",
                           "type" : "string"
                        },
                        "proxyprotocol" : {
                           "description" : "Is Proxyprotocol enabled or not",
                           "example" : false,
                           "type" : "boolean"
                        }
                     },
                     "required" : [
                        "protocol",
                        "listen_port",
                        "destination_port",
                        "proxyprotocol",
                        "health_check"
                     ],
                     "title" : "LoadBalancerService",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "update_service",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `update_service` Action"
            }
         },
         "summary" : "Update Service",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/actions/{action_id}" : {
      "get" : {
         "description" : "Returns a specific Action for a Load Balancer.",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "ID of the Action",
               "in" : "path",
               "name" : "action_id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_protection",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "load_balancer"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the Load Balancer Action"
            }
         },
         "summary" : "Get an Action for a Load Balancer",
         "tags" : [
            "Load Balancer Actions"
         ]
      }
   },
   "/load_balancers/{id}/metrics" : {
      "get" : {
         "description" : "You must specify the type of metric to get: `open_connections`, `connections_per_second`, `requests_per_second` or `bandwidth`. You can also specify more than one type by comma separation, e.g. `requests_per_second,bandwidth`.\n\nDepending on the type you will get different time series data:\n\n|Type | Timeseries | Unit | Description |\n|---- |------------|------|-------------|\n| open_connections | open_connections | number | Open connections |\n| connections_per_second | connections_per_second | connections/s | Connections per second |\n| requests_per_second | requests_per_second | requests/s | Requests per second |\n| bandwidth | bandwidth.in | bytes/s | Ingress bandwidth |\n|| bandwidth.out | bytes/s | Egress bandwidth |\n\nMetrics are available for the last 30 days only.\n\nIf you do not provide the step argument we will automatically adjust it so that 200 samples are returned.\n\nWe limit the number of samples to a maximum of 500 and will adjust the step parameter accordingly.\n",
         "parameters" : [
            {
               "description" : "ID of the Load Balancer",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Type of metrics to get",
               "in" : "query",
               "name" : "type",
               "required" : true,
               "schema" : {
                  "enum" : [
                     "open_connections",
                     "connections_per_second",
                     "requests_per_second",
                     "bandwidth"
                  ],
                  "type" : "string"
               }
            },
            {
               "description" : "Start of period to get Metrics for (in ISO-8601 format)",
               "in" : "query",
               "name" : "start",
               "required" : true,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "End of period to get Metrics for (in ISO-8601 format)",
               "in" : "query",
               "name" : "end",
               "required" : true,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Resolution of results in seconds",
               "in" : "query",
               "name" : "step",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "metrics" : {
                              "properties" : {
                                 "end" : {
                                    "description" : "End of period of metrics reported (in ISO-8601 format)",
                                    "example" : "2017-01-01T23:00:00+00:00",
                                    "type" : "string"
                                 },
                                 "start" : {
                                    "description" : "Start of period of metrics reported (in ISO-8601 format)",
                                    "example" : "2017-01-01T00:00:00+00:00",
                                    "type" : "string"
                                 },
                                 "step" : {
                                    "description" : "Resolution of results in seconds.",
                                    "example" : 60,
                                    "type" : "number"
                                 },
                                 "time_series" : {
                                    "additionalProperties" : {
                                       "properties" : {
                                          "values" : {
                                             "description" : "Metrics Timestamps with values",
                                             "items" : {
                                                "items" : {
                                                   "oneOf" : [
                                                      {
                                                         "type" : "number"
                                                      },
                                                      {
                                                         "type" : "string"
                                                      }
                                                   ]
                                                },
                                                "type" : "array"
                                             },
                                             "type" : "array"
                                          }
                                       },
                                       "required" : [
                                          "values"
                                       ],
                                       "type" : "object"
                                    },
                                    "description" : "Hash with timeseries information, containing the name of timeseries as key",
                                    "example" : {
                                       "name_of_timeseries" : {
                                          "values" : [
                                             [
                                                1435781470.622,
                                                "42"
                                             ],
                                             [
                                                1435781471.622,
                                                "43"
                                             ]
                                          ]
                                       }
                                    },
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "start",
                                 "end",
                                 "step",
                                 "time_series"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "metrics"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `metrics` key in the reply contains a metrics object with this structure"
            }
         },
         "summary" : "Get Metrics for a LoadBalancer",
         "tags" : [
            "Load Balancers"
         ]
      }
   },
   "/locations" : {
      "get" : {
         "description" : "Returns all Location objects.",
         "parameters" : [
            {
               "description" : "Can be used to filter Locations by their name. The response will only contain the Location matching the specified name.",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "locations" : {
                              "items" : {
                                 "properties" : {
                                    "city" : {
                                       "description" : "City the Location is closest to",
                                       "example" : "Falkenstein",
                                       "type" : "string"
                                    },
                                    "country" : {
                                       "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                       "example" : "DE",
                                       "type" : "string"
                                    },
                                    "description" : {
                                       "description" : "Description of the Location",
                                       "example" : "Falkenstein DC Park 1",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Location",
                                       "example" : 1,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "latitude" : {
                                       "description" : "Latitude of the city closest to the Location",
                                       "example" : 50.47612,
                                       "type" : "number"
                                    },
                                    "longitude" : {
                                       "description" : "Longitude of the city closest to the Location",
                                       "example" : 12.370071,
                                       "type" : "number"
                                    },
                                    "name" : {
                                       "description" : "Unique identifier of the Location",
                                       "example" : "fsn1",
                                       "type" : "string"
                                    },
                                    "network_zone" : {
                                       "description" : "Name of network zone this Location resides in",
                                       "example" : "eu-central",
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "description",
                                    "country",
                                    "city",
                                    "latitude",
                                    "longitude",
                                    "network_zone"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           }
                        },
                        "required" : [
                           "locations"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `locations` key in the reply contains an array of Location objects with this structure"
            }
         },
         "summary" : "Get all Locations",
         "tags" : [
            "Locations"
         ]
      }
   },
   "/locations/{id}" : {
      "get" : {
         "description" : "Returns a specific Location object.",
         "parameters" : [
            {
               "description" : "ID of Location",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "location" : {
                              "properties" : {
                                 "city" : {
                                    "description" : "City the Location is closest to",
                                    "example" : "Falkenstein",
                                    "type" : "string"
                                 },
                                 "country" : {
                                    "description" : "ISO 3166-1 alpha-2 code of the country the Location resides in",
                                    "example" : "DE",
                                    "type" : "string"
                                 },
                                 "description" : {
                                    "description" : "Description of the Location",
                                    "example" : "Falkenstein DC Park 1",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Location",
                                    "example" : 1,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "latitude" : {
                                    "description" : "Latitude of the city closest to the Location",
                                    "example" : 50.47612,
                                    "type" : "number"
                                 },
                                 "longitude" : {
                                    "description" : "Longitude of the city closest to the Location",
                                    "example" : 12.370071,
                                    "type" : "number"
                                 },
                                 "name" : {
                                    "description" : "Unique identifier of the Location",
                                    "example" : "fsn1",
                                    "type" : "string"
                                 },
                                 "network_zone" : {
                                    "description" : "Name of network zone this Location resides in",
                                    "example" : "eu-central",
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "description",
                                 "country",
                                 "city",
                                 "latitude",
                                 "longitude",
                                 "network_zone"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "location"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `location` key in the reply contains a Location object with this structure"
            }
         },
         "summary" : "Get a Location",
         "tags" : [
            "Locations"
         ]
      }
   },
   "/networks" : {
      "get" : {
         "description" : "Gets all existing networks that you have available.",
         "parameters" : [
            {
               "description" : "Can be used to filter networks by their name. The response will only contain the networks matching the specified name.",
               "in" : "query",
               "name" : "name",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used to filter networks by labels. The response will only contain networks with a matching label selector pattern.",
               "in" : "query",
               "name" : "label_selector",
               "required" : false,
               "schema" : {
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           },
                           "networks" : {
                              "items" : {
                                 "properties" : {
                                    "created" : {
                                       "description" : "Point in time when the Network was created (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:50:00+00:00",
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Network",
                                       "example" : 4711,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "ip_range" : {
                                       "description" : "IPv4 prefix of the whole Network",
                                       "example" : "10.0.0.0/16",
                                       "type" : "string"
                                    },
                                    "labels" : {
                                       "description" : "User-defined labels (key-value pairs)",
                                       "type" : "object"
                                    },
                                    "load_balancers" : {
                                       "description" : "Array of IDs of Load Balancers attached to this Network",
                                       "example" : [
                                          42
                                       ],
                                       "items" : {
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "type" : "array"
                                    },
                                    "name" : {
                                       "description" : "Name of the Network",
                                       "example" : "mynet",
                                       "type" : "string"
                                    },
                                    "protection" : {
                                       "description" : "Protection configuration for the Network",
                                       "properties" : {
                                          "delete" : {
                                             "description" : "If true, prevents the Network from being deleted",
                                             "example" : false,
                                             "type" : "boolean"
                                          }
                                       },
                                       "required" : [
                                          "delete"
                                       ],
                                       "type" : "object"
                                    },
                                    "routes" : {
                                       "description" : "Array of routes set in this Network",
                                       "items" : {
                                          "properties" : {
                                             "destination" : {
                                                "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                                                "example" : "10.100.1.0/24",
                                                "type" : "string"
                                             },
                                             "gateway" : {
                                                "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range and also cannot be 172.31.1.1 as this IP is being used as a gateway for the public network interface of Servers.",
                                                "example" : "10.0.1.1",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "destination",
                                             "gateway"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "servers" : {
                                       "description" : "Array of IDs of Servers attached to this Network",
                                       "example" : [
                                          42
                                       ],
                                       "items" : {
                                          "format" : "int64",
                                          "type" : "integer"
                                       },
                                       "type" : "array"
                                    },
                                    "subnets" : {
                                       "description" : "Array subnets allocated in this Network",
                                       "items" : {
                                          "properties" : {
                                             "gateway" : {
                                                "description" : "Gateway for Servers attached to this subnet. For subnets of type Server this is always the first IP of the network IP range.",
                                                "example" : "10.0.0.1",
                                                "type" : "string"
                                             },
                                             "ip_range" : {
                                                "description" : "Range to allocate IPs from. Must be a Subnet of the ip_range of the parent network object and must not overlap with any other subnets or with any destinations in routes. Minimum Network size is /30. We suggest that you pick a bigger Network with a /24 netmask.",
                                                "example" : "10.0.1.0/24",
                                                "type" : "string"
                                             },
                                             "network_zone" : {
                                                "description" : "Name of Network zone. The Location object contains the `network_zone` property each Location belongs to.",
                                                "example" : "eu-central",
                                                "type" : "string"
                                             },
                                             "type" : {
                                                "description" : "Type of Subnetwork",
                                                "enum" : [
                                                   "cloud",
                                                   "server",
                                                   "vswitch"
                                                ],
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "type",
                                             "network_zone",
                                             "gateway"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "name",
                                    "ip_range",
                                    "subnets",
                                    "routes",
                                    "servers",
                                    "protection",
                                    "labels",
                                    "created"
                                 ],
                                 "type" : "object"
                              },
                              "type" : "array"
                           }
                        },
                        "required" : [
                           "networks"
                        ],
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `networks` key contains a list of networks"
            }
         },
         "summary" : "Get all Networks",
         "tags" : [
            "Networks"
         ]
      },
      "post" : {
         "description" : "Creates a network with the specified `ip_range`.\n\nYou may specify one or more `subnets`. You can also add more Subnets later by using the [add subnet action](https://docs.hetzner.cloud/#network-actions-add-a-subnet-to-a-network). If you do not specify an `ip_range` in the subnet we will automatically pick the first available /24 range for you.\n\nYou may specify one or more routes in `routes`. You can also add more routes later by using the [add route action](https://docs.hetzner.cloud/#network-actions-add-a-route-to-a-network).\n",
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip_range" : {
                           "description" : "IP range of the whole network which must span all included subnets. Must be one of the private IPv4 ranges of RFC1918. Minimum network size is /24. We highly recommend that you pick a larger network with a /16 netmask.",
                           "example" : "10.0.0.0/16",
                           "type" : "string"
                        },
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "properties" : {
                              "labelkey" : {
                                 "description" : "New label",
                                 "example" : "value",
                                 "type" : "string"
                              }
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "Name of the network",
                           "example" : "mynet",
                           "type" : "string"
                        },
                        "routes" : {
                           "description" : "Array of routes set in this network. The destination of the route must be one of the private IPv4 ranges of RFC1918. The gateway must be a subnet/IP of the ip_range of the network object. The destination must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. The gateway cannot be the first IP of the networks ip_range and also cannot be 172.31.1.1.",
                           "items" : {
                              "properties" : {
                                 "destination" : {
                                    "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                                    "example" : "10.100.1.0/24",
                                    "type" : "string"
                                 },
                                 "gateway" : {
                                    "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range and also cannot be 172.31.1.1 as this IP is being used as a gateway for the public network interface of Servers.",
                                    "example" : "10.0.1.1",
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "destination",
                                 "gateway"
                              ],
                              "type" : "object"
                           },
                           "type" : "array"
                        },
                        "subnets" : {
                           "description" : "Array of subnets allocated.",
                           "items" : {
                              "properties" : {
                                 "ip_range" : {
                                    "description" : "Range to allocate IPs from. Must be a Subnet of the ip_range of the parent network object and must not overlap with any other subnets or with any destinations in routes. Minimum Network size is /30. We suggest that you pick a bigger Network with a /24 netmask.",
                                    "example" : "10.0.1.0/24",
                                    "type" : "string"
                                 },
                                 "network_zone" : {
                                    "description" : "Name of Network zone. The Location object contains the `network_zone` property each Location belongs to.",
                                    "example" : "eu-central",
                                    "type" : "string"
                                 },
                                 "type" : {
                                    "description" : "Type of Subnetwork",
                                    "enum" : [
                                       "cloud",
                                       "server",
                                       "vswitch"
                                    ],
                                    "type" : "string"
                                 },
                                 "vswitch_id" : {
                                    "description" : "ID of the robot vSwitch. Must be supplied if the subnet is of type vswitch.",
                                    "example" : 1000,
                                    "format" : "int64",
                                    "type" : "integer"
                                 }
                              },
                              "required" : [
                                 "type",
                                 "network_zone"
                              ],
                              "type" : "object"
                           },
                           "type" : "array"
                        }
                     },
                     "required" : [
                        "name",
                        "ip_range"
                     ],
                     "title" : "CreateNetworkRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "network" : {
                              "properties" : {
                                 "created" : {
                                    "description" : "Point in time when the Network was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:50:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Network",
                                    "example" : 4711,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "ip_range" : {
                                    "description" : "IPv4 prefix of the whole Network",
                                    "example" : "10.0.0.0/16",
                                    "type" : "string"
                                 },
                                 "labels" : {
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "load_balancers" : {
                                    "description" : "Array of IDs of Load Balancers attached to this Network",
                                    "example" : [
                                       42
                                    ],
                                    "items" : {
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "type" : "array"
                                 },
                                 "name" : {
                                    "description" : "Name of the Network",
                                    "example" : "mynet",
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Network",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Network from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "routes" : {
                                    "description" : "Array of routes set in this Network",
                                    "items" : {
                                       "properties" : {
                                          "destination" : {
                                             "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                                             "example" : "10.100.1.0/24",
                                             "type" : "string"
                                          },
                                          "gateway" : {
                                             "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range and also cannot be 172.31.1.1 as this IP is being used as a gateway for the public network interface of Servers.",
                                             "example" : "10.0.1.1",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "destination",
                                          "gateway"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "servers" : {
                                    "description" : "Array of IDs of Servers attached to this Network",
                                    "example" : [
                                       42
                                    ],
                                    "items" : {
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "type" : "array"
                                 },
                                 "subnets" : {
                                    "description" : "Array subnets allocated in this Network",
                                    "items" : {
                                       "properties" : {
                                          "gateway" : {
                                             "description" : "Gateway for Servers attached to this subnet. For subnets of type Server this is always the first IP of the network IP range.",
                                             "example" : "10.0.0.1",
                                             "type" : "string"
                                          },
                                          "ip_range" : {
                                             "description" : "Range to allocate IPs from. Must be a Subnet of the ip_range of the parent network object and must not overlap with any other subnets or with any destinations in routes. Minimum Network size is /30. We suggest that you pick a bigger Network with a /24 netmask.",
                                             "example" : "10.0.1.0/24",
                                             "type" : "string"
                                          },
                                          "network_zone" : {
                                             "description" : "Name of Network zone. The Location object contains the `network_zone` property each Location belongs to.",
                                             "example" : "eu-central",
                                             "type" : "string"
                                          },
                                          "type" : {
                                             "description" : "Type of Subnetwork",
                                             "enum" : [
                                                "cloud",
                                                "server",
                                                "vswitch"
                                             ],
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type",
                                          "network_zone",
                                          "gateway"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "ip_range",
                                 "subnets",
                                 "routes",
                                 "servers",
                                 "protection",
                                 "labels",
                                 "created"
                              ],
                              "type" : "object"
                           }
                        },
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `network` key contains the network that was just created"
            }
         },
         "summary" : "Create a Network",
         "tags" : [
            "Networks"
         ]
      }
   },
   "/networks/{id}" : {
      "delete" : {
         "description" : "Deletes a network. If there are Servers attached they will be detached in the background.\n\nNote: if the network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "204" : {
               "description" : "Network deleted"
            }
         },
         "summary" : "Delete a Network",
         "tags" : [
            "Networks"
         ]
      },
      "get" : {
         "description" : "Gets a specific network object.",
         "parameters" : [
            {
               "description" : "ID of the network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "schema" : {
                        "properties" : {
                           "network" : {
                              "properties" : {
                                 "created" : {
                                    "description" : "Point in time when the Network was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:50:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Network",
                                    "example" : 4711,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "ip_range" : {
                                    "description" : "IPv4 prefix of the whole Network",
                                    "example" : "10.0.0.0/16",
                                    "type" : "string"
                                 },
                                 "labels" : {
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "load_balancers" : {
                                    "description" : "Array of IDs of Load Balancers attached to this Network",
                                    "example" : [
                                       42
                                    ],
                                    "items" : {
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "type" : "array"
                                 },
                                 "name" : {
                                    "description" : "Name of the Network",
                                    "example" : "mynet",
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Network",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Network from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "routes" : {
                                    "description" : "Array of routes set in this Network",
                                    "items" : {
                                       "properties" : {
                                          "destination" : {
                                             "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                                             "example" : "10.100.1.0/24",
                                             "type" : "string"
                                          },
                                          "gateway" : {
                                             "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range and also cannot be 172.31.1.1 as this IP is being used as a gateway for the public network interface of Servers.",
                                             "example" : "10.0.1.1",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "destination",
                                          "gateway"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "servers" : {
                                    "description" : "Array of IDs of Servers attached to this Network",
                                    "example" : [
                                       42
                                    ],
                                    "items" : {
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "type" : "array"
                                 },
                                 "subnets" : {
                                    "description" : "Array subnets allocated in this Network",
                                    "items" : {
                                       "properties" : {
                                          "gateway" : {
                                             "description" : "Gateway for Servers attached to this subnet. For subnets of type Server this is always the first IP of the network IP range.",
                                             "example" : "10.0.0.1",
                                             "type" : "string"
                                          },
                                          "ip_range" : {
                                             "description" : "Range to allocate IPs from. Must be a Subnet of the ip_range of the parent network object and must not overlap with any other subnets or with any destinations in routes. Minimum Network size is /30. We suggest that you pick a bigger Network with a /24 netmask.",
                                             "example" : "10.0.1.0/24",
                                             "type" : "string"
                                          },
                                          "network_zone" : {
                                             "description" : "Name of Network zone. The Location object contains the `network_zone` property each Location belongs to.",
                                             "example" : "eu-central",
                                             "type" : "string"
                                          },
                                          "type" : {
                                             "description" : "Type of Subnetwork",
                                             "enum" : [
                                                "cloud",
                                                "server",
                                                "vswitch"
                                             ],
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type",
                                          "network_zone",
                                          "gateway"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "ip_range",
                                 "subnets",
                                 "routes",
                                 "servers",
                                 "protection",
                                 "labels",
                                 "created"
                              ],
                              "type" : "object"
                           }
                        },
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `network` key contains the network"
            }
         },
         "summary" : "Get a Network",
         "tags" : [
            "Networks"
         ]
      },
      "put" : {
         "description" : "Updates the network properties.\n\nNote that when updating labels, the network’s current set of labels will be replaced with the labels provided in the request body. So, for example, if you want to add a new label, you have to provide all existing labels plus the new label in the request body.\n\nNote: if the network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "labels" : {
                           "description" : "User-defined labels (key-value pairs)",
                           "properties" : {
                              "labelkey" : {
                                 "example" : "value",
                                 "type" : "string"
                              }
                           },
                           "type" : "object"
                        },
                        "name" : {
                           "description" : "New network name",
                           "example" : "new-name",
                           "type" : "string"
                        }
                     },
                     "title" : "UpdateNetworkRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "network" : {
                           "created" : "2016-01-30T23:50:00+00:00",
                           "id" : 4711,
                           "ip_range" : "10.0.0.0/16",
                           "labels" : {
                              "labelkey" : "value"
                           },
                           "load_balancers" : [
                              42
                           ],
                           "name" : "new-name",
                           "protection" : {
                              "delete" : false
                           },
                           "routes" : [
                              {
                                 "destination" : "10.100.1.0/24",
                                 "gateway" : "10.0.1.1"
                              }
                           ],
                           "servers" : [
                              42
                           ],
                           "subnets" : [
                              {
                                 "gateway" : "10.0.0.1",
                                 "ip_range" : "10.0.1.0/24",
                                 "network_zone" : "eu-central",
                                 "type" : "cloud"
                              }
                           ]
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "network" : {
                              "properties" : {
                                 "created" : {
                                    "description" : "Point in time when the Network was created (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:50:00+00:00",
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Network",
                                    "example" : 4711,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "ip_range" : {
                                    "description" : "IPv4 prefix of the whole Network",
                                    "example" : "10.0.0.0/16",
                                    "type" : "string"
                                 },
                                 "labels" : {
                                    "description" : "User-defined labels (key-value pairs)",
                                    "type" : "object"
                                 },
                                 "load_balancers" : {
                                    "description" : "Array of IDs of Load Balancers attached to this Network",
                                    "example" : [
                                       42
                                    ],
                                    "items" : {
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "type" : "array"
                                 },
                                 "name" : {
                                    "description" : "Name of the Network",
                                    "example" : "mynet",
                                    "type" : "string"
                                 },
                                 "protection" : {
                                    "description" : "Protection configuration for the Network",
                                    "properties" : {
                                       "delete" : {
                                          "description" : "If true, prevents the Network from being deleted",
                                          "example" : false,
                                          "type" : "boolean"
                                       }
                                    },
                                    "required" : [
                                       "delete"
                                    ],
                                    "type" : "object"
                                 },
                                 "routes" : {
                                    "description" : "Array of routes set in this Network",
                                    "items" : {
                                       "properties" : {
                                          "destination" : {
                                             "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                                             "example" : "10.100.1.0/24",
                                             "type" : "string"
                                          },
                                          "gateway" : {
                                             "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range and also cannot be 172.31.1.1 as this IP is being used as a gateway for the public network interface of Servers.",
                                             "example" : "10.0.1.1",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "destination",
                                          "gateway"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "servers" : {
                                    "description" : "Array of IDs of Servers attached to this Network",
                                    "example" : [
                                       42
                                    ],
                                    "items" : {
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "type" : "array"
                                 },
                                 "subnets" : {
                                    "description" : "Array subnets allocated in this Network",
                                    "items" : {
                                       "properties" : {
                                          "gateway" : {
                                             "description" : "Gateway for Servers attached to this subnet. For subnets of type Server this is always the first IP of the network IP range.",
                                             "example" : "10.0.0.1",
                                             "type" : "string"
                                          },
                                          "ip_range" : {
                                             "description" : "Range to allocate IPs from. Must be a Subnet of the ip_range of the parent network object and must not overlap with any other subnets or with any destinations in routes. Minimum Network size is /30. We suggest that you pick a bigger Network with a /24 netmask.",
                                             "example" : "10.0.1.0/24",
                                             "type" : "string"
                                          },
                                          "network_zone" : {
                                             "description" : "Name of Network zone. The Location object contains the `network_zone` property each Location belongs to.",
                                             "example" : "eu-central",
                                             "type" : "string"
                                          },
                                          "type" : {
                                             "description" : "Type of Subnetwork",
                                             "enum" : [
                                                "cloud",
                                                "server",
                                                "vswitch"
                                             ],
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "type",
                                          "network_zone",
                                          "gateway"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "name",
                                 "ip_range",
                                 "subnets",
                                 "routes",
                                 "servers",
                                 "protection",
                                 "labels",
                                 "created"
                              ],
                              "type" : "object"
                           }
                        },
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `network` key contains the updated network"
            }
         },
         "summary" : "Update a Network",
         "tags" : [
            "Networks"
         ]
      }
   },
   "/networks/{id}/actions" : {
      "get" : {
         "description" : "Returns all Action objects for a Network. You can sort the results by using the `sort` URI parameter, and filter them with the `status` parameter.",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            },
            {
               "description" : "Can be used multiple times.",
               "in" : "query",
               "name" : "sort",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "id",
                     "id:asc",
                     "id:desc",
                     "command",
                     "command:asc",
                     "command:desc",
                     "status",
                     "status:asc",
                     "status:desc",
                     "progress",
                     "progress:asc",
                     "progress:desc",
                     "started",
                     "started:asc",
                     "started:desc",
                     "finished",
                     "finished:asc",
                     "finished:desc"
                  ],
                  "title" : "ParameterSort",
                  "type" : "string"
               }
            },
            {
               "description" : "Can be used multiple times, the response will contain only Actions with specified statuses",
               "in" : "query",
               "name" : "status",
               "required" : false,
               "schema" : {
                  "enum" : [
                     "running",
                     "success",
                     "error"
                  ],
                  "title" : "ParameterStatus",
                  "type" : "string"
               }
            }
         ],
         "responses" : {
            "200" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "actions" : [
                           {
                              "command" : "add_subnet",
                              "error" : {
                                 "code" : "action_failed",
                                 "message" : "Action failed"
                              },
                              "finished" : "2016-01-30T23:56:00+00:00",
                              "id" : 13,
                              "progress" : 100,
                              "resources" : [
                                 {
                                    "id" : 42,
                                    "type" : "server"
                                 }
                              ],
                              "started" : "2016-01-30T23:55:00+00:00",
                              "status" : "success"
                           }
                        ]
                     },
                     "schema" : {
                        "properties" : {
                           "actions" : {
                              "items" : {
                                 "properties" : {
                                    "command" : {
                                       "description" : "Command executed in the Action",
                                       "example" : "start_server",
                                       "type" : "string"
                                    },
                                    "error" : {
                                       "description" : "Error message for the Action if error occurred, otherwise null",
                                       "nullable" : true,
                                       "properties" : {
                                          "code" : {
                                             "description" : "Fixed machine readable code",
                                             "example" : "action_failed",
                                             "type" : "string"
                                          },
                                          "message" : {
                                             "description" : "Humanized error message",
                                             "example" : "Action failed",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "code",
                                          "message"
                                       ],
                                       "type" : "object"
                                    },
                                    "finished" : {
                                       "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "nullable" : true,
                                       "type" : "string"
                                    },
                                    "id" : {
                                       "description" : "ID of the Resource",
                                       "example" : 42,
                                       "format" : "int64",
                                       "type" : "integer"
                                    },
                                    "progress" : {
                                       "description" : "Progress of Action in percent",
                                       "example" : 100,
                                       "type" : "number"
                                    },
                                    "resources" : {
                                       "description" : "Resources the Action relates to",
                                       "items" : {
                                          "properties" : {
                                             "id" : {
                                                "description" : "ID of the Resource",
                                                "example" : 42,
                                                "format" : "int64",
                                                "type" : "integer"
                                             },
                                             "type" : {
                                                "description" : "Type of resource referenced",
                                                "example" : "server",
                                                "type" : "string"
                                             }
                                          },
                                          "required" : [
                                             "id",
                                             "type"
                                          ],
                                          "type" : "object"
                                       },
                                       "type" : "array"
                                    },
                                    "started" : {
                                       "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                       "example" : "2016-01-30T23:55:00+00:00",
                                       "type" : "string"
                                    },
                                    "status" : {
                                       "description" : "Status of the Action",
                                       "enum" : [
                                          "success",
                                          "running",
                                          "error"
                                       ],
                                       "type" : "string"
                                    }
                                 },
                                 "required" : [
                                    "id",
                                    "command",
                                    "status",
                                    "progress",
                                    "started",
                                    "finished",
                                    "resources",
                                    "error"
                                 ],
                                 "title" : "Action",
                                 "type" : "object"
                              },
                              "type" : "array"
                           },
                           "meta" : {
                              "properties" : {
                                 "pagination" : {
                                    "properties" : {
                                       "last_page" : {
                                          "description" : "ID of the last page available. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "next_page" : {
                                          "description" : "ID of the next page. Can be null if the current page is the last one.",
                                          "example" : 4,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "page" : {
                                          "description" : "Current page number",
                                          "example" : 3,
                                          "type" : "number"
                                       },
                                       "per_page" : {
                                          "description" : "Maximum number of items shown per page in the response",
                                          "example" : 25,
                                          "type" : "number"
                                       },
                                       "previous_page" : {
                                          "description" : "ID of the previous page. Can be null if the current page is the first one.",
                                          "example" : 2,
                                          "nullable" : true,
                                          "type" : "number"
                                       },
                                       "total_entries" : {
                                          "description" : "The total number of entries that exist in the database for this query. Nullable if unknown.",
                                          "example" : 100,
                                          "nullable" : true,
                                          "type" : "number"
                                       }
                                    },
                                    "required" : [
                                       "page",
                                       "per_page",
                                       "previous_page",
                                       "next_page",
                                       "last_page",
                                       "total_entries"
                                    ],
                                    "type" : "object"
                                 }
                              },
                              "required" : [
                                 "pagination"
                              ],
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "actions"
                        ],
                        "title" : "ActionsResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `actions` key contains a list of Actions"
            }
         },
         "summary" : "Get all Actions for a Network",
         "tags" : [
            "Network Actions"
         ]
      }
   },
   "/networks/{id}/actions/add_route" : {
      "post" : {
         "description" : "Adds a route entry to a Network.\n\nNote: if the Network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "destination" : {
                           "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                           "example" : "10.100.1.0/24",
                           "type" : "string"
                        },
                        "gateway" : {
                           "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range, an IP behind a vSwitch or 172.31.1.1, as this IP is being used as a gateway for the public network interface of Servers.",
                           "example" : "10.0.1.1",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "destination",
                        "gateway"
                     ],
                     "title" : "AddDeleteRouteRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "add_route",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `add_route` Action"
            }
         },
         "summary" : "Add a route to a Network",
         "tags" : [
            "Network Actions"
         ]
      }
   },
   "/networks/{id}/actions/add_subnet" : {
      "post" : {
         "description" : "Adds a new subnet object to the Network. If you do not specify an `ip_range` for the subnet we will automatically pick the first available /24 range for you if possible.\n\nNote: if the parent Network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip_range" : {
                           "description" : "Range to allocate IPs from. Must be a Subnet of the ip_range of the parent network object and must not overlap with any other subnets or with any destinations in routes. If the Subnet is of type vSwitch, it also can not overlap with any gateway in routes. Minimum Network size is /30. We suggest that you pick a bigger Network with a /24 netmask.",
                           "example" : "10.0.1.0/24",
                           "type" : "string"
                        },
                        "network_zone" : {
                           "description" : "Name of Network zone. The Location object contains the `network_zone` property each Location belongs to.",
                           "example" : "eu-central",
                           "type" : "string"
                        },
                        "type" : {
                           "description" : "Type of Subnetwork",
                           "enum" : [
                              "cloud",
                              "server",
                              "vswitch"
                           ],
                           "type" : "string"
                        },
                        "vswitch_id" : {
                           "description" : "ID of the robot vSwitch. Must be supplied if the subnet is of type vswitch.",
                           "example" : 1000,
                           "format" : "int64",
                           "type" : "integer"
                        }
                     },
                     "required" : [
                        "type",
                        "network_zone"
                     ],
                     "title" : "AddSubnetRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "add_subnet",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `add_subnet` Action"
            }
         },
         "summary" : "Add a subnet to a Network",
         "tags" : [
            "Network Actions"
         ]
      }
   },
   "/networks/{id}/actions/change_ip_range" : {
      "post" : {
         "description" : "Changes the IP range of a Network. IP ranges can only be extended and never shrunk. You can only add IPs at the end of an existing IP range. This means that the IP part of your existing range must stay the same and you can only change its netmask.\n\nFor example if you have a range `10.0.0.0/16` you want to extend then your new range must also start with the IP `10.0.0.0`. Your CIDR netmask `/16` may change to a number that is smaller than `16` thereby increasing the IP range. So valid entries would be `10.0.0.0/15`, `10.0.0.0/14`, `10.0.0.0/13` and so on.\n\nAfter changing the IP range you will have to adjust the routes on your connected Servers by either rebooting them or manually changing the routes to your private Network interface.\n\nNote: if the Network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip_range" : {
                           "description" : "The new prefix for the whole Network",
                           "example" : "10.0.0.0/12",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "ip_range"
                     ],
                     "title" : "ChangeIPRangeRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_ip_range",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_ip_range` Action"
            }
         },
         "summary" : "Change IP range of a Network",
         "tags" : [
            "Network Actions"
         ]
      }
   },
   "/networks/{id}/actions/change_protection" : {
      "post" : {
         "description" : "Changes the protection configuration of a Network.\n\nNote: if the Network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "delete" : {
                           "description" : "If true, prevents the Network from being deleted",
                           "example" : true,
                           "type" : "boolean"
                        }
                     },
                     "title" : "ChangeProtectionRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "change_protection",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : "2016-01-30T23:56:00+00:00",
                           "id" : 13,
                           "progress" : 100,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:55:00+00:00",
                           "status" : "success"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `change_protection` Action"
            }
         },
         "summary" : "Change Network Protection",
         "tags" : [
            "Network Actions"
         ]
      }
   },
   "/networks/{id}/actions/delete_route" : {
      "post" : {
         "description" : "Delete a route entry from a Network.\n\nNote: if the Network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "destination" : {
                           "description" : "Destination network or host of this route. Must not overlap with an existing ip_range in any subnets or with any destinations in other routes or with the first IP of the networks ip_range or with 172.31.1.1. Must be one of the private IPv4 ranges of RFC1918.",
                           "example" : "10.100.1.0/24",
                           "type" : "string"
                        },
                        "gateway" : {
                           "description" : "Gateway for the route. Cannot be the first IP of the networks ip_range, an IP behind a vSwitch or 172.31.1.1, as this IP is being used as a gateway for the public network interface of Servers.",
                           "example" : "10.0.1.1",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "destination",
                        "gateway"
                     ],
                     "title" : "AddDeleteRouteRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "delete_route",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "nullable" : true,
                                    "type" : "string"
                                 },
                                 "id" : {
                                    "description" : "ID of the Resource",
                                    "example" : 42,
                                    "format" : "int64",
                                    "type" : "integer"
                                 },
                                 "progress" : {
                                    "description" : "Progress of Action in percent",
                                    "example" : 100,
                                    "type" : "number"
                                 },
                                 "resources" : {
                                    "description" : "Resources the Action relates to",
                                    "items" : {
                                       "properties" : {
                                          "id" : {
                                             "description" : "ID of the Resource",
                                             "example" : 42,
                                             "format" : "int64",
                                             "type" : "integer"
                                          },
                                          "type" : {
                                             "description" : "Type of resource referenced",
                                             "example" : "server",
                                             "type" : "string"
                                          }
                                       },
                                       "required" : [
                                          "id",
                                          "type"
                                       ],
                                       "type" : "object"
                                    },
                                    "type" : "array"
                                 },
                                 "started" : {
                                    "description" : "Point in time when the Action was started (in ISO-8601 format)",
                                    "example" : "2016-01-30T23:55:00+00:00",
                                    "type" : "string"
                                 },
                                 "status" : {
                                    "description" : "Status of the Action",
                                    "enum" : [
                                       "success",
                                       "running",
                                       "error"
                                    ],
                                    "type" : "string"
                                 }
                              },
                              "required" : [
                                 "id",
                                 "command",
                                 "status",
                                 "progress",
                                 "started",
                                 "finished",
                                 "resources",
                                 "error"
                              ],
                              "title" : "Action",
                              "type" : "object"
                           }
                        },
                        "required" : [
                           "action"
                        ],
                        "title" : "ActionResponse",
                        "type" : "object"
                     }
                  }
               },
               "description" : "The `action` key contains the `delete_route` Action"
            }
         },
         "summary" : "Delete a route from a Network",
         "tags" : [
            "Network Actions"
         ]
      }
   },
   "/networks/{id}/actions/delete_subnet" : {
      "post" : {
         "description" : "Deletes a single subnet entry from a Network. You cannot delete subnets which still have Servers attached. If you have Servers attached you first need to detach all Servers that use IPs from this subnet before you can delete the subnet.\n\nNote: if the Network object changes during the request, the response will be a “conflict” error.\n",
         "parameters" : [
            {
               "description" : "ID of the Network",
               "in" : "path",
               "name" : "id",
               "required" : true,
               "schema" : {
                  "format" : "int64",
                  "type" : "integer"
               }
            }
         ],
         "requestBody" : {
            "content" : {
               "application/json" : {
                  "schema" : {
                     "properties" : {
                        "ip_range" : {
                           "description" : "IP range of subnet to delete",
                           "example" : "10.0.1.0/24",
                           "type" : "string"
                        }
                     },
                     "required" : [
                        "ip_range"
                     ],
                     "title" : "DeleteSubnetRequest",
                     "type" : "object"
                  }
               }
            }
         },
         "responses" : {
            "201" : {
               "content" : {
                  "application/json" : {
                     "example" : {
                        "action" : {
                           "command" : "delete_subnet",
                           "error" : {
                              "code" : "action_failed",
                              "message" : "Action failed"
                           },
                           "finished" : null,
                           "id" : 13,
                           "progress" : 0,
                           "resources" : [
                              {
                                 "id" : 4711,
                                 "type" : "network"
                              }
                           ],
                           "started" : "2016-01-30T23:50:00+00:00",
                           "status" : "running"
                        }
                     },
                     "schema" : {
                        "properties" : {
                           "action" : {
                              "properties" : {
                                 "command" : {
                                    "description" : "Command executed in the Action",
                                    "example" : "start_server",
                                    "type" : "string"
                                 },
                                 "error" : {
                                    "description" : "Error message for the Action if error occurred, otherwise null",
                                    "nullable" : true,
                                    "properties" : {
                                       "code" : {
                                          "description" : "Fixed machine readable code",
                                          "example" : "action_failed",
                                          "type" : "string"
                                       },
                                       "message" : {
                                          "description" : "Humanized error message",
                                          "example" : "Action failed",
                                          "type" : "string"
                                       }
                                    },
                                    "required" : [
                                       "code",
                                       "message"
                                    ],
                                    "type" : "object"
                                 },
                                 "finished" : {
                                    "description" : "Point in time when the Action was finished (in ISO-8601 format). Only set if the Action is finished otherwise null.",
                                    "example" : "2016-01-30T23:55:00+00:00",
                