package Bencher::Scenario::HashSlicing::slice;

our $DATE = '2020-04-19'; # DATE
our $VERSION = '0.004'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark hash slicing',
    participants => [
        {
            module => 'Hash::Util::Pick',
            function => 'pick',
            code_template => 'state $hash = <hash>; state $keys = <keys>; Hash::Util::Pick::pick($hash, @$keys)',
        },
        {
            name => 'map',
            code_template => 'state $hash = <hash>; state $keys = <keys>; +{ map { (exists $hash->{$_} ? ($_ => $hash->{$_}) : ()) } @$keys}',
        },
        {
            name => 'map+grep',
            code_template => 'state $hash = <hash>; state $keys = <keys>; +{ map {$_ => $hash->{$_}} grep { exists $hash->{$_} } @$keys}',
        },
        {
            module => 'Hash::Subset',
            function => 'hashref_subset',
            code_template => 'state $hash = <hash>; state $keys = <keys>; Hash::Subset::hashref_subset($hash, $keys)',
        },
        {
            module => 'Hash::MoreUtils',
            function => 'slice_exists',
            code_template => 'state $hash = <hash>; state $keys = <keys>; my %h = Hash::MoreUtils::slice_exists($hash, @$keys); \%h',
        },
    ],

    datasets => [
        {
            name => 'keys=2, slice=2, exists=1',
            args => { hash=>{1=>1, 2=>1}, keys=>[1, 3] },
        },

        {
            name => 'keys=10, slice=2, exists=1',
            args => { hash=>{map {$_=>1} 1..10}, keys=>[1, 11] },
        },
        {
            name => 'keys=10, slice=10, exists=5',
            args => { hash=>{map {$_=>1} 1..10}, keys=>[1..5, 11..15] },
        },

        {
            name => 'keys=100, slice=2, exists=1',
            args => { hash=>{map {$_=>1} 1..100}, keys=>[1, 101] },
        },
        {
            name => 'keys=100, slice=10, exists=5',
            args => { hash=>{map {$_=>1} 1..100}, keys=>[1..5, 101..105] },
        },
        {
            name => 'keys=100, slice=100, exists=50',
            args => { hash=>{map {$_=>1} 1..100}, keys=>[1..50, 101..150] },
        },

        {
            name => 'keys=1000, slice=2, exists=1',
            args => { hash=>{map {$_=>1} 1..1000}, keys=>[1, 1001] },
        },
        {
            name => 'keys=1000, slice=10, exists=5',
            args => { hash=>{map {$_=>1} 1..1000}, keys=>[1..5, 1001..1005] },
        },
        {
            name => 'keys=1000, slice=100, exists=50',
            args => { hash=>{map {$_=>1} 1..1000}, keys=>[1..50, 1001..1050] },
        },
        {
            name => 'keys=1000, slice=1000, exists=500',
            args => { hash=>{map {$_=>1} 1..1000}, keys=>[1..500, 1001..1500] },
        },

        {
            name => 'keys=10000, slice=2, exists=1',
            args => { hash=>{map {$_=>1} 1..10000}, keys=>[1,10001] },
        },
        {
            name => 'keys=10000, slice=10, exists=5',
            args => { hash=>{map {$_=>1} 1..10000}, keys=>[1..5,10001..10005] },
        },
        {
            name => 'keys=10000, slice=100, exists=50',
            args => { hash=>{map {$_=>1} 1..10000}, keys=>[1..50,10001..10050] },
        },

        {
            name => 'keys=100000, slice=2, exists=1',
            args => { hash=>{map {$_=>1} 1..100000}, keys=>[1,100001] },
        },
    ],
};

1;
# ABSTRACT: Benchmark hash slicing

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::HashSlicing::slice - Benchmark hash slicing

=head1 VERSION

This document describes version 0.004 of Bencher::Scenario::HashSlicing::slice (from Perl distribution Bencher-Scenarios-HashSlicing), released on 2020-04-19.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m HashSlicing::slice

To run module startup overhead benchmark:

 % bencher --module-startup -m HashSlicing::slice

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Hash::Util::Pick> 0.13

L<Hash::Subset> 0.005

L<Hash::MoreUtils> 0.06

=head1 BENCHMARK PARTICIPANTS

=over

=item * Hash::Util::Pick::pick (perl_code)

Code template:

 state $hash = <hash>; state $keys = <keys>; Hash::Util::Pick::pick($hash, @$keys)



=item * map (perl_code)

Code template:

 state $hash = <hash>; state $keys = <keys>; +{ map { (exists $hash->{$_} ? ($_ => $hash->{$_}) : ()) } @$keys}



=item * map+grep (perl_code)

Code template:

 state $hash = <hash>; state $keys = <keys>; +{ map {$_ => $hash->{$_}} grep { exists $hash->{$_} } @$keys}



=item * Hash::Subset::hashref_subset (perl_code)

Code template:

 state $hash = <hash>; state $keys = <keys>; Hash::Subset::hashref_subset($hash, $keys)



=item * Hash::MoreUtils::slice_exists (perl_code)

Code template:

 state $hash = <hash>; state $keys = <keys>; my %h = Hash::MoreUtils::slice_exists($hash, @$keys); \%h



=back

=head1 BENCHMARK DATASETS

=over

=item * keys=2, slice=2, exists=1

=item * keys=10, slice=2, exists=1

=item * keys=10, slice=10, exists=5

=item * keys=100, slice=2, exists=1

=item * keys=100, slice=10, exists=5

=item * keys=100, slice=100, exists=50

=item * keys=1000, slice=2, exists=1

=item * keys=1000, slice=10, exists=5

=item * keys=1000, slice=100, exists=50

=item * keys=1000, slice=1000, exists=500

=item * keys=10000, slice=2, exists=1

=item * keys=10000, slice=10, exists=5

=item * keys=10000, slice=100, exists=50

=item * keys=100000, slice=2, exists=1

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 4.15.0-91-generic >>.

Benchmark with default options (C<< bencher -m HashSlicing::slice >>):

 #table1#
 {dataset=>"keys=10, slice=10, exists=5"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |    363000 |    2.75   |                 0.00% |               415.70% | 6.8e-10 |      30 |
 | Hash::Subset::hashref_subset  |    554740 |    1.8027 |                52.80% |               237.50% | 4.9e-12 |      20 |
 | map+grep                      |    566000 |    1.77   |                55.89% |               230.80% | 8.3e-10 |      20 |
 | map                           |    596190 |    1.6773 |                64.22% |               214.03% | 4.6e-12 |      20 |
 | Hash::Util::Pick::pick        |   1900000 |    0.53   |               415.70% |                 0.00% | 8.3e-10 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table2#
 {dataset=>"keys=10, slice=2, exists=1"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::Subset::hashref_subset  |   1100000 |     920   |                 0.00% |               299.50% | 1.3e-09 |      35 |
 | Hash::MoreUtils::slice_exists |   1160000 |     862   |                 7.12% |               272.93% | 4.2e-10 |      20 |
 | map+grep                      |   1900000 |     520   |                78.41% |               123.92% | 8.3e-10 |      20 |
 | map                           |   2162000 |     462.5 |                99.59% |               100.16% | 5.2e-12 |      20 |
 | Hash::Util::Pick::pick        |   4300000 |     230   |               299.50% |                 0.00% | 4.2e-10 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table3#
 {dataset=>"keys=100, slice=10, exists=5"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |    364000 |    2.75   |                 0.00% |               410.84% | 7.5e-10 |      25 |
 | Hash::Subset::hashref_subset  |    550080 |    1.8179 |                51.28% |               237.69% | 4.5e-12 |      20 |
 | map+grep                      |    560000 |    1.8    |                54.13% |               231.43% | 2.5e-09 |      20 |
 | map                           |    588000 |    1.7    |                61.68% |               215.95% | 7.3e-10 |      26 |
 | Hash::Util::Pick::pick        |   1900000 |    0.54   |               410.84% |                 0.00% | 8.3e-10 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table4#
 {dataset=>"keys=100, slice=100, exists=50"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |   41500   |   24.1    |                 0.00% |               330.30% |   2e-08 |      20 |
 | map+grep                      |   61500   |   16.3    |                48.03% |               190.70% | 5.8e-09 |      26 |
 | map                           |   64894.4 |   15.4097 |                56.22% |               175.44% | 4.9e-12 |      20 |
 | Hash::Subset::hashref_subset  |   81800   |   12.2    |                96.82% |               118.62% | 2.8e-09 |      28 |
 | Hash::Util::Pick::pick        |  180000   |    5.6    |               330.30% |                 0.00% | 6.7e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table5#
 {dataset=>"keys=100, slice=2, exists=1"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::Subset::hashref_subset  |   1069900 |    934.68 |                 0.00% |               301.67% |   5e-12 |      20 |
 | Hash::MoreUtils::slice_exists |   1160000 |    861    |                 8.51% |               270.18% | 4.2e-10 |      20 |
 | map+grep                      |   1930000 |    518    |                80.57% |               122.44% | 1.8e-10 |      28 |
 | map                           |   2182000 |    458.2  |               103.98% |                96.91% | 2.3e-11 |      20 |
 | Hash::Util::Pick::pick        |   4300000 |    233    |               301.67% |                 0.00% | 9.3e-11 |      25 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table6#
 {dataset=>"keys=1000, slice=10, exists=5"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |    362000 |    2.76   |                 0.00% |               419.95% | 6.9e-10 |      29 |
 | Hash::Subset::hashref_subset  |    559710 |    1.7867 |                54.65% |               236.21% |   5e-12 |      20 |
 | map+grep                      |    560000 |    1.8    |                55.24% |               234.93% | 2.1e-09 |      29 |
 | map                           |    590610 |    1.6932 |                63.19% |               218.62% | 5.7e-12 |      20 |
 | Hash::Util::Pick::pick        |   1900000 |    0.53   |               419.95% |                 0.00% | 6.2e-10 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table7#
 {dataset=>"keys=1000, slice=100, exists=50"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |     35000 |    28     |                 0.00% |               383.15% | 5.3e-08 |      20 |
 | map+grep                      |     60000 |    16.7   |                69.26% |               185.45% | 5.5e-09 |      29 |
 | map                           |     61600 |    16.2   |                73.66% |               178.22% | 6.7e-09 |      20 |
 | Hash::Subset::hashref_subset  |     80068 |    12.489 |               125.74% |               114.03% | 8.8e-11 |      29 |
 | Hash::Util::Pick::pick        |    170000 |     5.8   |               383.15% |                 0.00% | 8.3e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table8#
 {dataset=>"keys=1000, slice=1000, exists=500"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |      3820 |       262 |                 0.00% |               240.25% | 2.1e-07 |      21 |
 | map+grep                      |      5570 |       179 |                45.85% |               133.29% | 5.3e-08 |      20 |
 | map                           |      5730 |       175 |                49.92% |               126.96% | 5.1e-08 |      22 |
 | Hash::Subset::hashref_subset  |      7770 |       129 |               103.23% |                67.42% | 5.3e-08 |      20 |
 | Hash::Util::Pick::pick        |     13000 |        77 |               240.25% |                 0.00% |   1e-07 |      22 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table9#
 {dataset=>"keys=1000, slice=2, exists=1"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::Subset::hashref_subset  |   1080000 |    930    |                 0.00% |               302.11% | 4.2e-10 |      20 |
 | Hash::MoreUtils::slice_exists |   1200000 |    860    |                 7.69% |               273.41% | 1.2e-09 |      20 |
 | map+grep                      |   1948000 |    513.34 |                81.13% |               122.00% | 4.5e-12 |      20 |
 | map                           |   2162000 |    462.5  |               101.06% |                99.99% | 5.7e-12 |      20 |
 | Hash::Util::Pick::pick        |   4325000 |    231.2  |               302.11% |                 0.00% | 5.7e-12 |      23 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table10#
 {dataset=>"keys=10000, slice=10, exists=5"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |    356000 |    2.81   |                 0.00% |               417.42% | 7.2e-10 |      27 |
 | map+grep                      |    550190 |    1.8176 |                54.53% |               234.84% | 4.8e-12 |      20 |
 | Hash::Subset::hashref_subset  |    550190 |    1.8176 |                54.53% |               234.83% | 4.8e-12 |      20 |
 | map                           |    573520 |    1.7436 |                61.08% |               221.21% |   5e-12 |      20 |
 | Hash::Util::Pick::pick        |   1800000 |    0.54   |               417.42% |                 0.00% | 8.3e-10 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table11#
 {dataset=>"keys=10000, slice=100, exists=50"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::MoreUtils::slice_exists |     40200 |      24.9 |                 0.00% |               313.52% | 6.7e-09 |      20 |
 | map+grep                      |     57000 |      18   |                41.74% |               191.74% | 3.3e-08 |      20 |
 | map                           |     61100 |      16.4 |                51.87% |               172.28% | 6.4e-09 |      22 |
 | Hash::Subset::hashref_subset  |     76000 |      13   |                89.49% |               118.23% | 2.5e-08 |      22 |
 | Hash::Util::Pick::pick        |    170000 |       6   |               313.52% |                 0.00% | 6.7e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table12#
 {dataset=>"keys=10000, slice=2, exists=1"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::Subset::hashref_subset  |   1068100 |    936.27 |                 0.00% |               303.37% | 4.9e-12 |      20 |
 | Hash::MoreUtils::slice_exists |   1140000 |    875    |                 6.95% |               277.16% | 3.1e-10 |      37 |
 | map+grep                      |   1917000 |    521.6  |                79.50% |               124.71% | 5.2e-12 |      26 |
 | map                           |   2140000 |    467.28 |               100.37% |               101.32% | 4.5e-12 |      20 |
 | Hash::Util::Pick::pick        |   4308000 |    232.1  |               303.37% |                 0.00% | 5.2e-12 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table13#
 {dataset=>"keys=100000, slice=2, exists=1"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::Subset::hashref_subset  |   1076500 |    928.92 |                 0.00% |               304.62% | 5.7e-12 |      22 |
 | Hash::MoreUtils::slice_exists |   1156300 |    864.86 |                 7.41% |               276.71% | 5.2e-12 |      20 |
 | map+grep                      |   1936500 |    516.41 |                79.88% |               124.94% | 4.5e-12 |      20 |
 | map                           |   2200000 |    460    |               101.74% |               100.56% | 8.3e-10 |      20 |
 | Hash::Util::Pick::pick        |   4356000 |    229.6  |               304.62% |                 0.00% | 4.3e-12 |      22 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table14#
 {dataset=>"keys=2, slice=2, exists=1"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Hash::Subset::hashref_subset  |   1100000 |       920 |                 0.00% |               297.46% | 1.7e-09 |      20 |
 | Hash::MoreUtils::slice_exists |   1170000 |       858 |                 6.93% |               271.70% | 3.3e-10 |      32 |
 | map+grep                      |   1890000 |       530 |                73.22% |               129.45% | 2.1e-10 |      20 |
 | map                           |   2100000 |       480 |                92.34% |               106.64% | 5.8e-10 |      23 |
 | Hash::Util::Pick::pick        |   4300000 |       230 |               297.46% |                 0.00% | 5.2e-10 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m HashSlicing::slice --module-startup >>):

 #table15#
 +---------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant         | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +---------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | Hash::MoreUtils     |        10 |                 0 |                 0.00% |                35.95% |   0.00041 |      21 |
 | Hash::Subset        |        10 |                 0 |                10.65% |                22.86% |   0.00027 |      20 |
 | Hash::Util::Pick    |        12 |                 2 |                26.66% |                 7.33% | 5.9e-05   |      20 |
 | perl -e1 (baseline) |        10 |                 0 |                35.95% |                 0.00% |   0.00026 |      20 |
 +---------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-HashSlicing>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-HashSlicing>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-HashSlicing>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
