package Tcl::Tk;

use strict;
use Tcl;
use Exporter;
use vars qw(@ISA @EXPORT_OK %EXPORT_TAGS);
@ISA = qw(Exporter Tcl);

$Tcl::Tk::VERSION = '0.87';

# For users that want to ensure full debugging from initial use call,
# including the checks for other Tk modules loading following Tcl::Tk
# loading, add the following code *after* 'use Tcl::Tk':
#
# BEGIN { $Tcl::Tk::DEBUG = 1; }
#
$Tcl::Tk::DEBUG ||= 0;
sub DEBUG() {0}
sub WIDGET_CLEANUP() {0}
sub Tcl::Tk::Widget::DEBUG() {0}
sub _DEBUG {
    # Allow for optional debug level and message to be passed in.
    # If level is passed in, return true only if debugging is at
    # that level.
    # If message is passed in, output that message if the level
    # is appropriate (with any extra args passed to output).
    my $lvl = shift;
    return $Tcl::Tk::DEBUG unless defined $lvl;
    my $msg = shift;
    if (defined($msg) && ($Tcl::Tk::DEBUG >= $lvl)) { print STDERR $msg, @_; }
    return ($Tcl::Tk::DEBUG >= $lvl);
}

if (DEBUG()) {
    # The gestapo throws warnings whenever Perl/Tk modules are requested.
    # It also hijacks such requests and returns an empty module in its
    # place.
    unshift @INC, \&tk_gestapo;
}

=head1 NAME

Tcl::Tk - Extension module for Perl giving access to Tk via the Tcl extension

=head1 SYNOPSIS

    use Tcl::Tk;
    my $int = new Tcl::Tk;
    my $mw = $int->mainwindow;
    my $lab = $mw->Label(-text => "Hello world")->pack;
    my $btn = $mw->Button(-text => "test", -command => sub {
      $lab->configure(-text=>"[". $lab->cget('-text')."]");
    })->pack;
    $int->MainLoop;

Or    

    use Tcl::Tk;
    my $int = new Tcl::Tk;
    $int->Eval(<<'EOS');
    # pure-tcl code to create widgets (e.g. generated by some GUI builder)
    entry .e
    button .inc -text {increment by Perl}
    pack .e .inc
    EOS
    my $btn = $int->widget('.inc'); # get .inc button into play
    my $e = $int->widget('.e');     # get .e entry into play
    $e->configure(-textvariable=>\(my $var='aaa'));
    $btn->configure(-command=>sub{$var++});
    $int->MainLoop;

=head1 DESCRIPTION

The C<Tcl::Tk> module provides access to the Tk library within Tcl/Tk
installation. By using this module an interpreter object created, which
then gain access to entire variety of installed Tcl libraries (Tk, Tix,
BWidgets, BLT, etc) and existing features (for example natively looking
widgets using C<tile>).

=head2 Access to the Tcl and Tcl::Tk extensions

To get access to the Tcl and Tcl::Tk extensions, put the command near
the top of your program.

    use Tcl::Tk;

Export tag :perltk exports few convenience functions similar to perl/Tk,
so use syntax C<use Tcl::Tk qw(:perlTk);> in case you're in habit of writing 
directly C<MainLoop> instead of C<< $interp->MainLoop >> or C<Tcl::Tk::MainLoop>
    

=head2 Creating a Tcl interpreter for Tk

Before you start using widgets, an interpreter (at least one) should be
created, which will manage all things in Tcl.

To create a Tcl interpreter initialised for Tk, use

    my $int = new Tcl::Tk (DISPLAY, NAME, SYNC);

All arguments are optional. This creates a Tcl interpreter object $int,
and creates a main toplevel window. The window is created on display
DISPLAY (defaulting to the display named in the DISPLAY environment
variable) with name NAME (defaulting to the name of the Perl program,
i.e. the contents of Perl variable $0). If the SYNC argument is present
and true then an I<XSynchronize()> call is done ensuring that X events
are processed synchronously (and thus slowly). This is there for
completeness and is only very occasionally useful for debugging errant
X clients (usually at a much lower level than Tk users will want).

=head2 Entering the main event loop

The Perl method call

    $int->MainLoop;

on the Tcl::Tk interpreter object enters the Tk event loop. You can
instead do C<Tcl::Tk::MainLoop> or C<Tcl::Tk-E<gt>MainLoop> if you prefer.
You can even do simply C<MainLoop> if you import it from Tcl::Tk in
the C<use> statement.

=head2 Creating and using widgets

Two different approaches are used to manipulate widgets (or, more commonly,
to manipulate any Tcl objects behaving similarly)

=over

=item * access with a special widget accessing syntax of kind C<< $widget->method; >>

=item * random access with C<< Eval >>

=back

First way to manipulate widgets is identical to perl/Tk calling conventions,
second one deploys Tcl syntax. Both ways are very interchangeable in that
sence, a widget created with one way could be used by another way.

Usually Perl programs operate with Tcl/Tk via perl/Tk syntax, so user have no
need to deal with Tcl language directly, only some basic understanding of
widget is needed.

A possibility to use both approaches interchangeably gives an opportunity to
use Tcl code created elsewhere (some WYSIWIG IDE or such).

In order to get better understanding on usage of Tcl/Tk widgets from within
Perl, a bit of Tcl/Tk knowledge is needed, so we'll start from 2nd approach,
with Tcl's Eval (C<< $int->Eval('...') >>) and then smoothly move to 1st,
approach with perl/Tk syntax.

=head4 Tcl/Tk syntax

=over

=item * interpreter

Tcl interpreter is used to process Tcl/Tk widgets; within C<Tcl::Tk> you
create it with C<new>, and, given any widget object, you can retreive it by
C<< $widget->interp >> method. Within pure Tcl/Tk it is already exist.

=item * widget path

Widget path is a string starting with a dot and consisting of several
names separated by dots. These names are widget names that comprise
widget's hierarchy. As an example, if there exists a frame with a path
C<.fram> and you want to create a button on it and name it C<butt> then
you should specify name C<.fram.butt>. Widget paths are refered in
miscellaneous widget operations, and geometry management is one of them.

At any time widget's path could be retreived with C<< $widget->path; >>
within C<Tcl::Tk>.

=item * widget as Tcl/Tk command

when widget is created, a special command is created within Tk, the name of
this command is widget's path. That said, C<.fr.b> is Tk's command and this
command has subcommands, those will help manipulating widget. That is why
C<< $int->Eval('.fr.b configure -text {new text}'); >> makes sence.
Note that C<< $button->configure(-text=>'new text'); >> does exactly that,
provided a fact C<$button> corresponds to C<.fr.b> widget.

=back

C<use Tcl::Tk;> not only creates C<Tcl::Tk> package, but also it creates
C<Tcl::Tk::Widget> package, responsible for widgets. Each widget (object
blessed to C<Tcl::Tk::Widget>, or other widgets in ISA-relationship)
behaves in such a way that its method will result in calling it's path on
interpreter.

=head4 Perl/Tk syntax

C<Tcl::Tk::Widget> package within C<Tcl::Tk> module fully aware of perl/Tk
widget syntax, which has long usage. This means that any C<Tcl::Tk> widget
has a number of methods like C<Button>, C<Frame>, C<Text>, C<Canvas> and so
on, and invoking those methods will create appropriate child widget.
C<Tcl::Tk> module will generate an unique name of newly created widget.

To demonstrate this concept:

    my $label = $frame->Label(-text => "Hello world");

executes the command

    $int->call("label", ".l", "-text", "Hello world");

and this command similar to

    $int->Eval("label .l -text {Hello world}");

This way Tcl::Tk widget commands are translated to Tcl syntax and directed to
Tcl interpreter; understanding this helps in idea, why two approaches with
dealing with widgets are interchangeable.

Newly created widget C<$label> will be blessed to package C<Tcl::Tk::Widget::Label>
which is isa-C<Tcl::Tk::Widget>

=head3 OO explanations of Widget-s of Tcl::Tk

C<Tcl::Tk> widgets use object-oriented approach, which means a quite concrete
object hierarchy presents. Interesting point about this object system - 
it is very dynamic. Initially no widgets objects and no widget classes present,
but they immediately appear at the time when they needed.

So they virtually exist, but come into actual existance dynamically. This
dynamic approach allows same usage of widget library without any mention from
within C<Tcl::Tk> module at all.

Let us look into following few lines of code:

  my $text = $mw->Text->pack;
  $text->insert('end', -text=>'text');
  $text->windowCreate('end', -window=>$text->Label(-text=>'text of label'));

Internally, following mechanics comes into play.
Text method creates Text widget (known as C<text> in Tcl/Tk environment). 
When this creation method invoked first time, a package 
C<Tcl::Tk::Widget::Text> is created, which will be OO presentation of all
further Text-s widgets. All such widgets will be blessed to that package
and will be in ISA-relationship with C<Tcl::Tk::Widget>.

Second line calls method C<insert> of C<$text> object of type
C<Tcl::Tk::Widget::Text>. When invoked first time, a method C<insert> is 
created in package C<Tcl::Tk::Widget::Text>, with destiny to call
C<invoke> method of our widget in Tcl/Tk world.

At first time when C<insert> is called, this method does not exist, so AUTOLOAD
comes to play and creates such a method. Second time C<insert> called already
existing subroutine will be invoked, thus saving execution time.

As long as widgets of different type 'live' in different packages, they do not
intermix, so C<insert> method of C<Tcl::Tk::Widget::Listbox> will mean
completely different behaviour.

=head3 explanations how Widget-s of Tcl::Tk methods correspond to Tcl/Tk

Suppose C<$widget> isa-C<Tcl::Tk::Widget>, its path is C<.path> and method
C<method> invoked on it with a list of parameters, C<@parameters>:

  $widget->method(@parameters);

In this case as a first step all C<@parameters> will be preprocessed, during
this preprocessing following actions are performed:

=over

=item 1.

for each variable reference its Tcl variable will be created and tied to it

=item 2.

for each code reference its Tcl command will be created and tied to it

=item 3.

each array reference considered as callback, and proper actions will be taken

=back

After adoptation of C<@parameters> Tcl/Tk interpreter will be requested to
perform following operation:

=over

=item if C<$method> is all lowercase, C<m/^[a-z]$/>

C<.path method parameter1 parameter2> I<....>

=item if C<$method> contains exactly one capital letter inside name, C<m/^[a-z]+[A-Z][a-z]+$/>

C<.path method submethod parameter1 parameter2> I<....>

=head4 faster way of invoking methods on widgets

In case it is guaranteed that preprocessing of C<@parameters> are not required
(in case no parameters are Perl references to scalar, subroutine or array), then
preprocessing step described above could be skipped.

To achieve that, prepend method name with underscore, C<_>. Mnemonically it means
you are using some internal method that executes faster, but normally you use
"public" method, which includes all preprocessing.

Example:

   # at following line faster method is incorrect, as \$var must be
   # preprocessed for Tcl/Tk:
   $button->configure(-textvariable=>\$var);

   # faster version of insert method of "Text" widget is perfectly possible
   $text->_insert('end','text to insert','tag');
   # following line does exactly same thing as previous line:
   $text->_insertEnd('text to insert','tag');

When doing many inserts to text widget, faster version could fasten execution.

=back

=head2 using any Tcl/Tk feature with Tcl::Tk module

Tcl::Tk module allows using any widget from Tcl/Tk widget library with either
Tcl syntax (via Eval), or with regular Perl syntax.

In order to provide perlTk syntax to any Tcl/Tk widget, only single call should
be made, namely 'Declare' widget's method.

Syntax is

 $widget->Declare('perlTk_widget_method_name','tcl/tk-widget_method_name',
    @options);
 
Options are:

  -require=>'tcl-package-name'
  -prefix=>'some-prefix'

'-require' option specifies that said widget requires a Tcl package with a name
of 'tcl-package-name';
'-prefix' option used to specify a part of autogenerated widget name, usually
used when Tcl widget name contain non-alphabet characters (e.g. ':') so
to keep autogenerated names syntaxically correct.

A typical example of such invocation is:

  $mw->Declare('BLTNoteBook','blt::tabnotebook',-require=>'BLT',-prefix=>'bltnbook');

After such a call Tcl::Tk module will take a knowledge about tabnotebook widget
from within BLT package and create proper widget creation method for it with a 
name BLTNoteBook. This means following statement:

 my $tab = $mw->BLTNoteBook;

will create blt::tabnotebook widget. Effectively, this is similar to following
Tcl/Tk code:

  package require BLT # but invoked only once
  blt::tabnotebook .bltnbook1

Also, Perl variable $tab will contain ordinary Tcl/Tk widget that behaves in
usual way, for example:

  $tab->insert('end', -text=>'text');
  $tab->tabConfigure(0, -window=>$tab->Label(-text=>'text of label'));

These two lines are Tcl/Tk equivalent of:

  .bltnbook1 insert end -text {text}
  .bltnbook1 tab configure 0 -window [label .bltnbook1.lab1 -text {text of label}]

Given all previously said, you can also write intermixing both approaches:

  $interp->Eval('package require BLT;blt::tabnotebook .bltnbook1');
  $tab = $interp->widget('.bltnbook1');
  $tab->tabConfigure(0, -window=>$tab->Label(-text=>'text of label'));

=head3 using documentation of Tcl/Tk widgets for applying within Tcl::Tk module

As a general rule, you need to consult TCL man pages to realize how to
use a widget, and after that invoke perl command that creates it properly.
When reading Tcl/Tk documentation about widgets, quite simple transformation is
needed to apply to Tcl::Tk module.

Suppose it says:

  pathName method-name optional-parameters
     (some description)
     
you should understand, that widget in question has method C<method-name> and you could
invoke it as

  $widget->method-name(optional-parameters);

$widget is that widget with pathName, created with perl/Tk syntax, or fetched by
C<< $int->widget >> method.

Sometimes in Tcl/Tk method-name consist of two words (verb1 verb2), in this
case there are two ways to invoke it, C<< $widget->verb1('verb2',...); >> or it
C<< $widget->verb1Verb2(...); >> - those are identical.

Widget options are same within Tcl::Tk and Tcl/Tk.

=head3 C<< $int->widget() >> method

When widgets are created they are stored internally and could be retreived
by C<widget()>, which takes widget path as first parameter, and optionally
widget type (such as Button, or Text etc.). Example:

    # this will retrieve widget, and then call configure on it
    widget(".fram.butt")->configure(-text=>"new text");

    # this will retrieve widget as Button (Tcl::Tk::Widget::Button object)
    my $button = widget(".fram.butt",'Button');
    
    # same but retrieved widget considered as general widget, without
    # concrete specifying its type (Tcl::Tk::Widget object)
    my $button = widget(".fram.butt");

Please note that this method will return to you a widget object even if it was
not created within this module, and check will not be performed whether a 
widget with given path exists, despite of fact that checking for existence of
a widget is an easy task (invoking C<< $interp->Eval("info commands $path"); >>
will do this). Instead, you will receive perl object that will try to operate
with widget that has given path even if such path do not exists. In case it do
not actually exist, you will receive an error from Tcl/Tk.

To check if a widget with a given path exists use C<Tcl::Tk::Exists($widget)>
subroutine. It queries Tcl/Tk for existance of said widget.

=head3 C<widget_data> method

If you need to associate any data with particular widget, you can do this with 
C<widget_data> method of either interpreter or widget object itself. This method
returns same anonymous hash and it should be used to hold any keys/values pairs.

Examples:

  $interp->widget_data('.fram1.label2')->{var} = 'value';
  $label->widget_data()->{var} = 'value';

=head2 Non-widget Tk commands

Many non-widget Tk commands are also available within Tcl::Tk module, such
as C<focus>, C<wm>, C<winfo> and so on. If some of them not present directly,
you can always use C<< $int->Eval('...') >> approach.

=head1 BUGS

Currently work is in progress, and some features could change in future
versions.

=head1 AUTHORS

Malcolm Beattie, mbeattie@sable.ox.ac.uk
Vadim Konovalov, vkonovalov@peterstar.ru, 19 May 2003.
Jeff Hobbs, jeffh _a_ activestate com, February 2004.
Gisle Aas, gisle _a_ activestate . com, 14 Apr 2004.

=head1 COPYRIGHT

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

See http://www.perl.com/perl/misc/Artistic.html

=cut

my @widgets = 
    qw(frame toplevel label labelframe button checkbutton radiobutton scale
       message listbox scrollbar spinbox entry menu menubutton 
       canvas text panedwindow
       widget awidget
     );
my @misc = qw(MainLoop after destroy focus grab lower option place raise
              image font
	      selection tk grid tkwait update winfo wm);
my @perlTk = qw(MainLoop MainWindow tkinit update);

@EXPORT_OK = (@widgets, @misc, @perlTk);
%EXPORT_TAGS = (widgets => \@widgets, misc => \@misc, perlTk => \@perlTk);

## TODO -- module's private $tkinterp should go away!
my $tkinterp = undef;		# this gets defined when "new" is done

# Hash to keep track of all created widgets and related instance data
# Tcl::Tk will maintain PATH (Tk widget pathname) and INT (Tcl interp)
# and the user can create other info.
my %W = (
    INT => {},
    PATH => {},
    RPATH => {},
    DATA => {},
    MWID => {},
);
# few shortcuts for %W to be faster
my $Wint = $W{INT};
my $Wpath = $W{PATH};
my $Wdata = $W{DATA};

# hash to keep track on preloaded Tcl/Tk modules, such as Tix, BWidget
my %preloaded_tk; # (interpreter independent thing. is this right?)

#
sub new {
    my ($class, $name, $display, $sync) = @_;
    Carp::croak 'Usage: $interp = new Tcl::Tk([$name [, $display [, $sync]]])'
	if @_ > 4;
    my($i, $arg, @argv);

    if (defined($display)) {
	push(@argv, -display => $display);
    } else {
	$display = $ENV{DISPLAY} || '';
    }
    if (defined($name)) {
	push(@argv, -name => $name);
    } else {
	($name = $0) =~ s{.*/}{};
    }
    if (defined($sync)) {
	push(@argv, "-sync");
    } else {
	$sync = 0;
    }
    $i = new Tcl;
    bless $i, $class;
    $i->SetVar2("env", "DISPLAY", $display, Tcl::GLOBAL_ONLY);
    $i->SetVar("argv0", $0, Tcl::GLOBAL_ONLY);
    if (defined $::tcl_library) {
	# hack to redefine search path for TCL installation
	$i->SetVar('tcl_library',$::tcl_library);
    }
    push(@argv, "--", @ARGV) if scalar(@ARGV);
    $i->SetVar("argv", [@argv], Tcl::GLOBAL_ONLY);
    # argc is just the values after the --, if any.
    # The other args are consumed by Tk.
    $i->SetVar("argc", scalar(@ARGV), Tcl::GLOBAL_ONLY);
    $i->SetVar("tcl_interactive", 0, Tcl::GLOBAL_ONLY);
    $i->SUPER::Init();
    $i->pkg_require('Tk', $i->GetVar('tcl_version'));
    # $i->update; # WinCE helper. TODO - remove from RELEASE
    my $mwid = $i->invoke('winfo','id','.');
    $W{PATH}->{$mwid} = '.';
    $W{INT}->{$mwid} = $i;
    $W{MWID}->{'.'} = $mwid;
    my $_mainwindow = \$mwid;
    $W{mainwindow}->{"$i"} = $_mainwindow;
    bless($_mainwindow, 'Tcl::Tk::Widget::MainWindow');
    $i->call('trace', 'add', 'command', '.', 'delete',
	 sub { for (keys %W) {$W{$_}->{$mwid} = undef; }});
    $i->ResetResult();
    $Tcl::Tk::TK_VERSION = $i->GetVar("tk_version");
    # Only do this for DEBUG() ?
    $Tk::VERSION = $Tcl::Tk::TK_VERSION;
    $Tk::VERSION =~ s/^(\d)\.(\d)/${1}0$2/;
    _DEBUG(1, "USING Tk $Tcl::Tk::TK_VERSION ($Tk::VERSION)\n") if DEBUG;
    unless (defined $tkinterp) {
	# first call, create command-helper in TCL to trace widget destruction
	$i->CreateCommand("::perl::w_del", \&widget_deletion_watcher);
    }
    $tkinterp = $i;
    return $i;
}

sub mainwindow {
    # this is a window with path '.'
    my $interp = shift;
    return $W{mainwindow}->{"$interp"};
}
sub tkinit {
    my $interp = Tcl::Tk->new(@_);
    $interp->mainwindow;
}
sub MainWindow {
    my $interp = Tcl::Tk->new(@_);
    $interp->mainwindow;
}

sub MainLoop {
    # This perl-based mainloop differs from Tk_MainLoop in that it
    # relies on the traced deletion of '.' instead of using the
    # Tk_GetNumMainWindows C API.
    # This could optionally be implemented with 'vwait' on a specially
    # named variable that gets set when '.' is destroyed.
    my $int = (ref $_[0]?shift:$tkinterp);
    my $mwid = $W{MWID}->{'.'};
    while (defined $Wpath->{$mwid}) {
	$int->DoOneEvent(0);
    }
}

#
# declare_widget, method of interpreter object
# args:
#   - a path of existing Tcl/Tk widget to declare its existance in Tcl::Tk
#   - (optionally) package name where this widget will be declared, default
#     is 'Tcl::Tk::Widget', but could be 'Tcl::Tk::Widget::somewidget'
sub declare_widget {
    my $int = shift;
    my $path = shift;
    my $widget_class = shift || 'Tcl::Tk::Widget';
    # JH: This is all SOOO wrong, but works for the simple case.
    # Issues that need to be addressed:
    #  1. You can create multiple interpreters, each containing identical
    #     pathnames.  This var should be better scoped.
    #	  VK: mostly resolved, such interpreters with pathnames allowed now
    #  2. There is NO cleanup going on.  We should somehow detect widget
    #     destruction (trace add command delete ... in 8.4) and interp
    #     destruction to clean up package variables.
    #my $id = $path=~/^\./ ? $int->invoke('winfo','id',$path) : $path;
    $int->invoke('trace', 'add', 'command', $path, 'delete', "::perl::w_del $path")
        if WIDGET_CLEANUP;
    my $id = $path;
    my $w = bless(\$id, $widget_class);
    $Wpath->{$id} = $path; # widget pathname
    $Wint->{$id}  = $int; # Tcl interpreter
    $W{RPATH}->{$path} = $w;
    return $w;
}
sub widget_deletion_watcher {
    my (undef,$int,undef,$path) = @_;
    #print STDERR "[D:$path]";
    $int->delete_widget_refs($path);
}

# widget_data return anonymous hash that could be used to hold any 
# user-specific data
sub widget_data {
    my $int = shift;
    my $path = shift;
    $Wdata->{$path} ||= {};
    return $Wdata->{$path};
}

sub frame($@) {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("frame", @_);
    return $int->declare_widget($path);
}
sub toplevel {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("toplevel", @_);
    return $int->declare_widget($path);
}
sub label {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("label", @_);
    return $int->declare_widget($path);
}
sub labelframe {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("labelframe", @_);
    return $int->declare_widget($path);
}
sub button {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("button", @_);
    return $int->declare_widget($path);
}
sub checkbutton {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("checkbutton", @_);
    return $int->declare_widget($path);
}
sub radiobutton {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("radiobutton", @_);
    return $int->declare_widget($path);
}
sub scale {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("scale", @_);
    return $int->declare_widget($path);
}
sub spinbox {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("spinbox", @_);
    return $int->declare_widget($path);
}
sub message {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("message", @_);
    return $int->declare_widget($path);
}
sub listbox {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("listbox", @_);
    return $int->declare_widget($path);
}
sub image {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("image", @_);
    return $int->declare_widget($path);
}
sub font {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("font", @_);
    return $int->declare_widget($path);
}
sub scrollbar {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("scrollbar", @_);
    return $int->declare_widget($path);
}
sub entry {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("entry", @_);
    return $int->declare_widget($path);
}
sub menu {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("menu", @_);
    return $int->declare_widget($path);
}
sub menubutton {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("menubutton", @_);
    return $int->declare_widget($path);
}
sub canvas {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("canvas", @_);
    return $int->declare_widget($path);
}
sub text {
    my $int = (ref $_[0]?shift:$tkinterp);
    my ($path) = $int->call("text", @_);
    return $int->declare_widget($path);
}
# subroutine awidget used to create [a]ny [widget]. Nothing complicated here,
# mainly needed for keeping track of this new widget and blessing it to right
# package
sub awidget {
    my $int = (ref $_[0]?shift:$tkinterp);
    my $wclass = shift;
    # Following is a suboptimal way of autoloading, there should exist a way
    # to Improve it.
    my $sub = sub {
        my $int = (ref $_[0]?shift:$tkinterp);
        my ($path) = $int->call($wclass, @_);
        return $int->declare_widget($path);
    };
    unless ($wclass=~/^\w+$/) {
	die "widget name '$wclass' contains not allowed characters";
    }
    # create appropriate method ...
    no strict 'refs';
    *{"Tcl::Tk::$wclass"} = $sub;
    # ... and call it (if required)
    if ($#_>-1) {
	return $sub->($int,@_);
    }
}
sub widget($@) {
    my $int = (ref $_[0]?shift:$tkinterp);
    my $wpath = shift;
    my $wtype = shift || 'Tcl::Tk::Widget';
    $wtype = "Tcl::Tk::Widget::$wtype" unless $wtype=~/^(?:Tcl::Tk::Widget)/;
    #if (exists $W{RPATH}->{$wpath}) {
    #    return $W{RPATH}->{$wpath};
    #}
    # We could ask Tcl about it by invoking
    # my @res = $int->Eval("winfo exists $wpath");
    # but we don't do it, as long as we allow any widget paths to
    # be used by user.
    my $w = $int->declare_widget($wpath,$wtype);
    return $w;
}
sub Exists($) {
    my $wid = shift;
    return 0 unless defined($wid);
    if (ref($wid)=~/^Tcl::Tk::Widget\b/) {
        my $wp = $wid->path;
        return $wid->interp->icall('winfo','exists',$wp);
    }
    return $tkinterp->icall('winfo','exists',$wid);
}
# do this only when tk_gestapo on?
# In normal case Tcl::Tk::Exists should be used.
#*{Tk::Exists} = \&Tcl::Tk::Exists;

sub widgets {
    \%W;
}

#sub after { 
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("after", @_) }
#sub bell { 
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("bell", @_) }
#sub bindtags {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("bindtags", @_) }
#sub clipboard { 
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("clipboard", @_) }
#sub destroy { 
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("destroy", @_) }
#sub exit { 
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("exit", @_) }
#sub fileevent {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("fileevent", @_) }
#sub focus {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("focus", @_) }
#sub grab {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("grab", @_) }
#sub lower {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("lower", @_) }
#sub option {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("option", @_) }
#sub place {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("place", @_) }
#sub raise {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("raise", @_) }
#sub selection {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("selection", @_) }
#sub tk {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("tk", @_) }
#sub tkwait {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("tkwait", @_) }
#sub update {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("update", @_) }
#sub winfo {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("winfo", @_) }
#sub wm {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("wm", @_) }
#sub property {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("property", @_);
#}
#
#sub grid {
#    my $int = (ref $_[0]?shift:$tkinterp);
#    $int->call("grid", @_);
#}
#sub bind {
#    my $int = shift;
#    $int->call("bind", @_);
#}
#sub pack {
#    my $int = shift;
#    $int->call("pack", @_);
#}

sub pkg_require {
    # Do Tcl package require with optional version, cache result.
    my $int = shift;
    my $pkg = shift;
    my $ver = shift;

    my $id = "$int$pkg"; # to made interpreter-wise, do stringification of $int

    _DEBUG(1, "PKG REQUIRE $pkg\n") if DEBUG;
    return $preloaded_tk{$id} if $preloaded_tk{$id};

    my @args = ("package", "require", $pkg);
    push(@args, $ver) if defined($ver);
    eval { $preloaded_tk{$id} = $int->icall(@args); };
    if ($@) {
	# Don't cache failures, as the package may become available by
	# changing auto_path and such.
	return;
    }
    return $preloaded_tk{$id};
}

sub need_tk {
    # DEPRECATED: Use pkg_require and call instead.
    my $int = shift;
    my $pkg = shift;
    my $cmd = shift || '';

    _DEBUG(1, "DEPRECATED CALL: need_tk($pkg, $cmd), use pkg_require\n") if DEBUG;
    if ($pkg eq 'pure-perl-Tk') {
        require Tcl::Tk::Widget;
    }
    elsif ($pkg eq 'ptk-Table') {
        require Tcl::Tk::Table;
    }
    else {
	# Only require the actual package once
	my $ver = $int->pkg_require($pkg);
	return 0 if !defined($ver);
	$int->Eval($cmd) if $cmd;
    }

    return 1;
}

sub tk_gestapo {
    # When placed first on the INC path, this will allow us to hijack
    # any requests for 'use Tk' and any Tk::* modules and replace them
    # with our own stuff.
    my ($coderef, $module) = @_;  # $coderef is to myself
    return undef unless $module =~ m!^Tk(/|\.pm$)!;

    my ($package, $callerfile, $callerline) = caller;

    my $fakefile;
    open(my $fh, '<', \$fakefile) || die "oops";

    $module =~ s!/!::!g;
    $module =~ s/\.pm$//;
    $fakefile = <<EOS;
package $module;
warn "### $callerfile:$callerline not really loading $module ###";
sub foo { 1; }
1;
EOS
    return $fh;
}
# subroutine findINC copied from perlTk/Tk.pm
sub findINC {
    my $file = join('/',@_);
    my $dir;
    $file  =~ s,::,/,g;
    foreach $dir (@INC) {
	my $path;
	return $path if (-e ($path = "$dir/$file"));
    }
    return undef;
}

#
# AUTOLOAD method for Tcl::Tk interpreter object, which will bring into
# existance interpreter methods
sub AUTOLOAD {
    my $int = shift;
    my $method = $Tcl::Tk::AUTOLOAD;
    # Separate method to autoload from (sub)package
    $method =~ s/^(Tcl::Tk::)//
	or die "weird inheritance ($method)";
    my $package = $1;

    # if someone calls $interp->_method(...) then it is considered as faster
    # version of method, similar to calling $interp->method(...) but via
    # 'invoke' instead of 'call', thus faster
    my $fast = '';
    $method =~ s/^_// and do {
	$fast='_';
	if (exists $::Tcl::Tk::{$method}) {
	    no strict 'refs';
	    *{"::Tcl::Tk::_$method"} = *{"::Tcl::Tk::$method"};
	    return $int->$method(@_);
	}
    };

    # search for right corresponding Tcl/Tk method, and create it afterwards
    # (so no consequent AUTOLOAD will happen)

    # Check to see if it is a camelCase method.  If so, split it apart.
    # code below will always create subroutine that calls a method.
    # This could be changed to create only known methods and generate error
    # if method is, for example, misspelled.
    # so following check will be like 
    #    if (exists $knows_method_names{$method}) {...}
    my $sub;
    if ($method =~ /^([a-z]+)([A-Z][a-z]+)$/) {
        my ($meth, $submeth) = ($1, lcfirst($2));
	# break into $method $submethod and call
	$sub = $fast ? sub {
	    my $int = shift;
	    $int->invoke($meth, $submeth, @_);
	} : sub {
	    my $int = shift;
	    $int->call($meth, $submeth, @_);
	};
    }
    else {
	# Default case, call as method of $int
	$sub = $fast ? sub {
	    my $int = shift;
	    $int->invoke($method, @_);
	} : sub {
	    my $int = shift;
	    $int->call($method, @_);
	};
    }
    no strict 'refs';
    *{"$package$fast$method"} = $sub;
    return $sub->($int,@_);
}

## ------------------------------------------------------------------------
## Widget package, responsible for all Tcl/Tk widgets and any other widgets
## Widgets are blessed to this package or to its sub-packages
## such as Tcl:Tk::Widget::Button, which ISA-Tcl::Tk::Widget
##

package Tcl::Tk::Widget;

use overload
    '""' => \&path,
    'eq' => sub {my $self = shift; return $self->path eq shift},
    'ne' => sub {my $self = shift; return $self->path ne shift};

sub iconimage {
    # this should set the wm iconimage/iconbitmap with an image
    warn "NYI: iconimage";
};

sub path {
    return $Wpath->{${$_[0]}};
}
# returns interpreter that is associated with widget
sub interp {
    unless (exists $Wint->{${$_[0]}}) {
	print caller;
	die "do not exist: ",${$_[0]};
    }
    return $Wint->{${$_[0]}};
}
# returns (and optionally creates) data hash assotiated with widget
sub widget_data {
    my $self = shift;
    return ($Wdata->{$self->path} || ($Wdata->{$self->path}={}));
}

#
# few geometry methods here
sub pack {
    my $self = shift;
    $self->interp->call("pack",$self,@_);
    $self;
}
sub grid {
    my $self = shift;
    $self->interp->call("grid",$self,@_);
    $self;
}
sub gridSlaves {
    # grid slaves returns widget names, so map them to their objects
    my $self = shift;
    my $int  = $self->interp;
    my @wids = $int->call("grid","slaves",$self,@_);
    map($int->widget($_), @wids);
}
sub place {
    my $self = shift;
    $self->interp->call("place",$self,@_);
    $self;
}
sub lower {
    my $self = shift;
    $self->interp->call("lower",$self,@_);
    $self;
}
# helper sub _bind_widget_helper inserts into subroutine callback
# widget as parameter
sub _bind_widget_helper {
    my $self = shift;
    my $sub = shift;
    if (ref($sub) eq 'ARRAY') {
	if ($#$sub>0) {
	    if (ref($sub->[1]) eq 'Tcl::Ev') {
		$sub = [$sub->[0],$sub->[1],$self,@$sub[2..$#$sub]];
	    }
	    else {
		$sub = [$sub->[0],$self,@$sub[1..$#$sub]];
	    }
	}
	else {
	    $sub = [$sub->[0], $self];
	}
	return $sub;
    }
    else {
	return sub{$sub->($self,@_)};
    }
}
sub bind {
    my $self = shift;
    # 'text' and 'canvas' binding could be different compared to common case
    # as long as Text uses 'tag bind' then we do not need to process it here
    if (ref($self) =~ /^Tcl::Tk::Widget::(?:Canvas)$/) {
	if ($#_==2) {
	    my ($tag, $seq, $sub) = @_;
	    $sub = $self->_bind_widget_helper($sub);
	    $self->interp->call($self,'bind',$tag,$seq,$sub);
	}
	elsif ($#_==1 && ref($_[1]) =~ /^(?:ARRAY|CODE)$/) {
	    my ($seq, $sub) = @_;
	    $sub = $self->_bind_widget_helper($sub);
	    $self->interp->call($self,'bind',$seq,$sub);
	}
	else {
	    $self->interp->call($self,'bind',@_);
	}
    }
    elsif (ref($self) =~ /^Tcl::Tk::Widget::(?:Listbox)$/) {
	if ($#_=1 && ref($_[1]) =~ /^(?:ARRAY|CODE)$/) {
	    my ($seq, $sub) = @_;
	    $sub = $self->_bind_widget_helper($sub);
	    $self->interp->call('bind',$self->path,$seq,$sub);
	}
	else {
	    $self->interp->call('bind',$self->path,@_);
	}
    }
    else {
	if ($_[0] =~ /^</) {
	    # A sequence was specified - assume path from widget instance
	    $self->interp->call("bind",$self->path,@_);
	} else {
	    # Not a sequence as first arg - don't assume path
	    $self->interp->call("bind",@_);
	}
    }
}
sub tag {
    my ($self,$verb,$tag, @rest) = @_;
    if ($verb eq 'bind') {
	return $self->tagBind($tag,@rest);
    }
    $self->interp->call($self, 'tag', $verb, $tag, @rest);
}
sub tagBind {
    my $self = shift;
    if ($#_==3 and ref($_[2]) eq 'REF') {
        my ($tag, $seq, $ref, $sub) = @_;
        $sub = $self->_bind_widget_helper($sub);
        return $self->interp->call($self,'tag','bind',$tag,$seq,$ref,$sub);
    }
    my ($tag, $seq, $sub) = @_;
    # 'text'
    # following code needs only to insert widget as a first argument to 
    # subroutine
    $sub = $self->_bind_widget_helper($sub);
    $self->interp->call($self, 'tag', 'bind', $tag, $seq, $sub);
}

# TODO - creating package/method in package only when needed
# move to separate file?
create_method_in_widget_package ('Canvas', 
    raise => sub {
	my $self = shift;
	my $wp = $self->path;
	$self->interp->call($wp,'raise',@_);
    },
    CanvasBind => sub {
	my $self = shift;
	$self->bind('item',@_);
    },
    CanvasFocus => sub {
	my $self = shift;
	$self->interp->call($self->path,'focus',@_);
    },
);

sub form {
    my $self = shift;
    my $int = $self->interp;
    $int->pkg_require("Tix");
    my @arg = @_;
    for (@arg) {
	if (ref && ref eq 'ARRAY') {
	    $_ = join ' ', map {
		  (ref && (ref =~ /^Tcl::Tk::Widget\b/))?
		    $_->path  # in this case there is form geometry relative
		              # to widget; substitute its path
		  :$_} @$_;
	    s/^& /&/;
	}
    }
    $int->call("tixForm",$self,@arg);
    $self;
}

# TODO -- these methods could be AUTOLOADed
sub focus {
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('focus',$wp,@_);
}
sub destroy {
    my $self = shift;
    my $int = $self->interp;
    my $wp = $self->path;
    $int->call('destroy',$wp,@_);
    $int->delete_widget_refs($wp);
}

# for compatibility (TODO -- more methods could be AUTOLOADed)
sub GeometryRequest {
    my $self = shift;
    my $wp = $self->path;
    my ($width,$height) = @_;
    $self->interp->call('wm','geometry',$wp,"=${width}x$height");
}
sub OnDestroy {
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('bind','<Destroy>',$wp,@_);
}
sub grab {
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('grab',$wp,@_);
}
sub grabRelease {
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('grab','release',$wp,@_);
}
sub packAdjust {
    # old name, becomes pack configure
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('pack','configure',$wp,@_);
}
sub optionGet {
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('option','get',$wp,@_);
}
sub raise {
    my $self = shift;
    my $wp = $self->path;
    $self->interp->call('raise',$wp,@_);
}

sub update {
    my $self = shift;
    $self->interp->update;
}
sub ItemStyle {
    my $self = shift;
    my $styl = shift;
    my $wp   = $self->path;
    my $int  = $self->interp;
    $int->pkg_require('Tix');
    my %args = @_;
    $args{'-refwindow'} = $wp unless exists $args{'-refwindow'};
    $int->call('tixDisplayStyle', $styl, %args);
}
sub getOpenFile {
    my $self = shift;
    my %args = @_;
    $args{'-parent'} = $self->path unless defined $args{'-parent'};
    $self->interp->call('tk_getOpenFile', %args);
}
sub getSaveFile {
    my $self = shift;
    my %args = @_;
    $args{'-parent'} = $self->path unless defined $args{'-parent'};
    $self->interp->call('tk_getSaveFile', %args);
}
sub chooseDirectory {
    my $self = shift;
    my %args = @_;
    $args{'-parent'} = $self->path unless defined $args{'-parent'};
    $self->interp->call('tk_chooseDirectory', %args);
}
sub messageBox {
    my $self = shift;
    my %args = @_;
    $args{'-parent'} = $self->path unless defined $args{'-parent'};
    # messageBox should handle pTk's "YesNo" and return "Yes" in
    # addition to Tk's standard all-lc in/out.
    #$args{'-type'} = lc $args{'-type'} if defined $args{'-type'};
    $self->interp->call('tk_messageBox', %args);
}

# TODO all Busy subs
sub Busy {
    my $self = shift;
    print STDERR "Busy = TODO\n";
    $self;
}
sub Unbusy {
    my $self = shift;
    print STDERR "Unbusy = TODO\n";
    $self;
}

# subroutine Darken copied from perlTk/Widget.pm
# tkDarken --
# Given a color name, computes a new color value that darkens (or
# brightens) the given color by a given percent.
#
# Arguments:
# color - Name of starting color.
# perecent - Integer telling how much to brighten or darken as a
# percent: 50 means darken by 50%, 110 means brighten
# by 10%.
sub Darken
{
    my ($w,$color,$percent) = @_;
    my @l = $w->rgb($color);
    my $red = $l[0]/256;
    my $green = $l[1]/256;
    my $blue = $l[2]/256;
    $red = int($red*$percent/100);
    $red = 255 if ($red > 255);
    $green = int($green*$percent/100);
    $green = 255 if ($green > 255);
    $blue = int($blue*$percent/100);
    $blue = 255 if ($blue > 255);
    sprintf('#%02x%02x%02x',$red,$green,$blue);
}

sub PathName {
    my $wid = shift;
    return $wid->path;
}
sub Exists {
    my $wid = shift;
    my $wp = $wid->path;
    return $wid->interp->icall('winfo','exists',$wp);
}
sub toplevel {
    my $wid = shift;
    my $int = $wid->interp;
    my $tlp = $int->icall('winfo','toplevel',$wid->path);
    if ($tlp eq '.') {return $int->mainwindow}
    return $int->widget($tlp);
}
sub parent {
    my $wid = shift;
    my $int = $wid->interp;
    my $res = $int->icall('winfo','parent',$wid->path);
    if ($res eq '') {return ''}
    if ($res eq '.') {return $int->mainwindow}
    return $int->widget($res);
}

sub bell {
    my $self = shift;
    my $int = $self->interp;
    my $ret = $int->call('bell', @_);
}
sub children {
    my $self = shift;
    my $int  = $self->interp;
    my @wids = $int->call('winfo', 'children', $self->path, @_);
    # winfo children returns widget paths, so map them to objects
    return map ($int->widget($_), @wids);
}

# although this is not the case, we'll think of object returned by 'after'
# as a widget.
sub after {
    my $self = shift;
    my $int = $self->interp;
    my $ret = $int->call('after', @_);
    return $int->declare_widget($ret);
}
sub cancel {
    my $self = shift;
    return $self->interp->call('after','cancel',$self);
}

#
# Getimage compatability routine
#

my %image_formats =
    (
     xpm => 'photo',
     gif => 'photo',
     ppm => 'photo',
     xbm => 'bitmap'
     );

sub Getimage {
    my $self = shift;
    my $name = shift;
    my $images;

    return $images->{$name} if $images->{$name};

    my $int = $self->interp;
    foreach my $ext (keys %image_formats) {
	my $path;
	foreach my $dir (@INC) {
	    $path = "$dir/Tk/$name.$ext";
	    last if -f $path;
	}
	next unless -f $path;
	_DEBUG(2, "Getimage: FOUND IMAGE $path\n") if DEBUG;
	if ($ext eq "xpm") {
	    $int->pkg_require('img::xpm');
	}
	my @args = ('image', 'create', $image_formats{$ext}, -file => $path);
	if ($image_formats{$ext} ne "bitmap") {
	    push @args, -format => $ext;
	}
	$images->{$name} = $int->call(@args);
	return $images->{$name};
    }

    # Try built-in bitmaps from Tix
    #$images->{$name} = $w->Pixmap( -id => $name );
    #return $images->{$name};
    _DEBUG(1, "Getimage: MISSING IMAGE $name\n") if DEBUG;
    return;
}

#
# some class methods to provide same syntax as perlTk do
# In this case all widget names are generated automatically.
#

# global widget counter, only for autogenerated widget names.
my $gwcnt = '01'; 

sub w_uniq {
    my ($self, $type) = @_;
    # create unique widget id with path "$self.$type<uniqid>"
    # assume produced names are unique (without checking for already generated
    # names) since $gwcnt incremented *each* call to w_uniq
    # Issues to resolve:
    #  - widgets created in Tcl could (rarely!) have same hence conflicting
    #    name, should detect such cases
    #  - could be reasonable to respect user's -name option, for compatibility
    if (!defined($type)) {
	my ($package, $callerfile, $callerline) = caller;
	warn "$callerfile:$callerline called w_uniq(@_)";
	$type = "unk";
    }
    my $wp = $self->path;
    # Ensure that we don't end up with '..btn01' as a widget name
    $wp = '' if $wp eq '.';
    $gwcnt++;
    Tcl::_current_refs_widget("$wp.$type$gwcnt");
    return "$wp.$type$gwcnt";
}

# perlTk<->Tcl::Tk mapping in form [widget, wprefix, ?package?]
# These will be looked up 1st in AUTOLOAD
my %ptk2tcltk =
    (
     Button      => ['button', 'btn',],
     Checkbutton => ['checkbutton', 'cb',],
     Canvas      => ['canvas', 'can',],
     Entry       => ['entry', 'ent',],
     Frame       => ['frame', 'f',],
     LabelFrame  => ['labelframe', 'lf',],
     #LabFrame    => ['labelframe', 'lf',],
     Label       => ['label', 'lbl',],
     Listbox     => ['listbox', 'lb',],
     Message     => ['message', 'msg',],
     Menu        => ['menu', 'mnu',],
     Menubutton  => ['menubutton', 'mbtn',],
     Panedwindow => ['panedwindow', 'pw',],
     Bitmap	 => ['image', 'bmp',],
     Photo	 => ['image', 'pht',],
     Radiobutton => ['radiobutton', 'rb',],
     ROText	 => ['text', 'rotext',],
     Text        => ['text', 'text',],
     Scrollbar   => ['scrollbar','sb',],
     Scale       => ['scale','scl',],
     TextUndo    => ['text', 'utext',],
     Toplevel    => ['toplevel', 'top',],

     #Table       => ['*perlTk/Table',]
     Table       => ['table', 'tbl', 'Tktable'],

     BrowseEntry => ['ComboBox', 'combo', 'BWidget'],
     ComboBox    => ['ComboBox', 'combo', 'BWidget'],
     ListBox     => ['ListBox', 'lb', 'BWidget'],
     BWTree      => ['Tree', 'bwtree', 'BWidget'],
     ScrolledWindow => ['ScrolledWindow', 'sw', 'BWidget'],

     TileNoteBook => ['tile::notebook', 'tnb', 'tile'],

     Treectrl    => ['treectrl', 'treectrl', 'treectrl'],

     Balloon     => ['tixBalloon', 'bl', 'Tix'],
     DirTree     => ['tixDirTree', 'dirtr', 'Tix'],
     HList       => ['tixHList', 'hlist', 'Tix'],
     TList       => ['tixTList', 'tlist', 'Tix'],
     NoteBook    => ['tixNoteBook', 'nb', 'Tix'],
     );

# Mapping of pTk camelCase names to Tcl commands.
# These do not require the actual widget name.
# These will be looked up 2nd in AUTOLOAD
# $w->mapCommand(...) => @qwargs ...
my %ptk2tcltk_mapper =
    (
     "optionAdd"        => [ qw(option add) ],
     "font"             => [ qw(font) ],
     "fontCreate"       => [ qw(font create) ],
     "fontNames"        => [ qw(font names) ],
     "waitVariable"     => [ qw(vwait) ], # was tkwait variable
     "idletasks"        => [ qw(update idletasks) ],
     );

# wm or winfo subroutines, to be checked 4th in AUTOLOAD
# $w->wmcommand(...) => wm|winfo wmcommand $w ...
my %ptk2tcltk_wm =
    (
     "deiconify"     => 'wm',
     "geometry"      => 'wm', # note 'winfo geometry' isn't included
     "group"         => 'wm',
     "iconify"       => 'wm',
     "iconname"      => 'wm',
     "minsize"       => 'wm',
     "maxsize"       => 'wm',
     "protocol"      => 'wm',
     "resizable"     => 'wm',
     "stackorder"    => 'wm',
     "state"         => 'wm',
     "title"         => 'wm',
     "transient"     => 'wm',
     "withdraw"      => 'wm',
     ( 
	 # list of widget pTk methods mapped to 'winfo' Tcl/Tk methods
	 # following lines result in pairs  'method' => 'winfo'
	 map {$_=>'winfo'} qw(
	     atom atomname
	     cells children class colormapfull containing
	     depth
	     fpixels
	     height
	     id interps ismapped
	     manager
	     name
	     pathname pixels pointerx pointery
	     reqheight reqwidth  rgb  rootx rooty
	     screen screencells screendepth screenvisual
	     screenheight screenwidth screenmmheight screenmmwidth server
	     viewable visual visualid visualsavailable vrootheight vrootwidth
	     vrootx vrooty
	     width
	     x y
         ),
     )
     );

my $ptk_w_names = join '|', sort keys %ptk2tcltk;


#  create_ptk_widget_sub creates subroutine similar to following:
#sub Button {
#  my $self = shift; # this will be a parent widget for newer button
#  my $int = $self->interp;
#  my $w    = w_uniq($self, "btn");
#  # create 'button' widget with a unique path
#  return $int->button($w,@_);
#}
my %replace_options =
    (
     tixHList   => {separator=>'-separator'},
     ComboBox   => {-choices=>'-values'},
     table      => {-columns=>'-cols'},
     toplevel   => {-title=>sub{shift->title(@_)},OnDestroy=>sub{},-overanchor=>undef},
     labelframe => {-label=>'-text', -labelside => undef},
     );
my %pure_perl_tk = (); # hash to keep track of pure-perl widgets

sub create_ptk_widget_sub {
    my ($interp,$wtype,$fast) = @_;
    my ($ttktype,$wpref,$tpkg,$tcmd) = @{$ptk2tcltk{$wtype}};
    $wpref ||= lcfirst $wtype;

    $interp->pkg_require($tpkg) if $tpkg; # should be moved into widget creation sub?
    $interp->Eval($tcmd)        if $tcmd; # should be moved into widget creation sub? 

    if ($ttktype =~ s/^\*perlTk\///) {
	# should create pure-perlTk widget and bind it to Tcl variable so that
	# anytime a method invoked it will be redirected to Perl
	return sub {
	  my $self = shift; # this will be a parent widget for newer widget
	  my $int  = $self->interp;
          my $w    = w_uniq($self, $wpref); # create uniq pref's widget id
	  die "pure-perlTk widgets are not implemented";
	};
    }
    if (exists $replace_options{$ttktype}) {
	return sub {
	    my $self = shift; # this will be a parent widget for newer widget
	    my $int = $self->interp;
	    my $w    = w_uniq($self, $wpref); # create uniq pref's widget id
	    my %args = @_;
	    my @code_todo;
	    for (keys %{$replace_options{$ttktype}}) {
		if (defined($replace_options{$ttktype}->{$_})) {
		    if (exists $args{$_}) {
		        if (ref($replace_options{$ttktype}->{$_}) eq 'CODE') {
			    push @code_todo, [$replace_options{$ttktype}->{$_}, delete $args{$_}];
			}
			else {
			    $args{$replace_options{$ttktype}->{$_}} =
			        delete $args{$_};
			}
		    }
		} else {
		    delete $args{$_} if exists $args{$_};
		}
	    }
	    my $wid = $int->declare_widget($int->call($ttktype,$w,%args), "Tcl::Tk::Widget::$wtype");
	    $_->[0]->($wid,$_->[1]) for @code_todo;
	    return $wid;
	};
    }
    return $fast ? sub {
	my $self = shift; # this will be a parent widget for newer widget
	my $int  = $self->interp;
        my $w    = w_uniq($self, $wpref); # create uniq pref's widget id
	my $wid  = $int->declare_widget($int->invoke($ttktype,$w,@_), "Tcl::Tk::Widget::$wtype");
	return $wid;
    } : sub {
	my $self = shift; # this will be a parent widget for newer widget
	my $int  = $self->interp;
        my $w    = w_uniq($self, $wpref); # create uniq pref's widget id
	my $wid  = $int->declare_widget($int->call($ttktype,$w,@_), "Tcl::Tk::Widget::$wtype");
	return $wid;
    };
}
my %special_widget_abilities = ();
sub LabFrame {
    my $self = shift; # this will be a parent widget for newer labframe
    my $int  = $self->interp;
    my $w    = w_uniq($self, "lf"); # create uniq pref's widget id
    my $ttktype = "labelframe";
    my %args = @_;
    for (keys %{$replace_options{$ttktype}}) {
	if (defined($replace_options{$ttktype}->{$_})) {
	    $args{$replace_options{$ttktype}->{$_}} =
		delete $args{$_} if exists $args{$_};
	} else {
	    delete $args{$_} if exists $args{$_};
	}
    }
    my $wtype = 'LabFrame';
    create_widget_package($wtype);
    my $lf = $int->declare_widget($int->call($ttktype, $w, %args), "Tcl::Tk::Widget::$wtype");
    create_method_in_widget_package($wtype,
	Subwidget => sub {
	    my $lf = shift;
	    _DEBUG(1, "LabFrame $lf ignoring Subwidget(@_)\n") if DEBUG;
	    return $lf;
	},
    );
    return $lf;
}
sub ROText {
    # Read-only text
    # This just needs to intercept the programmatic insert/delete
    # and reenable the text widget for that duration.
    my $self = shift; # this will be a parent widget for newer ROText
    my $int  = $self->interp;
    my $w    = w_uniq($self, "rotext"); # create uniq pref's widget id
    my $wtype = 'ROText';
    create_widget_package($wtype);
    my $text = $int->declare_widget($int->call('text', $w, @_), "Tcl::Tk::Widget::$wtype");
    create_method_in_widget_package($wtype,
	insert => sub {
	    my $wid = shift;
	    my $int = $self->interp;
	    $wid->configure(-state => "normal");
	    # avoid recursive call by going directly to interp
	    $int->call($wid, 'insert', @_);
	    $wid->configure(-state => "disabled");
	},
	delete => sub {
	    my $wid = shift;
	    my $int = $self->interp;
	    $wid->configure(-state => "normal");
	    # avoid recursive call by going directly to interp
	    $int->call($wid, 'delete', @_);
	    $wid->configure(-state => "disabled");
	}
    );
    $text->configure(-state => "disabled");
    return $text;
}

# menu compatibility
sub _process_menuitems;
sub _process_underline {
    # Suck out "~" which represents the char to underline
    my $args = shift;
    if (defined($args->{'-label'}) && $args->{'-label'} =~ /~/) {
	my $und = index($args->{'-label'}, '~');
	$args->{'-underline'} = $und;
	$args->{'-label'} =~ s/~//;
    }
};
# internal sub helper for menu
sub _addcascade {
    my $mnu = shift;
    my $mnup = $mnu->path;
    my $int = $mnu->interp;
    my $smnu = $mnu->Menu; # return unique widget id
    my %args = @_;
    my $tearoff = delete $args{'-tearoff'};
    if (defined($tearoff)) {
        $smnu->configure(-tearoff => $tearoff);
    }
    $args{'-menu'} = $smnu;
    my $mis = delete $args{'-menuitems'};
    _process_menuitems($int,$smnu,$mis);
    _process_underline(\%args);
    #$int->call("$mnu",'add','cascade', %args);
    $mnu->add('cascade',%args);
    return $smnu;
}
# internal helper sub to process perlTk's -menuitmes option
sub _process_menuitems {
    my ($int,$mnu,$mis) = @_;
    for (@$mis) {
	if (ref) {
	    my $label = $_->[1];
	    my %a = @$_[2..$#$_];
	    $a{'-state'} = delete $a{state} if exists $a{state};
	    $a{'-label'} = $label;
	    my $cmd = lc($_->[0]);
	    if ($cmd eq 'separator') {$int->invoke($mnu->path,'add','separator');}
	    elsif ($cmd eq 'cascade') {
		_process_underline(\%a);
	        _addcascade($mnu, %a);
	    }
	    else {
		$cmd=~s/^button$/command/;
		_process_underline(\%a);
	        $int->call($mnu->path,'add',$cmd, %a);
	    }
	}
	else {
	    if ($_ eq '-' or $_ eq '') {
		$int->invoke($mnu->path,'add','separator');
	    }
	    else {
		die "in menubutton: '$_' not implemented";
	    }
	}
    }
}
sub Menubutton {
    my $self = shift; # this will be a parent widget for newer menubutton
    my $int = $self->interp;
    my $w    = w_uniq($self, "mb"); # create uniq pref's widget id
    my %args = @_;
    my $mcnt = '01';
    my $mis = delete $args{'-menuitems'};
    my $tearoff = delete $args{'-tearoff'};
    $args{'-state'} = delete $args{state} if exists $args{state};

    create_widget_package('Menu');
    create_widget_package('Menubutton');
    create_method_in_widget_package('Menubutton',
	command=>sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    my %args = @_;
	    _process_underline(\%args);
	    $int->call("$wid.m",'add','command',%args);
	},
	checkbutton => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    $int->call("$wid.m",'add','checkbutton',@_);
	},
	radiobutton => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    $int->call("$wid.m",'add','radiobutton',@_);
	},
	separator => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    $int->call("$wid.m",'add','separator',@_);
	},
	menu => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    return $int->widget("$wid.m");
	},
	cget => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    if ($_[0] eq "-menu") {
		return $int->widget($int->invoke("$wid",'cget','-menu'));
	    } else {
		_DEBUG(2, "CALL $wid cget @_\n") if DEBUG;
		die "Finish cget implementation for Menubutton";
	    }
	});
    my $mnub = $int->menubutton($w, -menu => "$w.m", %args);
    my $mnu  = $int->menu("$w.m");
    bless $mnub, "Tcl::Tk::Widget::Menubutton";
    bless $mnu, "Tcl::Tk::Widget::Menu";
    _process_menuitems($int,$mnu,$mis);
    $int->update if DEBUG;
    if (defined($tearoff)) {
        $mnu->configure(-tearoff => $tearoff);
    }
    return $mnub;
}
sub Menu {
    my $self = shift; # this will be a parent widget for newer menu
    my $int  = $self->interp;
    my $w    = w_uniq($self, "menu"); # return unique widget id
    my %args = @_;

    my $mis         = delete $args{'-menuitems'};
    $args{'-state'} = delete $args{state} if exists $args{state};

    _DEBUG(2, "MENU (@_), creating $w\n") if DEBUG;

    create_widget_package('Menu');
    create_method_in_widget_package('Menu',
	command => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    my %args = @_;
	    _process_underline(\%args);
	    $int->call("$wid",'add','command',%args);
	},
	checkbutton => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    $int->call("$wid",'add','checkbutton',@_);
	},
	radiobutton => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    $int->call("$wid",'add','radiobutton',@_);
	},
	cascade => sub {
	    my $wid = shift;
	    _addcascade($wid, @_);
	},
	separator => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    $int->call("$wid",'add','separator',@_);
	},
	menu => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    return $int->widget("$wid");
	},
	cget => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    if ($_[0] eq "-menu") {
		return $int->widget("$wid");
	    } else {
		_DEBUG(1, "CALL $wid cget @_\n") if DEBUG;
		die "Finish cget implementation for Menu";
	    }
	},
	entryconfigure => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    my $label = shift;
	    $label =~ s/~//;
	    $int->call("$wid", 'entryconfigure', $label, @_);
	},
    );
    my $mnu = $int->menu($w, %args);
    $int->update if DEBUG;
    bless $mnu, "Tcl::Tk::Widget::Menu";
    _process_menuitems($int,$mnu,$mis);
    return $mnu;
}
# Balloon widget's method are in Tcl/Tk/Widget/Balloon.pm
sub Balloon {
    my $self = shift; # this will be a parent widget for newer balloon
    my $int = $self->interp;
    my $w    = w_uniq($self, "bln"); # return unique widget id
    $int->pkg_require('Tix');
    my $wtype = 'Balloon';
    require "Tcl/Tk/Widget/$wtype.pm";
    my $bw = $int->declare_widget($int->call('tixBalloon', $w, @_), "Tcl::Tk::Widget::$wtype");
    return $bw;
}
sub NoteBook {
    my $self = shift; # this will be a parent widget for newer notebook
    my $int = $self->interp;
    my $w    = w_uniq($self, "nb"); # return unique widget id
    $int->pkg_require('Tix');
    my %args = @_;
    delete $args{'-tabpady'};
    delete $args{'-inactivebackground'};
    my $wtype = 'NoteBook';
    create_widget_package($wtype);
    my $bw = $int->declare_widget($int->call('tixNoteBook', $w, %args), "Tcl::Tk::Widget::$wtype");
    create_method_in_widget_package($wtype,
	add=>sub {
	    my $bw = shift;
	    my $int = $bw->interp;
	    my $wp = $int->call($bw,'add',@_);
	    my $ww = $int->declare_widget($wp);
	    return $ww;
	},
    );
    return $bw;
}
sub DialogBox {
    # pTk DialogBox compat sub
    # XXX: This is not complete, needs to handle additional options
    my $self = shift; # this will be a parent widget for newer DialogBox
    my $int  = $self->interp;
    my $wn    = w_uniq($self, "dlgbox"); # return unique widget id
    my %args = @_;
    my $dlg  = $int->declare_widget($int->call('toplevel', $wn,
					       -class => "Dialog"));
    $dlg->withdraw();
    $dlg->title($args{'-title'} || "Dialog Box");
    my $topparent = $int->call('winfo', 'toplevel', $self);
    $dlg->transient($topparent);
    $dlg->group($topparent);
    my $bot  = $dlg->Frame();
    $bot->pack(-side => "bottom", -fill => "x", -expand => 0);
    my $btn;
    my $defbtn;
    foreach (reverse @{$args{'-buttons'}}) {
	$btn = $bot->Button(-text => $_,
			    -command => ['set', '::tk::Priv(button)', "$_"]);
	if ($args{'-default_button'} && $_ eq $args{'-default_button'}) {
	    $defbtn = $btn;
	    $btn->configure(-default => "active");
	    # Add <Return> binding to invoke the default button
	    $dlg->bind('<Return>', ["$btn", "invoke"]);
	}
	if ($^O eq "MSWin32") {
	    # should be done only on Tk >= 8.4
	    $btn->configure(-width => "-11");
	}
	$btn->pack(-side => "right", -padx => 4, -pady => 5);
    }
    # We need to create instance methods for dialogs to handle their
    # perl-side instance variables -popover and -default_button
    $dlg->widget_data->{'-popover'} = $args{'-popover'} || "cursor";
    $dlg->widget_data->{'-default'} = $defbtn;
    # Add Escape and Destroy bindings to trigger vwait
    # XXX Remove special hash items as well
    $dlg->bind('<Destroy>', 'set ::tk::Priv(button) {}');
    $dlg->bind('<Escape>', 'set ::tk::Priv(button) {}');
    my $wtype = 'DialogBox';
    create_widget_package($wtype);
    create_method_in_widget_package($wtype,
	add => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    my $wtype = shift;
	    my %args  = @_;
	    my $subw;
	    {
		no strict 'refs';
		$subw = &{"Tcl::Tk::Widget::$wtype"}($wid, %args);
	    }
	    $subw->pack(-side => "top", -fill => "x", -expand => 1);
	    return $subw;
	},
	Show => sub {
	    my $wid = shift;
	    my $int = $wid->interp;
	    my $grabtype = shift;
	    # Grab pertinent instance data
	    my $defbtn  = $wid->widget_data->{'-default'};
	    my $popover = $wid->widget_data->{'-popover'};

	    # ::tk::PlaceWindow is Tk 8.4+
	    if ($popover eq "cursor") {
		$int->call('::tk::PlaceWindow', $wid, 'pointer', 'center');
	    } elsif (Tcl::Tk::Exists($popover)) {
		$int->call('::tk::PlaceWindow', $wid, 'widget', $popover);
	    } else {
		$int->call('::tk::PlaceWindow', $wid);
	    }
	    $int->grab($wid);
	    $int->focus($defbtn) if $defbtn;
	    $int->call('vwait', '::tk::Priv(button)');
	    my $val = $int->GetVar2('::tk::Priv', 'button');
	    eval {
		# Window may have been destroyed
		$int->call('grab', 'release', $wid);
		$int->call('wm', 'withdraw', $wid);
	    };
	    return $val;
	},
	Hide => sub {
	    # This will trigger grab release and withdraw
	    $int->SetVar2('::tk::Priv', 'button', '');
	},
    );
    return bless $dlg, "Tcl::Tk::Widget::$wtype";
}
sub Dialog {DialogBox(@_)}
sub Photo {
    my $self = shift; # this will be a parent widget for newer Photo
    my $int = $self->interp;
    my $w    = w_uniq($self, "pht"); # return unique widget id
    # XXX Do we really want to require all of 'Img' here?  Perhaps the
    # XXX requirement on Img should be pushed to the user level, or only
    # XXX require those formats that Perl/Tk auto-supported (jpeg, ???)
    # VK how differents format should be differentiated? TBD
    #$int->pkg_require('Img');
    create_widget_package('Photo');
    my $bw = $int->declare_widget($int->call('image','create', 'photo', @_),
         "Tcl::Tk::Widget::Photo");
    return $bw;
}
sub Bitmap {
    my $self = shift; # this will be a parent widget for newer Bitmap
    my $int = $self->interp;
    my $w    = w_uniq($self, "bmp"); # return unique widget id
    create_widget_package('Bitmap');
    my $bw = $int->declare_widget($int->call('image','create', 'bitmap', @_),
	"Tcl::Tk::Widget::Bitmap");
    return $bw;
}

my %subwidget_options =
    (
     Tree => [
	 '-columns', '-drawbranch', '-gap', '-header', '-height',
	 '-indent', '-indicator', '-indicatorcmd', '-itemtype',
	 '-padx', '-pady', '-sizecmd', '-separator', '-width',
     ],
     );
sub Tree {
    my $self = shift; # this will be a parent widget for newer tree
    my $int = $self->interp;
    my $w    = w_uniq($self, "tree"); # return unique widget id
    $int->pkg_require('Tix');
    my %args = @_;
    my %sub_args;
    foreach (@{$subwidget_options{'Tree'}}) {
	$sub_args{$_} = delete $args{$_} if exists $args{$_};
    }
    # The hlist options must be passed in -options are creation time
    # as a Tcl list.  Build a Perl array that will be auto-converted
    # to a Tcl list in 'call'.
    my @opts;
    foreach my $opt (keys %sub_args) {
	my $cname = $opt;
	$cname =~ s/^-//;
	push @opts, "hlist.$cname", $sub_args{$opt};
    }
    $args{'-options'} = \@opts;
    my $wtype = 'Tree';
    create_widget_package($wtype);
    my $tree = $int->declare_widget($int->call('tixTree', $w, %args),
    	"Tcl::Tk::Widget::$wtype");
    # We don't need special_widget_abilities as long as a recent Tix
    # is used that passes the HList method calls to its subwidget
    # automatically.
    return $tree;
}

# Scrolled is implemented via BWidget ScrolledWindow using MultipleWidget
sub Scrolled {
    _DEBUG(1, "SCROLLED (@_)\n") if DEBUG;
    my $self = shift; # this will be a parent widget for newer Scrolled
    my $int = $self->interp;
    my $wtype = shift; # what type of scrolled widget
    die "wrong 'scrolled' type $wtype" unless $wtype =~ /^\w+$/;

    # translate Scrolled parameter
    my %args = @_;
    my $sb = delete $args{'-scrollbars'};
    if ($sb) {
	# TODO (easy one) -- really process parameters to scrollbar. 
	# Now let them be just like 'osoe'
    }

    # We need to create a list of widgets that do their own scrolling
    if ($wtype eq 'Tree') {
	$args{'-scrollbar'} = "auto";
	return Tree($self, %args);
    }

    # Use BWidget ScrolledWindow as wrapper widget
    $int->pkg_require('BWidget');
    my $w  = w_uniq($self, "sc"); # return unique widget id
    my $sw = $int->icall('ScrolledWindow', $w,
			-auto=>'both', -scrollbar=>'both');
    $sw = $int->declare_widget($sw);
    my $subw;
    {
	no strict 'refs';  # another option would be hash with values as subroutines
	$subw = &{"Tcl::Tk::Widget::$wtype"}($sw, %args);
    }
    $sw->setwidget($subw);
    my $mmw = new Tcl::Tk::Widget::MultipleWidget (
	$int,
	$subw, ['&','-'], # all methods and options redirected to $subw
	$sw, ['*'],       # all geometry methods redirected to $sw
    );
    return $mmw;
}

# substitute Tk's "tk_optionMenu" for this
sub Optionmenu {
    my $self = shift; # this will be a parent widget for newer Optionmenu
    my $int = $self->interp;

    # translate parameters
    my %args = @_;

    my $w  = w_uniq($self, "om"); # return unique widget id
    my $vref = \do{my $r};
    $vref = delete $args{'-variable'} if exists $args{'-variable'};
    my $options = delete $args{'-options'} if exists $args{'-options'};
    my $replopt = {};
    for (@$options) {
	if (ref) {
	    # anon array [lab=>val]
	    $replopt->{$_->[0]} = $_->[1];
	    $_ = $_->[0];
	}
    }
    my $mnu = $int->call('tk_optionMenu', $w, $vref, @$options);
    $mnu = $int->declare_widget($mnu);
    $w = $int->declare_widget($w);
    my $mmw;
    $mmw = new Tcl::Tk::Widget::MultipleWidget (
        $int,
        $w, ['&','-','*','-variable'=>\$vref,
	    '-textvariable'=>sub {
		my ($w,$optnam,$optval) = @_;
		if (exists $mmw->{_replopt}->{$$vref}) {
		    return \$mmw->{_replopt}->{$$vref};
		}
		return $vref;
	    },
	    '-menu'=> \$mnu,
	    '-options'=>sub {
		print STDERR "***options: {@_}\n";
		my ($w,$optnam,$optval) = @_;
		for (@$optval) {
		    $w->add('command',$_);
		}
	    },
         ],
	 $mnu, ['&entrycget',],
    );
    $mmw->{_replopt} = $replopt if defined $replopt;
    #for (keys %args) {$mmw->configure($_=>$args{$_})}
    return $mmw;
}

# TODO -- document clearly how to use this subroutine
sub Declare {
    my $w       = shift;
    my $wtype   = shift;
    my $ttktype = shift;
    my %args    = @_;

    # Allow overriding of existing widgets.
    # XXX This should still die if we have created any single instance
    # XXX of this widget already.
    #die "$wtype already created\n" if defined $ptk2tcltk{$wtype};
    if (!exists $args{'-prefix'}) {
	$args{'-prefix'} ||= lcfirst $ttktype;
	$args{'-prefix'} =~ s/\W+//g;
    }
    $wtype = quotemeta($wtype); # to prevent chars corrupting regexp
    $ptk2tcltk{$wtype} = [$ttktype, $args{'-prefix'}, $args{'-require'},
			  $args{'-command'}];
    $ptk_w_names .= "|$wtype";
}

# here we create Widget package, used for both standard cases like
# 'Button', 'Label', and so on, and for all other widgets like Baloon
# TODO : document better and provide as public way of doing things?
my %created_w_packages; # (may be look in global stash %:: ?)
sub create_widget_package {
    my $widgetname = shift;
    _DEBUG(2, "AUTOCREATE widget $widgetname (@_)\n") if DEBUG;
    unless (exists $created_w_packages{$widgetname}) {
        _DEBUG(1, "c-PACKAGE $widgetname (@_)\n") if DEBUG;
	$created_w_packages{$widgetname} = {};
	die "not allowed widg name $widgetname" unless $widgetname=~/^\w+$/;
	# here we create Widget package
	my $package = $Tcl::Tk::VTEMP;
	$package =~ s/\[\[widget-repl\]\]/$widgetname/g;
	eval "$package";
	die $@ if $@;
	# Add this widget class to ptk_w_names so the AUTOLOADer properly
	# identifies it for creating class methods
	$widgetname = quotemeta($widgetname); # to prevent chars corrupting regexp
	$ptk_w_names .= "|$widgetname";
    }
}
# this subroutine creates a method in widget's package
sub create_method_in_widget_package {
    my $widgetname = shift;
    create_widget_package($widgetname);
    while ($#_>0) {
	my $widgetmethod = shift;
	my $sub = shift;
	next if exists $created_w_packages{$widgetname}->{$widgetmethod};
	$created_w_packages{$widgetname}->{$widgetmethod}++; #(look in global stash?)
	no strict 'refs';
	my $package = "Tcl::Tk::Widget::$widgetname";
	*{"${package}::$widgetmethod"} = $sub;
    }
}

sub DESTROY {}			# do not let AUTOLOAD catch this method

#
# Let Tcl/Tk process required method via AUTOLOAD mechanism
#

sub AUTOLOAD {
    _DEBUG(3, "(($_[0]|$Tcl::Tk::Widget::AUTOLOAD|@_))\n") if DEBUG;
    my $w = shift;
    my $method = $Tcl::Tk::Widget::AUTOLOAD;
    # Separate method to autoload from (sub)package
    $method =~ s/^(Tcl::Tk::Widget::((MainWindow|$ptk_w_names)::)?)//
	or die "weird inheritance ($method)";
    my $package = $1;
    my $super;
    $method =~ s/^SUPER::// and $super=1; # super-method of child class?

    # if someone calls $widget->_method(...) then it is considered as faster
    # version of method, similar to calling $widget->method(...) but via
    # 'invoke' instead of 'call', thus faster
    my $fast = '';
    $method =~ s/^_// and do {
	$fast='_';
	if (exists $::Tcl::Tk::Widget::{$method}) {
	    no strict 'refs';
	    *{"::Tcl::Tk::Widget::_$method"} = *{"::Tcl::Tk::Widget::$method"};
	    return $w->$method(@_);
	}
    };

    _DEBUG(3, "AUTOLOAD $method IN $package\n") if DEBUG;

    # search for right corresponding Tcl/Tk method, and create it afterwards
    # (so no consequent AUTOLOAD will happen)

    # Precedence ordering is important

    # 1. Check to see if it is a known widget method
    if (exists $ptk2tcltk{$method}) {
	create_widget_package($method);
	my $sub = create_ptk_widget_sub($w->interp,$method,$fast);
	no strict 'refs';
	*{"$package$fast$method"} = $sub;
	return $sub->($w,@_);
    }
    # 2. Check to see if it is a known mappable sub (widget unused)
    if (exists $ptk2tcltk_mapper{$method}) {
        _DEBUG(2, "AUTOCREATE $package$method mapped (@_)\n") if DEBUG;
	my $sub = $fast ? sub {
	    my $self = shift;
	    $self->interp->invoke(@{$ptk2tcltk_mapper{$method}},@_);
	} : sub {
	    my $self = shift;
	    $self->interp->call(@{$ptk2tcltk_mapper{$method}},@_);
	};
	no strict 'refs';
	*{"$package$fast$method"} = $sub;
	return $sub->($w,@_);
    }
    # 3. Check to see if it is a known special widget ability (subcommand)
    # (now this is commented out and probably will go away, as long as a 
    # widget should just create method in its package. But probably such 
    # method could be used for something else)
    #if (exists $special_widget_abilities{$wp} 
    #    && exists $special_widget_abilities{$wp}->{$method}) {
    #    no strict 'refs';
    #    return $special_widget_abilities{$wp}->{$method}->(@_);
    #}
    # 4. Check to see if it is a known 'wm' command
    # XXX: What about toplevel vs. inner widget checking?
    if (exists $ptk2tcltk_wm{$method}) {
        _DEBUG(2, "AUTOCREATE $package$method $ptk2tcltk_wm{$method} (@_)\n") if DEBUG;
	my $sub = $fast ? sub {
	    my $self = shift;
	    $self->interp->invoke($ptk2tcltk_wm{$method}, $method, $self->path, @_);
	} : sub {
	    my $self = shift;
	    $self->interp->call($ptk2tcltk_wm{$method}, $method, $self->path, @_);
	};
	no strict 'refs';
	*{"$package$fast$method"} = $sub;
	return $sub->($w,@_);
    }
    # 5. Check to see if it is a camelCase method.  If so, split it apart.
    # code below will always create subroutine that calls a method.
    # This could be changed to create only known methods and generate error
    # if method is, for example, misspelled.
    # so following check will be like 
    #    if (exists $knows_method_names{$method}) {...}
    my $sub;
    if ($method =~ /^([a-z]+)([A-Z][a-z]+)$/) {
        my ($meth, $submeth) = ($1, lcfirst($2));
	if ($meth eq "grid" || $meth eq "pack") {
	    # grid/pack commands reorder $wp in the call
	    _DEBUG(2, "AUTOCREATE $package$method $meth $submeth (@_)\n") if DEBUG;
	    $sub = $fast ? sub {
		my $w = shift;
		$w->interp->invoke($meth, $submeth, $w->path, @_);
	    } : sub {
		my $w = shift;
		$w->interp->call($meth, $submeth, $w->path, @_);
	    };
	} elsif ($meth eq "after") {
	    # after commands don't include $wp in the call
	    _DEBUG(2, "AUTOCREATE $package$method $meth $submeth (@_)\n") if DEBUG;
	    $sub = $fast ? sub {
		my $w = shift;
		$w->interp->invoke($meth, $submeth, @_);
	    } : sub {
		my $w = shift;
		$w->interp->call($meth, $submeth, @_);
	    };
	} else {
	    # Default case, break into $wp $method $submethod and call
	    _DEBUG(2, "AUTOCREATE $package$method $meth $submeth (@_)\n") if DEBUG;
	    $sub = $fast ? sub {
		my $w = shift;
		$w->interp->invoke($w->path, $meth, $submeth, @_);
	    } : sub {
		my $w = shift;
		$w->interp->call($w->path, $meth, $submeth, @_);
	    };
	}
    }
    else {
	# Default case, call as submethod of $wp
	_DEBUG(2, "AUTOCREATE $package$method $method (@_)\n") if DEBUG;
	$sub = $fast ? sub {
	    my $w = shift;
	    $w->interp->invoke($w->path, $method, @_);
	} : sub {
	    my $w = shift;
	    $w->interp->call($w->path, $method, @_);
	};
    }
    _DEBUG(2, "creating ($package)$method (@_)\n") if DEBUG;
    no strict 'refs';
    *{"$package$fast$method"} = $sub unless $super;
    return $sub->($w,@_);
}

BEGIN {
# var to generate pTk package from
#(test implementation, will be implemented l8r better)
$Tcl::Tk::VTEMP = <<'EOWIDG';
package Tcl::Tk::Widget::[[widget-repl]];

use vars qw/@ISA/;
@ISA = qw(Tcl::Tk::Widget);

sub DESTROY {}			# do not let AUTOLOAD catch this method

sub AUTOLOAD {
    print STDERR "<<@_>>\n" if $Tcl::Tk::DEBUG > 2;
    $Tcl::Tk::Widget::AUTOLOAD = $Tcl::Tk::Widget::[[widget-repl]]::AUTOLOAD;
    return &Tcl::Tk::Widget::AUTOLOAD;
}
1;
print STDERR "<<starting [[widget-repl]]>>\n" if $Tcl::Tk::DEBUG > 2;
EOWIDG
}

package Tcl::Tk::Widget::MultipleWidget;
# multiple widget is an object that for each option has a path
# to refer in Tcl/Tk and for method has corresponding method in Tcl/Tk

my %geometries;
BEGIN {%geometries = map {$_=>1} qw(grid pack form place);}

#syntax
# my $ww = new Tcl::Tk::Widget::MultipleWidget(
#   $int,
#   $w1, [qw(-opt1 -opt2 ...), '-optn=-opttcltk', -optm=>sub{...}],
#   $w2, [qw(-opt1 -opt2 ...), -optk=>\$scalar],
#   ...
# );
# methods are specified like options with starting '&' with optional
# list of replacement options after slash.
#
# specifying '&' without method name will result in declaring said widget
# to be used for all methods that are not listed
# 
# specifying '-' without method name will result in declaring said widget
# to be used for all options that are not listed
# 
# specifying '*' alone will result in declaring said widget
# to be used for all geometry methods
# 
# Example:
# my $ww = new Tcl::Tk::Widget::MultipleWidget($int,
#   $w1, ['-opt1', '-opt2', '-opt3=opttcltk', -opt4=>sub{print 'opt4'}],
#   $w2, ['-opt2=-tkopt2', '-opt5', 
#         '&meth=tkmethod/-opt7=-tkopt7,-opt8,-opt9'],
# );
# In this example:
#   * changing '-opt2' for widget will cause changing '-opt2' for $w1 and
#     changing '-tkopt2' for $w2
#   * changing '-opt1' for widget will cause changing '-opt1' for $w1
#   * invoking method 'meth' for widget will cause invoking 'tkmethod'
#     for $w2 and with options renamed appropriately
#   
# $w1, $w2, ... must be path of a widget or Tcl::Tk::Widget objects
# Also could be called as $int->MultipleWidget(...); TODO
sub new {
    my $package = shift;
    my $int = shift;
    my $self = {
        _int => $int,  # interpreter
	_subst => {},  # hash to hold replacement of option names for pTk=>Tcl/Tk
		       # keys are perlTk option/method, values are array refs 
		       # describing behaviour
	               # "-opt2"=>[$w1,'-opt2',{},$w2,'-tkopt2',{}],
		       # "&meth"=>[$w2,'tkmethod',{-opt7=>'-tkopt7',-opt8=>'-opt8',-opt9=>'-tkopt9'}]
	_def_opt => undef,  # widget to accept unrecognized options
	_def_meth => undef, # widget to accept unrecognized methods
	_def_geom => undef, # widget to accept geometry requests
	_w => {},      # hash of all subwidgets
    };
    my @args = @_;
    for (my $i=0; $i<$#args; $i+=2) {
        my $w = $args[$i];
        $w = $int->declare_widget($w) unless ref $w;
	$self->{_w}->{$w}++;
        my @a = @{$args[$i+1]};
        for (my $j=0; $j<=$#a; $j++) {
            my ($p, $prepl) = ($a[$j]);
	    if ($p eq '-') {
		$self->{_def_opt} = $w;
		next;
	    }
	    elsif ($p eq '&') {
		$self->{_def_meth} = $w;
		next;
	    }
	    elsif ($p eq '*') {
		$self->{_def_geom} = $w;
		$self->{_path} = $w->path;
		next;
	    }
	    my $meth = ($p=~s/^&// ? "&":"");
	    my $hsubst = {};
	    if ($p=~s/\/(.*)$//) {
		$hsubst = {map {m/^(.*)=(.*)$/? ($1=>$2) : ($_=>$_)} split /,/, $1};
	    }
            if ($p=~/^(.*)=(.*)$/) {
                ($p, $prepl) = ($1,$2);
            }
            else {$prepl = $p}
            if ($j+1<=$#a) {
		if (ref($a[$j+1])) {
		    $prepl = $a[$j+1];
		    splice @a, $j+1, 1;
		}
            }
	    $self->{_subst}->{"$meth$p"} ||= []; # create empty array if not exists
	    push @{$self->{_subst}->{"$meth$p"}}, $w, $prepl, $hsubst;
        }
    }
    $self->{_path} ||= ($self->{_def_geom} || $self->{_def_meth} || $args[0])->path;
    return bless $self, $package;
}
sub path {
  $_[0]->{_path};
}

#
# 'configure' and 'cget' could not be processed using common AUTOLOAD
# so must process separatedly
sub configure {
    my $w = shift;
    if ($#_>0) {
	# more than 1 argument, this is setting of many configure options
	my %args = @_;
	my @res;
	for my $optname (keys %args) {
	    if (exists $w->{_subst}->{$optname}) {
		my $mdo = $w->{_subst}->{$optname};
		for my $i (0 .. ($#$mdo-2)/3) {
		    my ($replwid, $replnam, $replopt) = 
		       ($mdo->[3*$i],$mdo->[3*$i+1],$mdo->[3*$i+2]);
		    if (ref($replnam)) {
			if (ref($replnam) eq 'CODE') {
			    @res = $replnam->($replwid,$optname,$args{$optname});
			}
			else {
			    # suppose it's scalar ref to operate with
			    $$replnam = $args{$optname};
			}
		    }
		    else {
		    	@res = $replwid->configure($replnam,$args{$optname});
		    }
		}
	    }
	    elsif (exists $w->{_def_opt}) {
		# default options receiver
		@res = $w->{_def_opt}->configure($optname,$args{$optname});
	    }
	    else {
		die "this MultipleWidget is not able to process $optname";
	    }
	}
	return @res;
    }
    elsif ($#_==0) {
	# 1 argument, in array context return a list of five or two elements
	die "NYI MultipleWidget configure 1";
    }
    else {
	# here $#_==-1, no arguments given
	# Returns a list of lists for all the options supported by widget
	die "NYI MultipleWidget configure 2";
    }
    die "NYI MultipleWidget configure 3";
}
sub cget {
    my $w = shift;
    my $optname = shift;
    if (exists $w->{_subst}->{$optname}) {
	my $mdo = $w->{_subst}->{$optname};
	for my $i (0 .. ($#$mdo-2)/3) {
	    my ($replwid, $replnam, $replopt) = 
	       ($mdo->[3*$i],$mdo->[3*$i+1],$mdo->[3*$i+2]);
	    if (ref($replnam)) {
		if (ref($replnam) eq 'CODE') {
		    return $replnam->($replwid,$optname);
		}
		else {
		    # suppose it's scalar ref to operate with
		    return $$replnam;
		}
	    }
	    if (ref($replnam) && ref($replnam) eq 'CODE') {
	    }
	    else {
		return $replwid->cget($replnam);
	    }
	}
    }
    elsif (exists $w->{_def_opt}) {
	# default options receiver
	return $w->{_def_opt}->cget($optname);
    }
    else {
	die "this MultipleWidget is not able to process CGET $optname";
    }
}
sub Subwidget {
    my ($self,$name) = @_;
    return $self;
}

sub DESTROY {}			# do not let AUTOLOAD catch this method

#
# Unlike for Tcl::Tk::Widget::Button and similar, does not autovivify
# required method; instead it uses autoloading every time, because
# otherwise methods from different MultipleWidgets will mix
sub AUTOLOAD {
    # print STDERR "##@_($Tcl::Tk::Widget::MultipleWidget::AUTOLOAD)##\n";
    # first look into substitute hash
    # if not found - call that method from "default" widget ...->{_def_meth}
    my $wmeth = $Tcl::Tk::Widget::MultipleWidget::AUTOLOAD;
    $wmeth=~s/::MultipleWidget\b//;
    my ($pmeth) = ($wmeth=~/::([^:]+)$/);
    my $self = $_[0];
    my @res;
    if (exists $self->{_subst}->{"&$pmeth"}) {
	my $mdo = $self->{_subst}->{"&$pmeth"};
	my @args = @_[1..$#_];
	my %args = @args;
	for my $i (0 .. ($#$mdo-2)/3) {
	    my ($replwid, $replnam, $replopt) = ($mdo->[3*$i],$mdo->[3*$i+1],$mdo->[3*$i+2]);
	    my %opts;
	    for my $opt (keys %args) {
		if (exists $replopt->{$opt}) {
		    $opts{$replopt->{$opt}} = $args{$opt};
		}
	    }
	    # TODO - when same method should be invoked on several widgets
	    if (wantarray) {
	        @res = $replwid->$replnam(%opts);
	    } else {
	        $res[0] = $replwid->$replnam(%opts);
	    }
	    ## this is not very strict place, but there's no 100% solution.
	    ## if a function returns our sub-widget, then we must return our self.
	    my $_w = $self->{_w};
	    @res = map {exists $_w->{$_} ? $self : $_} @res;
	    return @res if wantarray;
	    return $res[0];
	}
    }
    elsif (exists $geometries{$pmeth} && exists $self->{_def_geom}) {
	if (wantarray) {
	    @res = $self->{_def_geom}->$pmeth(@_[1..$#_]);
	} else {
	    $res[0] = $self->{_def_geom}->$pmeth(@_[1..$#_]);
	}
	## this is not very strict place, but there's no 100% solution.
	## if a function returns our sub-widget, then we must return our self.
	my $_w = $self->{_w};
	@res = map {exists $_w->{$_} ? $self : $_} @res;
	return @res if wantarray;
	return $res[0];
    }
    elsif (exists $self->{_def_meth}) {
	my $replwid = $self->{_def_meth};
	# print STDERR "_def_meth: $replwid $pmeth (@_[1..$#_])\n";
	if (wantarray) {
	    @res = $replwid->$pmeth(@_[1..$#_]);
	} else {
	    $res[0] = $replwid->$pmeth(@_[1..$#_]);
	}
	## this is not very strict place, but there's no 100% solution.
	## if a function returns our sub-widget, then we must return our self.
	my $_w = $self->{_w};
	@res = map {exists $_w->{$_} ? $self : $_} @res;
	return @res if wantarray;
	return $res[0];
    }
    die "this MultipleWidget is not able to process $wmeth";
    # currently not reached
    $Tcl::Tk::Widget::AUTOLOAD = $Tcl::Tk::Widget::MultipleWidget::AUTOLOAD;
    return &Tcl::Tk::Widget::AUTOLOAD;
}

package Tcl::Tk::Widget::MainWindow;

use vars qw/@ISA/;
@ISA = qw(Tcl::Tk::Widget);

sub DESTROY {}			# do not let AUTOLOAD catch this method

sub AUTOLOAD {
    $Tcl::Tk::Widget::AUTOLOAD = $Tcl::Tk::Widget::MainWindow::AUTOLOAD;
    return &Tcl::Tk::Widget::AUTOLOAD;
}

sub path {'.'}

# subroutine for compatibility with perlTk
my $invcnt=0;
sub new {
    my $self = shift;
    if ($invcnt==0) {
        $invcnt++;
        return $self;
    }
    return $self->Toplevel(@_);
}

# provide -title option for 'configure', for perlTk compatibility
sub configure {
    my $self = shift;
    my %args = @_;
    if (exists $args{'-title'}) {
	$self->interp->invoke('wm','title',$self->path,$args{'-title'});
	delete $args{'-title'};
    }
    if (scalar keys %args > 0) {
	# following line should call configure on base class, Tcl::Tk::Widget
	# for some reason, AUTOLOAD sub receives 'SUPER::' within AUTOLOAD
	$self->SUPER::configure(%args);
    }
}
sub cget {
    my $self = shift;
    my $opt = shift;
    if ($opt eq '-title') {
	return $self->interp->invoke('wm','title',$self->path);
    }
    return $self->SUPER::cget($opt);
}

1;
