package Acme::CPANModules::TextTable;

our $DATE = '2020-08-10'; # DATE
our $VERSION = '0.004'; # VERSION

use 5.010001;
use strict;
use warnings;

sub _make_table {
    my ($cols, $rows) = @_;
    my $res = [];
    push @$res, [];
    for (0..$cols-1) { $res->[0][$_] = "col" . ($_+1) }
    for my $row (1..$rows) {
        push @$res, [ map { "row$row.$_" } 1..$cols ];
    }
    $res;
}

our $LIST = {
    summary => 'Modules that generate text tables',
    entry_features => {
        wide_char => {summary => 'Whether the use of wide characters (e.g. Kanji) in cells does not cause the table to be misaligned'},
        color => {summary => 'Whether the module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)'},
        box_char => {summary => 'Whether the module can utilize box-drawing characters'},
    },
    entries => [
        {
            module => 'Text::ANSITable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ANSITable->new(
                    use_utf8 => 0,
                    use_box_chars => 0,
                    use_color => 0,
                    columns => $table->[0],
                    border_style => 'Default::single_ascii',
                );
                $t->add_row($table->[$_]) for 1..@$table-1;
                $t->draw;
            },
            features => {
                wide_char => 1,
                color => 1,
                box_char => 1,
            },
        },
        {
            module => 'Text::ASCIITable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ASCIITable->new();
                $t->setCols(@{ $table->[0] });
                $t->addRow(@{ $table->[$_] }) for 1..@$table-1;
                "$t";
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::FormatTable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::FormatTable->new(join('|', ('l') x @{ $table->[0] }));
                $t->head(@{ $table->[0] });
                $t->row(@{ $table->[$_] }) for 1..@$table-1;
                $t->render;
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::MarkdownTable',
            bench_code => sub {
                my ($table) = @_;
                my $out = "";
                my $t = Text::MarkdownTable->new(file => \$out);
                my $fields = $table->[0];
                foreach (1..@$table-1) {
                    my $row = $table->[$_];
                    $t->add( {
                        map { $fields->[$_] => $row->[$_] } 0..@$fields-1
                    });
                }
                $t->done;
                $out;
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::Table->new(@{ $table->[0] });
                $t->load(@{ $table }[1..@$table-1]);
                $t;
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => {value=>undef, summary=>'Does not draw borders'},
            },
        },
        {
            module => 'Text::Table::Tiny',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Tiny::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 1,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::Sprintf',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Sprintf::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::TinyColor',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColor::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 1,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::TinyColorWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColorWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 1,
                color => 1,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::TinyWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 1,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::Org',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Org::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::CSV',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::CSV::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 1,
                color => 0,
                box_char => {value=>undef, summary=>"Irrelevant"},
            },
        },
        {
            module => 'Text::Table::HTML',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::table(rows=>$table, header_row=>1);
            },
            features => {
            },
        },
        {
            module => 'Text::Table::HTML::DataTables',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::DataTables::table(rows=>$table, header_row=>1);
            },
            features => {
            },
        },
        {
            module => 'Text::TabularDisplay',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::TabularDisplay->new(@{ $table->[0] });
                $t->add(@{ $table->[$_] }) for 1..@$table-1;
                $t->render; # doesn't add newline
            },
            features => {
                wide_char => 1,
                color => 0,
                box_char => {value=>undef, summary=>"Irrelevant"},
            },
        },
    ],

    bench_datasets => [
        {name=>'tiny (1x1)'    , argv => [_make_table( 1, 1)],},
        {name=>'small (3x5)'   , argv => [_make_table( 3, 5)],},
        {name=>'wide (30x5)'   , argv => [_make_table(30, 5)],},
        {name=>'long (3x300)'  , argv => [_make_table( 3, 300)],},
        {name=>'large (30x300)', argv => [_make_table(30, 300)],},
    ],

};

1;
# ABSTRACT: Modules that generate text tables

__END__

=pod

=encoding UTF-8

=head1 NAME

Acme::CPANModules::TextTable - Modules that generate text tables

=head1 VERSION

This document describes version 0.004 of Acme::CPANModules::TextTable (from Perl distribution Acme-CPANModules-TextTable), released on 2020-08-10.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher --cpanmodules-module TextTable

To run module startup overhead benchmark:

 % bencher --module-startup --cpanmodules-module TextTable

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Text::ANSITable> 0.501

L<Text::ASCIITable> 0.22

L<Text::FormatTable> 1.03

L<Text::MarkdownTable> 0.3.1

L<Text::Table> 1.133

L<Text::Table::Tiny> 1.00

L<Text::Table::Sprintf> 0.001

L<Text::Table::TinyColor> 0.002

L<Text::Table::TinyColorWide> 0.001

L<Text::Table::TinyWide> 0.001

L<Text::Table::Org> 0.02

L<Text::Table::CSV> 0.023

L<Text::Table::HTML> 0.003

L<Text::Table::HTML::DataTables> 0.002

L<Text::TabularDisplay> 1.38

=head1 BENCHMARK PARTICIPANTS

=over

=item * Text::ANSITable (perl_code)

L<Text::ANSITable>



=item * Text::ASCIITable (perl_code)

L<Text::ASCIITable>



=item * Text::FormatTable (perl_code)

L<Text::FormatTable>



=item * Text::MarkdownTable (perl_code)

L<Text::MarkdownTable>



=item * Text::Table (perl_code)

L<Text::Table>



=item * Text::Table::Tiny (perl_code)

L<Text::Table::Tiny>



=item * Text::Table::Sprintf (perl_code)

L<Text::Table::Sprintf>



=item * Text::Table::TinyColor (perl_code)

L<Text::Table::TinyColor>



=item * Text::Table::TinyColorWide (perl_code)

L<Text::Table::TinyColorWide>



=item * Text::Table::TinyWide (perl_code)

L<Text::Table::TinyWide>



=item * Text::Table::Org (perl_code)

L<Text::Table::Org>



=item * Text::Table::CSV (perl_code)

L<Text::Table::CSV>



=item * Text::Table::HTML (perl_code)

L<Text::Table::HTML>



=item * Text::Table::HTML::DataTables (perl_code)

L<Text::Table::HTML::DataTables>



=item * Text::TabularDisplay (perl_code)

L<Text::TabularDisplay>



=back

=head1 BENCHMARK DATASETS

=over

=item * tiny (1x1)

=item * small (3x5)

=item * wide (30x5)

=item * long (3x300)

=item * large (30x300)

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 19.10 >>, OS kernel: I<< Linux version 5.3.0-62-generic >>.

Benchmark with default options (C<< bencher --cpanmodules-module TextTable >>):

 #table1#
 {dataset=>"large (30x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::ANSITable               |       2.6 |     380   |                 0.00% |             16761.03% |   0.0015  |      20 |
 | Text::ASCIITable              |      11   |      87   |               333.66% |              3788.05% |   0.00039 |      20 |
 | Text::Table::TinyColorWide    |      16   |      61   |               520.35% |              2617.98% |   0.00024 |      20 |
 | Text::FormatTable             |      18   |      57   |               567.02% |              2427.81% |   0.00021 |      20 |
 | Text::Table::TinyWide         |      23   |      43   |               783.26% |              1808.95% |   0.00022 |      20 |
 | Text::TabularDisplay          |      40   |      20   |              1499.37% |               954.23% |   0.00026 |      20 |
 | Text::Table::TinyColor        |      61   |      16   |              2204.74% |               631.58% |   0.00012 |      20 |
 | Text::Table::Tiny             |      67   |      15   |              2451.72% |               560.77% | 3.5e-05   |      21 |
 | Text::MarkdownTable           |      89   |      11   |              3273.84% |               399.76% | 3.1e-05   |      20 |
 | Text::Table                   |     110   |       9   |              4102.66% |               301.20% | 4.4e-05   |      20 |
 | Text::Table::HTML::DataTables |     120   |       8.2 |              4530.68% |               264.12% | 3.9e-05   |      20 |
 | Text::Table::HTML             |     120   |       8.1 |              4547.02% |               262.84% | 3.8e-05   |      20 |
 | Text::Table::CSV              |     220   |       4.6 |              8082.09% |               106.07% | 2.2e-05   |      20 |
 | Text::Table::Org              |     220   |       4.5 |              8294.84% |               100.85% | 3.4e-05   |      20 |
 | Text::Table::Sprintf          |     450   |       2.2 |             16761.03% |                 0.00% | 1.6e-05   |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

 #table2#
 {dataset=>"long (3x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::ANSITable               |        25 |    40     |                 0.00% |             12440.24% |   0.00015 |      20 |
 | Text::ASCIITable              |       120 |     8.5   |               364.79% |              2598.03% | 6.2e-05   |      21 |
 | Text::FormatTable             |       160 |     6.3   |               531.87% |              1884.61% | 1.5e-05   |      20 |
 | Text::Table::TinyColorWide    |       160 |     6.2   |               541.51% |              1854.80% | 3.5e-05   |      20 |
 | Text::Table::TinyWide         |       240 |     4.2   |               834.16% |              1242.40% | 1.1e-05   |      20 |
 | Text::TabularDisplay          |       340 |     3     |              1225.24% |               846.26% | 1.9e-05   |      21 |
 | Text::MarkdownTable           |       430 |     2.3   |              1584.10% |               644.63% | 1.5e-05   |      20 |
 | Text::Table                   |       500 |     2     |              1863.78% |               538.58% | 1.1e-05   |      20 |
 | Text::Table::Tiny             |       590 |     1.7   |              2227.61% |               438.76% | 2.9e-06   |      20 |
 | Text::Table::TinyColor        |       590 |     1.7   |              2240.66% |               435.76% | 1.3e-05   |      20 |
 | Text::Table::HTML::DataTables |       970 |     1     |              3725.85% |               227.78% | 7.6e-06   |      20 |
 | Text::Table::HTML             |      1000 |     0.96  |              4016.61% |               204.63% | 4.5e-06   |      20 |
 | Text::Table::CSV              |      1800 |     0.57  |              6889.39% |                79.42% | 6.9e-07   |      20 |
 | Text::Table::Org              |      1800 |     0.556 |              7009.65% |                76.38% | 4.8e-07   |      20 |
 | Text::Table::Sprintf          |      3200 |     0.32  |             12440.24% |                 0.00% | 6.8e-07   |      21 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

 #table3#
 {dataset=>"small (3x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::ANSITable               |       960 |  1000     |                 0.00% |              9820.22% | 8.7e-06 |      20 |
 | Text::ASCIITable              |      4900 |   210     |               406.80% |              1857.42% | 9.1e-07 |      20 |
 | Text::FormatTable             |      6900 |   140     |               617.83% |              1281.97% | 4.8e-07 |      20 |
 | Text::Table::TinyColorWide    |      7500 |   130     |               682.13% |              1168.37% | 8.5e-07 |      20 |
 | Text::Table                   |      7700 |   130     |               698.19% |              1142.83% | 1.9e-07 |      24 |
 | Text::Table::TinyWide         |     11000 |    91     |              1037.74% |               771.93% | 2.1e-07 |      20 |
 | Text::MarkdownTable           |     12000 |    81     |              1191.67% |               668.02% | 9.4e-08 |      26 |
 | Text::TabularDisplay          |     15000 |    66     |              1486.35% |               525.35% | 9.5e-08 |      25 |
 | Text::Table::Tiny             |     21000 |    48     |              2049.30% |               361.55% | 1.1e-07 |      20 |
 | Text::Table::HTML::DataTables |     23000 |    43     |              2292.39% |               314.66% | 6.7e-08 |      20 |
 | Text::Table::TinyColor        |     24000 |    41     |              2409.65% |               295.28% | 5.3e-08 |      20 |
 | Text::Table::HTML             |     47000 |    21     |              4808.67% |               102.10% | 2.7e-08 |      20 |
 | Text::Table::Org              |     55000 |    18.2   |              5624.03% |                73.31% | 6.7e-09 |      20 |
 | Text::Table::CSV              |     77523 |    12.899 |              7961.45% |                23.06% | 2.3e-11 |      20 |
 | Text::Table::Sprintf          |     95000 |    10     |              9820.22% |                 0.00% | 1.3e-08 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table4#
 {dataset=>"tiny (1x1)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::ANSITable               |      3700 |    270    |                 0.00% |              7771.18% | 1.1e-06 |      20 |
 | Text::ASCIITable              |     17000 |     58    |               368.96% |              1578.42% | 1.3e-07 |      20 |
 | Text::Table                   |     20000 |     49    |               453.69% |              1321.57% | 1.6e-07 |      20 |
 | Text::MarkdownTable           |     25000 |     41    |               565.30% |              1083.10% |   1e-07 |      21 |
 | Text::FormatTable             |     34000 |     29    |               829.44% |               746.88% | 6.5e-08 |      21 |
 | Text::Table::HTML::DataTables |     37000 |     27    |               886.46% |               697.92% | 5.1e-08 |      22 |
 | Text::Table::TinyWide         |     40000 |     20    |              1056.14% |               580.82% | 7.4e-07 |      22 |
 | Text::Table::TinyColorWide    |     44000 |     23    |              1081.99% |               565.92% | 2.7e-08 |      20 |
 | Text::TabularDisplay          |     61000 |     16    |              1550.59% |               376.87% | 2.7e-08 |      20 |
 | Text::Table::Tiny             |     64000 |     16    |              1633.92% |               353.95% | 5.3e-08 |      20 |
 | Text::Table::TinyColor        |     92600 |     10.8  |              2401.55% |               214.65% | 9.5e-09 |      22 |
 | Text::Table::Org              |    148000 |      6.75 |              3902.12% |                96.68% | 3.3e-09 |      20 |
 | Text::Table::HTML             |    185000 |      5.41 |              4892.61% |                57.66% | 1.5e-09 |      25 |
 | Text::Table::Sprintf          |    280000 |      3.6  |              7417.74% |                 4.70% | 6.4e-09 |      22 |
 | Text::Table::CSV              |    290000 |      3.4  |              7771.18% |                 0.00% |   5e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table5#
 {dataset=>"wide (30x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::ANSITable               |       120 |    8.3    |                 0.00% |             11903.43% | 3.2e-05 |      20 |
 | Text::ASCIITable              |       530 |    1.9    |               336.04% |              2652.85% |   3e-06 |      20 |
 | Text::FormatTable             |       730 |    1.4    |               504.55% |              1885.52% | 2.9e-06 |      20 |
 | Text::Table::TinyColorWide    |       870 |    1.2    |               615.28% |              1578.15% | 2.7e-06 |      20 |
 | Text::Table                   |      1200 |    0.83   |               888.71% |              1114.04% | 1.1e-06 |      20 |
 | Text::Table::TinyWide         |      1200 |    0.82   |               905.59% |              1093.68% | 3.2e-06 |      20 |
 | Text::Table::TinyColor        |      1600 |    0.64   |              1180.37% |               837.49% | 4.2e-06 |      20 |
 | Text::TabularDisplay          |      2310 |    0.432  |              1810.35% |               528.34% | 4.3e-07 |      20 |
 | Text::Table::Tiny             |      3070 |    0.326  |              2430.80% |               374.29% | 2.1e-07 |      20 |
 | Text::MarkdownTable           |      3600 |    0.28   |              2859.84% |               305.54% | 6.4e-07 |      20 |
 | Text::Table::HTML::DataTables |      5660 |    0.177  |              4568.99% |               157.09% | 5.2e-08 |      21 |
 | Text::Table::HTML             |      6400 |    0.16   |              5220.86% |               125.59% |   2e-07 |      23 |
 | Text::Table::Org              |      9070 |    0.11   |              7383.21% |                60.40% | 5.3e-08 |      20 |
 | Text::Table::CSV              |     11900 |    0.0837 |              9756.52% |                21.78% | 2.7e-08 |      20 |
 | Text::Table::Sprintf          |     15000 |    0.069  |             11903.43% |                 0.00% |   8e-08 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


Benchmark module startup overhead (C<< bencher --cpanmodules-module TextTable --module-startup >>):

 #table6#
 +-------------------------------+-----------+--------------------+-----------------------+-----------------------+-----------+---------+
 | participant                   | time (ms) | mod_overhead_time  | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+--------------------+-----------------------+-----------------------+-----------+---------+
 | Text::ANSITable               |     53    | 45                 |                 0.00% |               571.81% |   0.00026 |      20 |
 | Text::MarkdownTable           |     45    | 37                 |                18.67% |               466.09% |   0.0003  |      20 |
 | Text::Table::TinyColorWide    |     33    | 25                 |                61.65% |               315.61% | 8.6e-05   |      20 |
 | Text::Table::TinyWide         |     31    | 23                 |                72.49% |               289.49% |   0.00016 |      20 |
 | Text::Table                   |     23    | 15                 |               135.78% |               184.93% |   7e-05   |      20 |
 | Text::ASCIITable              |     18    | 10                 |               199.24% |               124.51% | 5.8e-05   |      20 |
 | Text::Table::Tiny             |     15    |  7                 |               252.52% |                90.57% | 2.1e-05   |      20 |
 | Text::FormatTable             |     15    |  7                 |               265.13% |                83.99% | 5.5e-05   |      21 |
 | Text::Table::TinyColor        |     14    |  6                 |               286.37% |                73.88% | 3.5e-05   |      20 |
 | Text::TabularDisplay          |     10.6  |  2.6               |               402.33% |                33.74% | 1.1e-05   |      20 |
 | Text::Table::HTML::DataTables |     10    |  2                 |               410.17% |                31.68% | 6.3e-05   |      20 |
 | Text::Table::HTML             |      9.85 |  1.85              |               442.91% |                23.74% | 9.2e-06   |      20 |
 | Text::Table::Sprintf          |      8.5  |  0.5               |               532.31% |                 6.25% | 3.9e-05   |      20 |
 | Text::Table::CSV              |      8.4  |  0.4               |               539.98% |                 4.97% | 2.2e-05   |      20 |
 | Text::Table::Org              |      8.2  |  0.199999999999999 |               548.17% |                 3.65% | 1.9e-05   |      21 |
 | perl -e1 (baseline)           |      8    |  0                 |               571.81% |                 0.00% | 6.7e-05   |      20 |
 +-------------------------------+-----------+--------------------+-----------------------+-----------------------+-----------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 FEATURE COMPARISON MATRIX

 +-------------------------------+--------------+-----------+---------------+
 | module                        | box_char *1) | color *2) | wide_char *3) |
 +-------------------------------+--------------+-----------+---------------+
 | Text::ANSITable               | yes          | yes       | yes           |
 | Text::ASCIITable              | no           | no        | no            |
 | Text::FormatTable             | no           | no        | no            |
 | Text::MarkdownTable           | no           | no        | no            |
 | Text::Table                   | N/A *4)      | no        | no            |
 | Text::Table::Tiny             | no           | yes       | no            |
 | Text::Table::Sprintf          | no           | no        | no            |
 | Text::Table::TinyColor        | no           | yes       | no            |
 | Text::Table::TinyColorWide    | no           | yes       | yes           |
 | Text::Table::TinyWide         | no           | no        | yes           |
 | Text::Table::Org              | no           | no        | no            |
 | Text::Table::CSV              | N/A *5)      | no        | yes           |
 | Text::Table::HTML             | N/A          | N/A       | N/A           |
 | Text::Table::HTML::DataTables | N/A          | N/A       | N/A           |
 | Text::TabularDisplay          | N/A *6)      | no        | yes           |
 +-------------------------------+--------------+-----------+---------------+


Notes:

=over

=item 1. box_char: Whether the module can utilize box-drawing characters

=item 2. color: Whether the module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)

=item 3. wide_char: Whether the use of wide characters (e.g. Kanji) in cells does not cause the table to be misaligned

=item 4. Does not draw borders

=item 5. Irrelevant

=item 6. Irrelevant

=back

=head1 INCLUDED MODULES

=over

=item * L<Text::ANSITable>

=item * L<Text::ASCIITable>

=item * L<Text::FormatTable>

=item * L<Text::MarkdownTable>

=item * L<Text::Table>

=item * L<Text::Table::Tiny>

=item * L<Text::Table::Sprintf>

=item * L<Text::Table::TinyColor>

=item * L<Text::Table::TinyColorWide>

=item * L<Text::Table::TinyWide>

=item * L<Text::Table::Org>

=item * L<Text::Table::CSV>

=item * L<Text::Table::HTML>

=item * L<Text::Table::HTML::DataTables>

=item * L<Text::TabularDisplay>

=back

=head1 FAQ

=head2 What are ways to use this module?

Aside from reading it, you can install all the listed modules using
L<cpanmodules>:

    % cpanmodules ls-entries TextTable | cpanm -n

or L<Acme::CM::Get>:

    % perl -MAcme::CM::Get=TextTable -E'say $_->{module} for @{ $LIST->{entries} }' | cpanm -n

This module contains benchmark instructions. You can run a benchmark
for some/all the modules listed in this Acme::CPANModules module using
L<bencher>:

    % bencher --cpanmodules-module TextTable

This module also helps L<lcpan> produce a more meaningful result for C<lcpan
related-mods> when it comes to finding related modules for the modules listed
in this Acme::CPANModules module.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Acme-CPANModules-TextTable>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Acme-CPANModules-TextTable>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Acme-CPANModules-TextTable>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Acme::CPANModules> - about the Acme::CPANModules namespace

L<cpanmodules> - CLI tool to let you browse/view the lists

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020, 2019 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
