package App::HTTPTinyUtils;

our $DATE = '2018-10-06'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;

our %SPEC;

$SPEC{http_tiny} = {
    v => 1.1,
    args => {
        url => {
            schema => 'str*',
            req => 1,
            pos => 0,
        },
        method => {
            schema => ['str*', match=>qr/\A[A-Z]+\z/],
            default => 'GET',
            cmdline_aliases => {
                delete => {summary => 'Shortcut for --method DELETE', is_flag=>1, code=>sub { $_[0]{method} = 'DELETE' } },
                get    => {summary => 'Shortcut for --method GET'   , is_flag=>1, code=>sub { $_[0]{method} = 'GET'    } },
                head   => {summary => 'Shortcut for --method HEAD'  , is_flag=>1, code=>sub { $_[0]{method} = 'HEAD'   } },
                post   => {summary => 'Shortcut for --method POST'  , is_flag=>1, code=>sub { $_[0]{method} = 'POST'   } },
                put    => {summary => 'Shortcut for --method PUT'   , is_flag=>1, code=>sub { $_[0]{method} = 'PUT'    } },
            },
        },
        headers => {
            schema => ['hash*', of=>'str*'],
            'x.name.is_plural' => 1,
            'x.name.singular' => 'header',
        },
        content => {
            schema => 'str*',
        },
        raw => {
            schema => 'bool*',
        },
        # XXX option: agent
        # XXX option: timeout
        # XXX option: post form
    },
};
sub http_tiny {
    require HTTP::Tiny;

    my %args = @_;
    my $url = $args{url};
    my $method = $args{method} // 'GET';

    my %opts;

    if (defined $args{content}) {
        $opts{content} = $args{content};
    } elsif (!(-t STDIN)) {
        local $/;
        $opts{content} = <STDIN>;
    }

    my $res = HTTP::Tiny->new->request($method, $url, \%opts);

    if ($args{raw}) {
        [200, "OK", $res];
    } else {
        [$res->{status}, $res->{reason}, $res->{content}];
    }
}

1;
# ABSTRACT: Command-line utilities related to HTTP::Tiny

__END__

=pod

=encoding UTF-8

=head1 NAME

App::HTTPTinyUtils - Command-line utilities related to HTTP::Tiny

=head1 VERSION

This document describes version 0.001 of App::HTTPTinyUtils (from Perl distribution App-HTTPTinyUtils), released on 2018-10-06.

=head1 SYNOPSIS

=head1 DESCRIPTION

This distribution includes several utilities related to L<HTTP::Tiny>:

=over

=item * L<http-tiny>

=back

=head1 FUNCTIONS


=head2 http_tiny

Usage:

 http_tiny(%args) -> [status, msg, result, meta]

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<content> => I<str>

=item * B<headers> => I<hash>

=item * B<method> => I<str> (default: "GET")

=item * B<raw> => I<bool>

=item * B<url>* => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-HTTPTinyUtils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-IDXUtils>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-HTTPTinyUtils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
