package Acme::CPANModules::TextTable;

our $DATE = '2020-08-10'; # DATE
our $VERSION = '0.003'; # VERSION

use 5.010001;
use strict;
use warnings;

sub _make_table {
    my ($cols, $rows) = @_;
    my $res = [];
    push @$res, [];
    for (0..$cols-1) { $res->[0][$_] = "col" . ($_+1) }
    for my $row (1..$rows) {
        push @$res, [ map { "row$row.$_" } 1..$cols ];
    }
    $res;
}

our $LIST = {
    summary => 'Modules that generate text tables',
    entry_features => {
        wide_char => {summary => 'Whether the use of wide characters (e.g. Kanji) in cells does not cause the table to be misaligned'},
        color => {summary => 'Whether the module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)'},
        box_char => {summary => 'Whether the module can utilize box-drawing characters'},
    },
    entries => [
        {
            module => 'Text::ANSITable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ANSITable->new(
                    use_utf8 => 0,
                    use_box_chars => 0,
                    use_color => 0,
                    columns => $table->[0],
                    border_style => 'Default::single_ascii',
                );
                $t->add_row($table->[$_]) for 1..@$table-1;
                $t->draw;
            },
            features => {
                wide_char => 1,
                color => 1,
                box_char => 1,
            },
        },
        {
            module => 'Text::ASCIITable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ASCIITable->new();
                $t->setCols(@{ $table->[0] });
                $t->addRow(@{ $table->[$_] }) for 1..@$table-1;
                "$t";
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::FormatTable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::FormatTable->new(join('|', ('l') x @{ $table->[0] }));
                $t->head(@{ $table->[0] });
                $t->row(@{ $table->[$_] }) for 1..@$table-1;
                $t->render;
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::MarkdownTable',
            bench_code => sub {
                my ($table) = @_;
                my $out = "";
                my $t = Text::MarkdownTable->new(file => \$out);
                my $fields = $table->[0];
                foreach (1..@$table-1) {
                    my $row = $table->[$_];
                    $t->add( {
                        map { $fields->[$_] => $row->[$_] } 0..@$fields-1
                    });
                }
                $t->done;
                $out;
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::Table->new(@{ $table->[0] });
                $t->load(@{ $table }[1..@$table-1]);
                $t;
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => {value=>undef, summary=>'Does not draw borders'},
            },
        },
        {
            module => 'Text::Table::Tiny',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Tiny::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 1,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::TinyColor',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColor::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 1,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::TinyColorWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColorWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 1,
                color => 1,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::TinyWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 1,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::Org',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Org::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 0,
                color => 0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::CSV',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::CSV::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char => 1,
                color => 0,
                box_char => {value=>undef, summary=>"Irrelevant"},
            },
        },
        {
            module => 'Text::Table::HTML',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::table(rows=>$table, header_row=>1);
            },
            features => {
            },
        },
        {
            module => 'Text::Table::HTML::DataTables',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::DataTables::table(rows=>$table, header_row=>1);
            },
            features => {
            },
        },
        {
            module => 'Text::TabularDisplay',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::TabularDisplay->new(@{ $table->[0] });
                $t->add(@{ $table->[$_] }) for 1..@$table-1;
                $t->render; # doesn't add newline
            },
            features => {
                wide_char => 1,
                color => 0,
                box_char => {value=>undef, summary=>"Irrelevant"},
            },
        },
    ],

    bench_datasets => [
        {name=>'tiny (1x1)'    , argv => [_make_table( 1, 1)],},
        {name=>'small (3x5)'   , argv => [_make_table( 3, 5)],},
        {name=>'wide (30x5)'   , argv => [_make_table(30, 5)],},
        {name=>'long (3x300)'  , argv => [_make_table( 3, 300)],},
        {name=>'large (30x300)', argv => [_make_table(30, 300)],},
    ],

};

1;
# ABSTRACT: Modules that generate text tables

__END__

=pod

=encoding UTF-8

=head1 NAME

Acme::CPANModules::TextTable - Modules that generate text tables

=head1 VERSION

This document describes version 0.003 of Acme::CPANModules::TextTable (from Perl distribution Acme-CPANModules-TextTable), released on 2020-08-10.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher --cpanmodules-module TextTable

To run module startup overhead benchmark:

 % bencher --module-startup --cpanmodules-module TextTable

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Text::ANSITable> 0.501

L<Text::ASCIITable> 0.22

L<Text::FormatTable> 1.03

L<Text::MarkdownTable> 0.3.1

L<Text::Table> 1.133

L<Text::Table::Tiny> 1.00

L<Text::Table::TinyColor> 0.002

L<Text::Table::TinyColorWide> 0.001

L<Text::Table::TinyWide> 0.001

L<Text::Table::Org> 0.02

L<Text::Table::CSV> 0.023

L<Text::Table::HTML> 0.003

L<Text::Table::HTML::DataTables> 0.002

L<Text::TabularDisplay> 1.38

=head1 BENCHMARK PARTICIPANTS

=over

=item * Text::ANSITable (perl_code)

L<Text::ANSITable>



=item * Text::ASCIITable (perl_code)

L<Text::ASCIITable>



=item * Text::FormatTable (perl_code)

L<Text::FormatTable>



=item * Text::MarkdownTable (perl_code)

L<Text::MarkdownTable>



=item * Text::Table (perl_code)

L<Text::Table>



=item * Text::Table::Tiny (perl_code)

L<Text::Table::Tiny>



=item * Text::Table::TinyColor (perl_code)

L<Text::Table::TinyColor>



=item * Text::Table::TinyColorWide (perl_code)

L<Text::Table::TinyColorWide>



=item * Text::Table::TinyWide (perl_code)

L<Text::Table::TinyWide>



=item * Text::Table::Org (perl_code)

L<Text::Table::Org>



=item * Text::Table::CSV (perl_code)

L<Text::Table::CSV>



=item * Text::Table::HTML (perl_code)

L<Text::Table::HTML>



=item * Text::Table::HTML::DataTables (perl_code)

L<Text::Table::HTML::DataTables>



=item * Text::TabularDisplay (perl_code)

L<Text::TabularDisplay>



=back

=head1 BENCHMARK DATASETS

=over

=item * tiny (1x1)

=item * small (3x5)

=item * wide (30x5)

=item * long (3x300)

=item * large (30x300)

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 19.10 >>, OS kernel: I<< Linux version 5.3.0-62-generic >>.

Benchmark with default options (C<< bencher --cpanmodules-module TextTable >>):

 #table1#
 {dataset=>"large (30x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::ANSITable               |       2.4 |     410   |                 0.00% |              8922.57% |   0.0027  |      20 |
 | Text::ASCIITable              |      11   |      90   |               357.49% |              1872.20% |   0.00027 |      20 |
 | Text::Table::TinyColorWide    |      15   |      65   |               528.50% |              1335.57% |   0.00033 |      20 |
 | Text::FormatTable             |      17   |      58   |               606.85% |              1176.45% |   0.00027 |      20 |
 | Text::Table::TinyWide         |      23   |      44   |               835.18% |               864.80% |   0.00029 |      20 |
 | Text::TabularDisplay          |      42   |      24   |              1628.15% |               422.09% |   0.00011 |      21 |
 | Text::Table::TinyColor        |      60   |      20   |              2201.33% |               292.06% |   0.00021 |      20 |
 | Text::Table::Tiny             |      65.7 |      15.2 |              2591.82% |               235.19% | 1.4e-05   |      20 |
 | Text::Table::HTML::DataTables |      90   |      10   |              3492.19% |               151.17% |   0.00041 |      24 |
 | Text::MarkdownTable           |      90   |      11   |              3602.43% |               143.69% | 4.1e-05   |      21 |
 | Text::Table                   |     110   |       8.9 |              4490.39% |                96.55% | 3.7e-05   |      20 |
 | Text::Table::HTML             |     120   |       8   |              5019.59% |                76.24% | 4.4e-05   |      20 |
 | Text::Table::Org              |     220   |       4.6 |              8863.35% |                 0.66% | 2.7e-05   |      20 |
 | Text::Table::CSV              |     220   |       4.5 |              8922.57% |                 0.00% | 2.1e-05   |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

 #table2#
 {dataset=>"long (3x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::ANSITable               |        23 |    43     |                 0.00% |              7683.96% |   0.0003  |      22 |
 | Text::ASCIITable              |       100 |    10     |               319.69% |              1754.70% |   0.00035 |      20 |
 | Text::Table::TinyColorWide    |       100 |     9     |               402.38% |              1449.43% |   0.00032 |      20 |
 | Text::FormatTable             |       160 |     6.4   |               581.36% |              1042.42% | 3.1e-05   |      20 |
 | Text::Table::TinyWide         |       230 |     4.3   |               910.36% |               670.42% | 3.8e-05   |      21 |
 | Text::TabularDisplay          |       300 |     3     |              1248.66% |               477.16% | 3.3e-05   |      20 |
 | Text::MarkdownTable           |       430 |     2.3   |              1782.02% |               313.60% | 9.4e-06   |      20 |
 | Text::Table                   |       490 |     2.1   |              2016.04% |               267.86% | 1.7e-05   |      20 |
 | Text::Table::Tiny             |       540 |     1.9   |              2242.42% |               232.30% | 1.3e-05   |      20 |
 | Text::Table::TinyColor        |       590 |     1.7   |              2466.37% |               203.31% | 4.8e-06   |      21 |
 | Text::Table::Org              |       990 |     1     |              4183.04% |                81.74% | 7.6e-06   |      20 |
 | Text::Table::HTML::DataTables |      1050 |     0.952 |              4464.93% |                70.52% | 6.9e-07   |      20 |
 | Text::Table::HTML             |      1100 |     0.92  |              4611.97% |                65.20% | 1.4e-06   |      24 |
 | Text::Table::CSV              |      1800 |     0.56  |              7683.96% |                 0.00% | 1.3e-06   |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

 #table3#
 {dataset=>"small (3x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::ANSITable               |       990 |    1000   |                 0.00% |              7985.60% | 5.9e-06 |      21 |
 | Text::ASCIITable              |      5000 |     200   |               401.08% |              1513.64% | 4.3e-07 |      20 |
 | Text::Table::TinyColor        |      7000 |     100   |               579.59% |              1089.77% |   2e-06 |      20 |
 | Text::FormatTable             |      6900 |     140   |               602.29% |              1051.32% | 2.1e-07 |      20 |
 | Text::Table::TinyColorWide    |      6900 |     140   |               603.04% |              1050.08% | 6.4e-07 |      20 |
 | Text::Table                   |      7600 |     130   |               669.96% |               950.13% | 2.1e-07 |      20 |
 | Text::Table::TinyWide         |     11000 |      90   |              1020.38% |               621.69% | 1.1e-07 |      20 |
 | Text::MarkdownTable           |     13000 |      78   |              1189.69% |               526.94% |   1e-07 |      22 |
 | Text::TabularDisplay          |     13000 |      75   |              1257.03% |               495.83% | 6.5e-07 |      21 |
 | Text::Table::Tiny             |     21000 |      48   |              1993.93% |               286.14% | 5.3e-08 |      20 |
 | Text::Table::HTML::DataTables |     23000 |      43   |              2230.92% |               246.89% | 7.7e-08 |      38 |
 | Text::Table::HTML             |     47500 |      21.1 |              4702.56% |                68.36% |   2e-08 |      20 |
 | Text::Table::Org              |     54000 |      19   |              5357.35% |                48.16% | 5.3e-08 |      20 |
 | Text::Table::CSV              |     80000 |      13   |              7985.60% |                 0.00% | 1.3e-08 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table4#
 {dataset=>"tiny (1x1)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::ANSITable               |      3870 |   258     |                 0.00% |              7500.20% | 2.5e-07 |      23 |
 | Text::ASCIITable              |     16000 |    64     |               304.53% |              1778.79% | 2.3e-07 |      22 |
 | Text::Table                   |     21000 |    48     |               437.41% |              1314.22% | 1.6e-07 |      20 |
 | Text::Table::TinyColorWide    |     23000 |    43     |               505.35% |              1155.50% | 5.3e-08 |      20 |
 | Text::MarkdownTable           |     25000 |    39     |               557.65% |              1055.66% | 6.4e-08 |      22 |
 | Text::FormatTable             |     34000 |    29     |               783.38% |               760.36% | 5.2e-08 |      21 |
 | Text::Table::HTML::DataTables |     38000 |    26     |               875.76% |               678.90% | 5.3e-08 |      20 |
 | Text::TabularDisplay          |     50000 |    20     |              1272.94% |               453.57% | 4.5e-07 |      20 |
 | Text::Table::TinyWide         |     58000 |    17     |              1409.51% |               403.49% | 2.7e-08 |      20 |
 | Text::Table::Tiny             |     65037 |    15.376 |              1580.82% |               352.17% | 2.3e-11 |      20 |
 | Text::Table::TinyColor        |     88000 |    11     |              2162.39% |               235.94% | 1.3e-08 |      20 |
 | Text::Table::Org              |    140000 |     7     |              3607.87% |               104.97% | 1.3e-08 |      20 |
 | Text::Table::HTML             |    190000 |     5.27  |              4804.23% |                54.97% | 1.6e-09 |      21 |
 | Text::Table::CSV              |    290000 |     3.4   |              7500.20% |                 0.00% |   5e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

 #table5#
 {dataset=>"wide (30x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::ANSITable               |       120 |     8.2   |                 0.00% |              9728.34% | 2.4e-05 |      21 |
 | Text::ASCIITable              |       500 |     2     |               291.25% |              2412.07% | 3.4e-05 |      21 |
 | Text::Table::TinyColorWide    |       600 |     2     |               360.89% |              2032.48% | 6.6e-05 |      20 |
 | Text::FormatTable             |       730 |     1.4   |               495.16% |              1551.37% | 1.5e-06 |      20 |
 | Text::Table::TinyWide         |      1200 |     0.85  |               867.13% |               916.24% | 4.9e-06 |      20 |
 | Text::Table                   |      1200 |     0.84  |               870.97% |               912.22% | 1.3e-06 |      20 |
 | Text::TabularDisplay          |      2200 |     0.46  |              1662.17% |               457.74% | 4.3e-06 |      20 |
 | Text::Table::TinyColor        |      2500 |     0.39  |              1981.23% |               372.24% | 3.1e-06 |      20 |
 | Text::Table::Tiny             |      2900 |     0.34  |              2306.90% |               308.34% | 6.9e-07 |      20 |
 | Text::MarkdownTable           |      3660 |     0.273 |              2898.54% |               227.77% | 2.1e-07 |      20 |
 | Text::Table::Org              |      5400 |     0.19  |              4304.73% |               123.13% | 9.1e-07 |      20 |
 | Text::Table::HTML::DataTables |      5600 |     0.18  |              4507.44% |               113.31% | 4.8e-07 |      20 |
 | Text::Table::HTML             |      6600 |     0.15  |              5295.00% |                82.18% | 2.1e-07 |      20 |
 | Text::Table::CSV              |     12000 |     0.083 |              9728.34% |                 0.00% | 1.3e-07 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


Benchmark module startup overhead (C<< bencher --cpanmodules-module TextTable --module-startup >>):

 #table6#
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant                   | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | Text::ANSITable               |      55   |              46.4 |                 0.00% |               533.41% |   0.00031 |      21 |
 | Text::MarkdownTable           |      45   |              36.4 |                22.56% |               416.82% |   0.00027 |      20 |
 | Text::Table::TinyColorWide    |      34   |              25.4 |                62.31% |               290.25% |   0.00015 |      20 |
 | Text::Table::TinyWide         |      32   |              23.4 |                68.92% |               274.97% |   0.00022 |      20 |
 | Text::Table                   |      30   |              21.4 |               104.38% |               209.92% |   0.00047 |      20 |
 | Text::ASCIITable              |      19   |              10.4 |               180.80% |               125.57% | 9.5e-05   |      20 |
 | Text::Table::Tiny             |      16   |               7.4 |               246.46% |                82.82% | 6.1e-05   |      20 |
 | Text::FormatTable             |      15   |               6.4 |               256.89% |                77.48% | 6.9e-05   |      20 |
 | Text::Table::TinyColor        |      14   |               5.4 |               283.33% |                65.24% | 4.1e-05   |      20 |
 | Text::TabularDisplay          |      11   |               2.4 |               385.83% |                30.38% | 2.9e-05   |      20 |
 | Text::Table::HTML::DataTables |      11   |               2.4 |               414.43% |                23.13% |   5e-05   |      21 |
 | Text::Table::HTML             |      11   |               2.4 |               416.60% |                22.61% | 5.1e-05   |      20 |
 | Text::Table::CSV              |       9.2 |               0.6 |               496.91% |                 6.11% | 5.9e-05   |      20 |
 | Text::Table::Org              |       9   |               0.4 |               506.43% |                 4.45% |   0.00011 |      21 |
 | perl -e1 (baseline)           |       8.6 |               0   |               533.41% |                 0.00% | 7.7e-05   |      20 |
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 FEATURE COMPARISON MATRIX

 +-------------------------------+--------------+-----------+---------------+
 | module                        | box_char *1) | color *2) | wide_char *3) |
 +-------------------------------+--------------+-----------+---------------+
 | Text::ANSITable               | yes          | yes       | yes           |
 | Text::ASCIITable              | no           | no        | no            |
 | Text::FormatTable             | no           | no        | no            |
 | Text::MarkdownTable           | no           | no        | no            |
 | Text::Table                   | N/A *4)      | no        | no            |
 | Text::Table::Tiny             | no           | yes       | no            |
 | Text::Table::TinyColor        | no           | yes       | no            |
 | Text::Table::TinyColorWide    | no           | yes       | yes           |
 | Text::Table::TinyWide         | no           | no        | yes           |
 | Text::Table::Org              | no           | no        | no            |
 | Text::Table::CSV              | N/A *5)      | no        | yes           |
 | Text::Table::HTML             | N/A          | N/A       | N/A           |
 | Text::Table::HTML::DataTables | N/A          | N/A       | N/A           |
 | Text::TabularDisplay          | N/A *6)      | no        | yes           |
 +-------------------------------+--------------+-----------+---------------+


Notes:

=over

=item 1. box_char: Whether the module can utilize box-drawing characters

=item 2. color: Whether the module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)

=item 3. wide_char: Whether the use of wide characters (e.g. Kanji) in cells does not cause the table to be misaligned

=item 4. Does not draw borders

=item 5. Irrelevant

=item 6. Irrelevant

=back

=head1 INCLUDED MODULES

=over

=item * L<Text::ANSITable>

=item * L<Text::ASCIITable>

=item * L<Text::FormatTable>

=item * L<Text::MarkdownTable>

=item * L<Text::Table>

=item * L<Text::Table::Tiny>

=item * L<Text::Table::TinyColor>

=item * L<Text::Table::TinyColorWide>

=item * L<Text::Table::TinyWide>

=item * L<Text::Table::Org>

=item * L<Text::Table::CSV>

=item * L<Text::Table::HTML>

=item * L<Text::Table::HTML::DataTables>

=item * L<Text::TabularDisplay>

=back

=head1 FAQ

=head2 What are ways to use this module?

Aside from reading it, you can install all the listed modules using
L<cpanmodules>:

    % cpanmodules ls-entries TextTable | cpanm -n

or L<Acme::CM::Get>:

    % perl -MAcme::CM::Get=TextTable -E'say $_->{module} for @{ $LIST->{entries} }' | cpanm -n

This module contains benchmark instructions. You can run a benchmark
for some/all the modules listed in this Acme::CPANModules module using
L<bencher>:

    % bencher --cpanmodules-module TextTable

This module also helps L<lcpan> produce a more meaningful result for C<lcpan
related-mods> when it comes to finding related modules for the modules listed
in this Acme::CPANModules module.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Acme-CPANModules-TextTable>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Acme-CPANModules-TextTable>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Acme-CPANModules-TextTable>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Acme::CPANModules> - about the Acme::CPANModules namespace

L<cpanmodules> - CLI tool to let you browse/view the lists

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020, 2019 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
