# NAME

DBIx::Class::Helper::Row::Enumeration - Add methods for emum values

# VERSION

version v0.1.2

# SYNOPSIS

In your result class:

```perl
use base qw/DBIx::Class::Core/;

__PACKAGE__->load_components(qw/ Helper::Row::Enumeration /);

__PACKAGE__->add_column(

  foo => {
      data_type => 'enum',
      extra     => {
          list => [qw/ good bad ugly /],
      },
  },
```

with a row:

```
if ($row->is_good) { ... }
```

# DESCRIPTION

This plugin is inspired by [MooseX::Enumeration](https://metacpan.org/pod/MooseX::Enumeration).

Suppose your database has a column with an enum value. Checks against
string values are prone to typos:

```
if ($row->result eq 'faol') { ... }
```

when instead you wanted

```
if ($row->result eq 'fail') { ... }
```

Using this plugin, you can avoid common bugs by checking against a
method instead:

```
if ($row->is_fail) { ... }
```

# Overriding method names

You can override method names by adding an extra `handles` attribute
to the column definition:

```perl
bar => {
    data_type => 'enum',
    extra     => {
        list   => [qw/ good bad ugly /],
        handles => {
            good_bar => 'good',
            coyote   => 'ugly',
        },
    },
},
```

Note that only methods you specify will be added. In the above case,
there is no "is\_bad" method added.

The `handles` attribute can also be set to a code reference so that
method names can be generated dynamically:

```perl
baz => {
    data_type => 'enum',
    extra     => {
        list   => [qw/ good bad ugly /],
        handles => sub {
            my ($value, $col, $class) = @_;

            return undef if $value eq 'deprecated';

            return "is_${col}_${value}";
        },
    },
},
);
```

If the function returns `undef`, then no method will be generated for
that value.

If `handles` is set to "0", then no methods will be generated for the
column at all.

# KNOWN ISSUES

See also ["BUGS"](#bugs) below.

## Overlapping enum values

Multiple columns with overlapping enum values will cause an error.
You'll need to specify a handler to rename methods or skip them
altogether.

## Autogenerated Classes

You cannot override the configuration with autogenerated classes.
This issue will be addressed in a later version.

# SEE ALSO

[DBIx::Class](https://metacpan.org/pod/DBIx::Class)

[MooseX::Enumeration](https://metacpan.org/pod/MooseX::Enumeration)

# SOURCE

The development version is on github at [https://github.com/robrwo/DBIx-Class-Helper-Row-Enumeration](https://github.com/robrwo/DBIx-Class-Helper-Row-Enumeration)
and may be cloned from [git://github.com/robrwo/DBIx-Class-Helper-Row-Enumeration.git](git://github.com/robrwo/DBIx-Class-Helper-Row-Enumeration.git)

# BUGS

Please report any bugs or feature requests on the bugtracker website
[https://github.com/robrwo/DBIx-Class-Helper-Row-Enumeration/issues](https://github.com/robrwo/DBIx-Class-Helper-Row-Enumeration/issues)

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

# AUTHOR

Robert Rothenberg <rrwo@cpan.org>

The initial development of this module was sponsored by Science Photo
Library [https://www.sciencephoto.com](https://www.sciencephoto.com).

# COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Robert Rothenberg.

This is free software, licensed under:

```
The Artistic License 2.0 (GPL Compatible)
```
