package Firefox::Util::Profile;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2020-04-18'; # DATE
our $DIST = 'Firefox-Util-Profile'; # DIST
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;
use Log::ger;

our %SPEC;

$SPEC{list_firefox_profiles} = {
    v => 1.1,
    summary => 'List available Firefox profiles',
    description => <<'_',

This utility will read ~/.mozilla/firefox/profiles.ini and extracts the list of
profiles.

_
    args => {
        detail => {
            schema => 'bool',
            cmdline_aliases => {l=>{}},
        },
    },
};
sub list_firefox_profiles {
    require Config::IOD::INI::Reader;
    require Sort::Sub;

    my %args = @_;

    my $ff_dir   = "$ENV{HOME}/.mozilla/firefox";
    my $ini_path = "$ff_dir/profiles.ini";
    unless (-f $ini_path) {
        return [412, "Cannot find $ini_path"];
    }

    my @rows;
    my $hoh = Config::IOD::INI::Reader->new->read_file($ini_path);
    my $naturally = Sort::Sub::get_sorter('naturally');
  SECTION:
    for my $section (sort $naturally keys %$hoh) {
        my $href = $hoh->{$section};
        if ($section =~ /\AProfile/) {
            my $path;
            if (defined($path = $href->{Path})) {
                $path = "$ff_dir/$path" if $href->{IsRelative};
                push @rows, {
                    name => $href->{Name} // $section,
                    path => $path,
                    ini_section => $section,
                };
            } else {
                log_warn "$ini_path: No Path parameter for section $section, section ignored";
                next SECTION;
            }
        }
        # XXX add info: which sections are default in which installation
        # ([Install...] sections)
    }

    unless ($args{detail}) {
        @rows = map { $_->{name} } @rows;
    }

    [200, "OK", \@rows];
}

1;
# ABSTRACT: List available Firefox profiles

__END__

=pod

=encoding UTF-8

=head1 NAME

Firefox::Util::Profile - List available Firefox profiles

=head1 VERSION

This document describes version 0.001 of Firefox::Util::Profile (from Perl distribution Firefox-Util-Profile), released on 2020-04-18.

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 FUNCTIONS


=head2 list_firefox_profiles

Usage:

 list_firefox_profiles(%args) -> [status, msg, payload, meta]

List available Firefox profiles.

This utility will read ~/.mozilla/firefox/profiles.ini and extracts the list of
profiles.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<detail> => I<bool>


=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (payload) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Firefox-Util-Profile>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Firefox-Util-Profile>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Firefox-Util-Profile>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

Other C<Firefox::Util::*> modules.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
